/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include "Engine.h"
#include "SRLUpdateRecords.h"
#include "Stream.h"
#include "Table.h"
#include "SerialLogControl.h"
#include "SerialLogTransaction.h"
#include "Dbb.h"
#include "Transaction.h"
#include "RecordVersion.h"
#include "Log.h"
#include "Sync.h"

SRLUpdateRecords::SRLUpdateRecords(void)
{
}

SRLUpdateRecords::~SRLUpdateRecords(void)
{
}

void SRLUpdateRecords::append(Transaction *transaction, RecordVersion *records)
{
	for (RecordVersion *record = records; record;)
		{
		START_RECORD(srlUpdateRecords, "SRLUpdateRecords::append");
		SerialLogTransaction *trans = log->getTransaction(transaction->transactionId);
		trans->transaction = transaction;
		putInt(transaction->transactionId);
		UCHAR *lengthPtr = putFixedInt(0);
		UCHAR *start = log->writePtr;
		UCHAR *end = log->writeWarningTrack;
		
		for (; record; record = record->next)
			{
			Table *table = record->table;
			Stream stream;
			
			if (record->hasRecord())
				record->getRecord(&stream);
			
			if (log->writePtr + byteCount(table->dataSection) + byteCount(record->recordNumber) +
					byteCount(stream.totalLength) + stream.totalLength >= end)
				break;
			
			uint32 sectionId = table->dataSection;
			
			if (sectionId >= log->sectionUseVector.length)
				log->extendSectionVector(sectionId + 10);
			
			++log->sectionUseVector.vector[sectionId];
			putInt((record->priorVersion) ? sectionId : -(int) sectionId - 1);
			ASSERT(record->recordNumber >= 0);
			putInt(record->recordNumber);
			putStream(&stream);
			}
		
		int len = log->writePtr - start;
		putFixedInt(len, lengthPtr);
		
		if (record)
			log->flush(true, 0, &sync);
		else
			sync.unlock();
		}
}

void SRLUpdateRecords::read(void)
{
	transactionId = getInt();
	dataLength = getInt();
	data = getData(dataLength);
}


void SRLUpdateRecords::redo(void)
{
	SerialLogTransaction *transaction = control->getTransaction(transactionId);
	
	if (transaction->state == sltCommitted)
		for (const UCHAR *p = data, *end = data + dataLength; p < end;)
			{
			int id = getInt(&p);
			uint sectionId = (id >= 0) ? id : -id - 1;
			int recordNumber = getInt(&p);
			int length = getInt(&p);
			
			if (sectionId < log->sectionUseVector.length)
				--log->sectionUseVector.vector[sectionId];
			
			if (log->bumpSectionIncarnation(sectionId, objInUse))
				{
				if (length)
					{
					if (id < 0)
						log->dbb->reInsertStub(sectionId, recordNumber, transactionId);
						
					Stream stream;
					stream.putSegment(length, (const char*) p, false);
					log->dbb->updateRecord(sectionId, recordNumber, &stream, transactionId, false);
					}
				else
					log->dbb->updateRecord(sectionId, recordNumber, NULL, transactionId, false);
				}
			
			p += length;
			}
	else
		pass1();
}

void SRLUpdateRecords::pass1(void)
{
	control->getTransaction(transactionId);

	for (const UCHAR *p = data, *end = data + dataLength; p < end;)
		{
		int id = getInt(&p);
		uint sectionId = (id >= 0) ? id : -id - 1;
		int recordNumber = getInt(&p);
		int length = getInt(&p);
		log->bumpSectionIncarnation(sectionId, objInUse);
		p += length;
		}
}

void SRLUpdateRecords::pass2(void)
{
	pass1();
}

void SRLUpdateRecords::commit(void)
{
	Sync sync(&log->syncSections, "SRLUpdateRecords::commit");
	sync.lock(Shared);
	
	for (const UCHAR *p = data, *end = data + dataLength; p < end;)
		{
		int id = getInt(&p);
		uint sectionId = (id >= 0) ? id : -id - 1;
		int recordNumber = getInt(&p);
		int length = getInt(&p);
		
		if (sectionId < log->sectionUseVector.length)
			--log->sectionUseVector.vector[sectionId];
		
		if (log->isSectionActive(sectionId))
			{
			if (length)
				{
				Stream stream;
				stream.putSegment(length, (const char*) p, false);
				log->dbb->updateRecord(sectionId, recordNumber, &stream, transactionId, false);
				}
			else
				log->dbb->updateRecord(sectionId, recordNumber, NULL, transactionId, false);
			}
		
		p += length;
		}
}

void SRLUpdateRecords::print(void)
{
	logPrint("UpdateRecords: transaction %d, length %d\n", transactionId, dataLength);
	
	for (const UCHAR *p = data, *end = data + dataLength; p < end;)
		{
		uint sectionId = getInt(&p);
		int recordNumber = getInt(&p);
		int length = getInt(&p);
		char temp[40];
		printf("   rec %d, len %d to section %d %s\n", 
				recordNumber, length, sectionId, format(length, p, sizeof(temp), temp));
		p += length;
		}
}
