/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "mysql_priv.h"
typedef longlong	int64;

#include "ScaledBinary.h"
#include "BigInt.h"

#define DIGITS_PER_INT			9
#define BASE					1000000000

#undef C64

#ifdef _WIN32
#define C64(x)	x##i64
#endif

#ifdef __GNUC__
#define C64(x)	x##ll
#endif

#ifndef C64
#define C64(x)	x
#endif

static const int bytesByDigit [] = {
	0,			// 0
	1,			// 10
	1,			// 100
	2,			// 1000
	2,			// 10000
	3,			// 100000
	3,			// 1000000
	4,			// 10000000
	4,			// 100000000
	4			// 1000000000
	};

static const int64 powersOfTen [] = {
	C64(1),
	C64(10),
	C64(100),
	C64(1000),
	C64(10000),
	C64(100000),
	C64(1000000),
	C64(10000000),
	C64(100000000),
	C64(1000000000),
	C64(10000000000),
	C64(100000000000),
	C64(1000000000000),
	C64(10000000000000),
	C64(100000000000000),
	C64(1000000000000000),
	C64(10000000000000000),
	C64(100000000000000000),
	C64(1000000000000000000)
	//C64(10000000000000000000)
	};
	
ScaledBinary::ScaledBinary(void)
{
}

ScaledBinary::~ScaledBinary(void)
{
}

int64 ScaledBinary::getInt64FromBinaryDecimal(const char *ptr, int precision, int scale)
{
	int position = 0;
	int64 value = getBinaryNumber(position, precision - scale, ptr, false);

	if (scale)
		{
		position += numberBytes(precision - scale);
		int64 fraction = getBinaryNumber(position, scale, ptr, true);
		value = value * powersOfTen[scale] + fraction;
		}
	
	return (ptr[0] & 0x80) ? value : -value;
}

uint ScaledBinary::getByte(int position, const char* ptr)
{
	char c = ptr[position];
	
	if (!(ptr[0] & 0x80))
		c = ~c;
	
	if (position == 0)
		c ^= 0x80;
	
	return (unsigned char) c;	
}

uint ScaledBinary::getBinaryGroup(int start, int bytes, const char* ptr)
{
	uint group = 0;
	int end = start + bytes;
	
	for (int n = start, end = start + bytes; n < end; ++n)
		group = (group << 8) + getByte(n, ptr);
	
	return group;
}


int ScaledBinary::numberBytes(int digits)
{
	return (digits / DIGITS_PER_INT) * sizeof(decimal_digit_t) + bytesByDigit[digits % DIGITS_PER_INT];
}

void ScaledBinary::putBinaryDecimal(int64 number, char* ptr, int precision, int scale)
{
	int64 absNumber = (number >= 0) ? number : -number;
	char *p = ptr;
	putBinaryNumber(absNumber / powersOfTen[scale], precision - scale, &p, false);
	
	if (scale)
		putBinaryNumber(absNumber % powersOfTen[scale], scale, &p, true);

	if (number < 0)
		for (char *q = ptr; q < p; ++q)
			*q ^= -1;
	
	*ptr ^= 0x80;
}

int64 ScaledBinary::getBinaryNumber(int start, int digits, const char* ptr, bool isFraction)
{
	int64 value = 0;
	int position = start;
	int groups = digits / DIGITS_PER_INT;
	int partialDigits = digits % DIGITS_PER_INT;
	int partialBytes = bytesByDigit[partialDigits];
	
	if (!isFraction && partialBytes)
		{
		value = getBinaryGroup(position, partialBytes, ptr);
		position += partialBytes;
		partialBytes = 0;
		}
		
	for (int n = 0; n < groups; ++n)
		{
		value = value * BASE + getBinaryGroup(position, sizeof(decimal_digit_t), ptr);
		position += sizeof(decimal_digit_t);
		}
	
	if (partialBytes)
		value = value * powersOfTen[partialDigits] + getBinaryGroup(position, partialBytes, ptr);
		
	return value;
}

void ScaledBinary::putBinaryNumber(int64 number, int digits, char** ptr, bool isFraction)
{
	int groups = digits / DIGITS_PER_INT;
	int partialDigits = digits % DIGITS_PER_INT;
	char *p = *ptr;
	int index = digits;
	
	if (!isFraction && partialDigits)
		{
		index -= partialDigits;
		putBinaryGroup((uint) (number / powersOfTen[index]), partialDigits, &p);
		partialDigits = 0;
		}
	
	for (int n = 0; n < groups; ++n)
		{
		index -= DIGITS_PER_INT;

		if (index)
			putBinaryGroup(((uint) (number / powersOfTen[index])) % BASE, DIGITS_PER_INT, &p);
		else
			putBinaryGroup((uint) number % BASE, DIGITS_PER_INT, &p);
		}
	
	if (partialDigits)
		putBinaryGroup((uint) (number % powersOfTen[partialDigits]), partialDigits, &p);
	
	*ptr = p;
}

void ScaledBinary::putBinaryGroup(uint number, int digits, char** ptr)
{
	char *p = *ptr;
	int bytes = bytesByDigit[digits];
	
	for (int n = (bytes - 1) * 8; n >= 0; n -= 8)
		*p++ = (char) (number >> n);

	*ptr = p;
}

void ScaledBinary::getBigIntFromBinaryDecimal(const char* ptr, int precision, int scale, BigInt *bigInt)
{
	int position = 0;
	getBigNumber(position, precision - scale, ptr, false, bigInt);

	if (scale)
		{
		position += numberBytes(precision - scale);
		getBigNumber(position, scale, ptr, true, bigInt);
		}
	
	bigInt->scale = -scale;
	bigInt->neg = (ptr[0] & 0x80) == 0;
}

void ScaledBinary::getBigNumber(int start, int digits, const char* ptr, bool isFraction, BigInt* bigInt)
{
	int position = start;
	int groups = digits / DIGITS_PER_INT;
	int partialDigits = digits % DIGITS_PER_INT;
	int partialBytes = bytesByDigit[partialDigits];
	
	if (!isFraction && partialBytes)
		{
		BigWord value = getBinaryGroup(position, partialBytes, ptr);
		bigInt->set(value);
		position += partialBytes;
		partialBytes = 0;
		}
		
	for (int n = 0; n < groups; ++n)
		{
		BigWord value = getBinaryGroup(position, sizeof(decimal_digit_t), ptr);
		bigInt->multiply(BASE);
		bigInt->add(0, value);
		position += sizeof(decimal_digit_t);
		}
	
	if (partialBytes)
		{
		bigInt->multiply((uint32) powersOfTen[partialDigits]);
		BigWord value = getBinaryGroup(position, partialBytes, ptr);
		bigInt->add(0, value);
		}
}

void ScaledBinary::putBigInt(BigInt *bigInt, char* ptr, int precision, int scale)
{
	char *p = ptr;
	int digits = precision - scale;
	int groups = digits / DIGITS_PER_INT;
	int partialDigits = digits % DIGITS_PER_INT;
	BigInt number(bigInt), *hack = &number;
	uint32 fraction = number.scaleTo(0);
	uint32 data[20];
	int n;
	
	for (n = 0; n < groups; ++n)
		data[n] = (number.length) ? hack->divide(BASE) : 0;
			
	if (partialDigits)
		putBinaryGroup((number.length) ? hack->divide(BASE) : 0, partialDigits, &p);
	
	while (--n >= 0)
		putBinaryGroup(data[n], DIGITS_PER_INT, &p);
	
	// Now handle fraction

	if (scale)
		putBinaryNumber(fraction, scale, &p, true);
	
	if (bigInt->neg)
		for (char *q = ptr; q < p; ++q)
			*q ^= -1;
	
	*ptr ^= 0x80;
}

int ScaledBinary::numberBytes(int precision, int fractions)
{
	return numberBytes(precision - fractions) + numberBytes(fractions);
}
