/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// SyncObject.cpp: implementation of the SyncObject class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <memory.h>

#ifdef _WIN32
#include <windows.h>
#undef ERROR
#undef ASSERT
#undef TRACE
#endif

#ifdef ENGINE
#define TRACE
#else
#define ASSERT(b)
#endif

#include "Engine.h"
#include "SyncObject.h"
#include "Thread.h"
#include "Threads.h"
#include "Sync.h"
#include "Interlock.h"
#include "LinkedList.h"
#include "Log.h"
#include "SQLError.h"

//#define STALL_THRESHOLD	1000

/***
#ifdef _DEBUG
static char THIS_FILE[]=__FILE__;
#endif
***/

static int cas_emulation (volatile int *state, int compare, int exchange);

#ifdef EMULATE
#undef COMPARE_EXCHANGE
#define COMPARE_EXCHANGE(target,compare,exchange) \
	(cas_emulation(target,compare,exchange) == compare)
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

int cas_emulation (volatile int *state, int compare, int exchange)
{
	int result = *state;

	if (result == compare)
		*state = exchange;
	
	return result;
}

SyncObject::SyncObject()
{
	waiters = 0;
	lockState = 0;
	que = NULL;
	monitorCount = 0;
	stalls = 0;
	exclusiveThread = NULL;
}

SyncObject::~SyncObject()
{
}

void SyncObject::lock(Sync *sync, LockType type)
{
	Thread *thread;
	
	if (type == Shared)
		{
		for(;;)
			{
			int oldState = lockState;

			if (oldState < 0)
				break;

			int newState = oldState + 1;

			if (COMPARE_EXCHANGE(&lockState,oldState,newState))
				return;
			}

		mutex.lock();
		bumpWaiters(1);

		for(;;)
			{
			int oldState = lockState;

			if (oldState < 0)
				break;
				
			int newState = oldState + 1;

			if (COMPARE_EXCHANGE(&lockState, oldState, newState))
				{
				bumpWaiters(-1);
				mutex.release();
				
				return;
				}
			}

		thread = Thread::getThread("SyncObject::lock");

		if (thread == exclusiveThread)
			{
			++monitorCount;
			bumpWaiters(-1);
			mutex.release();
			
			return;
			}
		}
	else
		{
		thread = Thread::getThread("SyncObject::lock");
		ASSERT(thread);

		if (thread == exclusiveThread)
			{
			++monitorCount;
			
			return;
			}

		while (waiters == 0)
			{
			int oldState = lockState;
			
			if (oldState != 0)
				break;
				
			if (COMPARE_EXCHANGE(&lockState, oldState, -1))
				{
				exclusiveThread = thread;
				return; 
				}
			}
			
		mutex.lock();
		Thread *oldExclusive = exclusiveThread;
		//ASSERT(!oldExclusive || (oldExclusive->threadId != thread->threadId));
		bumpWaiters(1);
		
		while (que == NULL)
			{
			int oldState = lockState;
			
			if (oldState != 0)
				break;
				
			if (COMPARE_EXCHANGE(&lockState, oldState, -1))
				{
				exclusiveThread = thread;
				bumpWaiters(-1);
				mutex.release();
				
				return;
				}
			}
		}

	wait (type, thread, sync);
}

void SyncObject::unlock(Sync *sync, LockType type)
{
	if (monitorCount)
		{
		ASSERT (monitorCount > 0);
		--monitorCount;

		return;
		}
		

	for (;;)
		{
		ASSERT ((type == Shared && lockState > 0) || (type == Exclusive && lockState == -1));
		int oldState = lockState;
		int newState = (type == Shared) ? oldState - 1 : 0;
		exclusiveThread = NULL;
		
		if (COMPARE_EXCHANGE(&lockState, oldState, newState))
			{
			if (waiters)
				grantLocks();
				
			return;
			}
		}
}

void SyncObject::downGrade(LockType type)
{
	ASSERT (monitorCount == 0);
	ASSERT (type == Shared);
	ASSERT (lockState == -1);
	
	for (;;)
		if (COMPARE_EXCHANGE(&lockState, -1, 1))
			{
			if (waiters)
				grantLocks();
				
			return;
			}
}

void SyncObject::wait(LockType type, Thread *thread, Sync *sync)
{
	++stalls;

#ifdef STALL_THRESHOLD
	if ((stalls % STALL_THRESHOLD) == STALL_THRESHOLD - 1)
		frequentStaller(thread, sync);
#endif

	Thread *volatile *ptr;
	
	for (ptr = &que; *ptr; ptr = &(*ptr)->que)
		if (*ptr == thread)
			{
			LOG_DEBUG ("Apparent single thread deadlock for thread %d (%x)\n", thread->threadId, thread);
			
			for (Thread *thread = que; thread; thread = thread->que)
				thread->print();
				
			mutex.release();
			throw SQLEXCEPTION (BUG_CHECK, "Single thread deadlock");
			}

	thread->que = NULL;
	thread->lockType = type;
	*ptr = thread;
	thread->lockGranted = false;
	thread->lockPending = sync;
	++thread->activeLocks;
	mutex.release();

	while (!thread->lockGranted)
		{
		bool wakeup = thread->sleep (10000);
		
		if (thread->lockGranted)
			break;
			
		if (!wakeup)
			{
			stalled (thread);
			break;
			}
		}

	while (!thread->lockGranted)
		thread->sleep();
}

bool SyncObject::isLocked()
{
	return lockState != 0;
}


void SyncObject::stalled(Thread *thread)
{
#ifdef TRACE
	ThreadState threadState;
	thread->saveState(&threadState);
	Sync *lockPending = thread->lockPending;
	
	Sync sync(&Log::syncObject, "SyncObject::stalled");
	sync.lock(Exclusive);
	mutex.lock();
	LinkedList threads;
	LinkedList syncObjects;
	thread->findLocks (threads, syncObjects);

	LOG_DEBUG ("Stalled threads\n");

	FOR_OBJECTS (Thread*, thrd, &threads)
		thrd->print();
	END_FOR;

	LOG_DEBUG ("Stalled synchronization objects:\n");

	FOR_OBJECTS (SyncObject*, syncObject, &syncObjects)
		syncObject->print();
	END_FOR;

	LOG_DEBUG ("------------------------------------\n");
	mutex.release();

	thread->restoreState(&threadState);
	thread->lockPending = lockPending;
#endif
}

void SyncObject::findLocks(LinkedList &threads, LinkedList &syncObjects)
{
#ifdef TRACE
	if (syncObjects.appendUnique (this))
		{
		for (Thread *thread = que; thread; thread = thread->que)
			thread->findLocks (threads, syncObjects);
			
		if (exclusiveThread)
			exclusiveThread->findLocks (threads, syncObjects);
		}
#endif
}

void SyncObject::print()
{
#ifdef TRACE
	LOG_DEBUG ("  SyncObject %lx: state %d, monitor %d, waiters %d\n", 
				this, lockState, monitorCount, waiters);

	if (exclusiveThread)
		exclusiveThread->print ("    Exclusive thread");

	for (Thread *volatile thread = que; thread; thread = thread->que)
		thread->print ("    Waiting thread");
#endif
}


void SyncObject::sysServiceFailed(const char* service, int code)
{
	throw SQLEXCEPTION (BUG_CHECK, "Single thread deadlock");
}

void SyncObject::bumpWaiters(int delta)
{
	for (;;)
		{
		int oldValue = waiters;
		int newValue = waiters + delta;
		
		if (COMPARE_EXCHANGE(&waiters, oldValue, newValue))
			return;
		}
}

void SyncObject::grantLocks(void)
{
	mutex.lock();
	ASSERT ((waiters && que) || (!waiters && !que));
	const char *description = NULL;

	for (Thread *thread = que, *prior = NULL, *next; thread; thread = next)
		{
		description = thread->description;
		bool granted = false;
		next = thread->que;
				
		if (thread->lockType == Shared)
			for (int oldState; (oldState = lockState) >= 0;)
				{
				int newState = oldState + 1;
				
				if (COMPARE_EXCHANGE(&lockState, oldState, newState))
					{
					granted = true;
					exclusiveThread = NULL;
					break;
					}
				}
		else
			{
			ASSERT(thread->lockType == Exclusive);
			
			while (lockState == 0)
				if (COMPARE_EXCHANGE(&lockState, 0, -1))
					{
					granted = true;
					exclusiveThread = thread;
					break;
					}
			}
		
		if (granted)
			{
			if (prior)
				prior->que = next;
			else
				que = next;
			
			bool shutdownInProgress = thread->shutdownInProgress;
			
			if (shutdownInProgress)
				Thread::lockExitMutex();
					
			bumpWaiters(-1);
			--thread->activeLocks;
			thread->grantLock (this);
			
			if (shutdownInProgress)
				Thread::unlockExitMutex();
			}
		else
			prior = thread;
		}
			
	mutex.release();
}

int SyncObject::getState(void)
{
	return lockState;
}

void SyncObject::validate(LockType lockType)
{
	switch (lockType)
		{
		case None:
			ASSERT (lockState == 0);
			break;
		
		case Shared:
			ASSERT (lockState > 0);
			break;
		
		case Exclusive:
			ASSERT (lockState == -1);
			break;
		
		case Invalid:
			break;
		}
}

void SyncObject::unlock(void)
{
	if (lockState > 0)
		unlock (NULL, Shared);
	else if (lockState == -1)
		unlock (NULL, Exclusive);
	else
		ASSERT(false);
}

bool SyncObject::ourExclusiveLock(void)
{
	if (lockState != -1)
		return false;

	return exclusiveThread == Thread::getThread("SyncObject::ourExclusiveLock");
}

void SyncObject::frequentStaller(Thread *thread, Sync *sync)
{
	Thread *threadQue = thread->que;
	LockType lockType = thread->lockType;
	bool lockGranted = thread->lockGranted;
	Sync *lockPending = thread->lockPending;

	if (sync)
		LOG_DEBUG("Frequent stall from %s\n", sync->where);
	else
		LOG_DEBUG("Frequent stall from unknown\n");
		
	thread->que = threadQue;
	thread->lockType = lockType;
	thread->lockGranted = lockGranted;
	thread->lockPending = lockPending;
}
