/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: base
// Release: 0.1
//
// @doc
//
// @module ROWSCROLL.CPP | IRowsetLocate interface implementation
//
//

// Includes ---------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"

#define START_FROM_BOF 1 
#define START_FROM_EOF -1
#define START_FROM_NUM 0
 
// Code -------------------------------------------------------------------------

// CImpIRowsetScroll::GetApproximatePosition-------------------------------------
//
// @mfunc Gets the approximate position of a row corresponding to a specified bookmark
//
// @rdesc HRESULT
//		Return Code 
//      @flag S_OK				| The method succeeded. 
//      @flag E_FAIL			| A provider-specific error occurred. 
//      @flag E_INVALIDARG		| cbBookmark was not zero and pBookmark was a null pointer. 
//      @flag E_UNEXPECTED		| ITransaction::Commit or ITransaction::Abort was called and 
// the object is in a zombie state. 
//      @flag DB_E_BADBOOKMARK	| *pBookmark was invalid, incorrectly formed, or DBBMK_INVALID. 
//								| *pBookmark did not match any of the rows in the rowset. 
// This includes the case when the row corresponding to the bookmark has been deleted.
//      @flag DB_E_BADCHAPTER	| The rowset was chaptered and hChapter was invalid. 
//								| The rowset was single-chaptered, and the specified chapter 
// was not the currently open chapter. The consumer must use the currently open chapter or 
// release the currently open chapter before specifying a new chapter.
//      @flag DB_E_NOTREENTRANT | The provider called a method from IRowsetNotify in the
// consumer that had not yet returned, and the provider does not support reentrancy in this 
// method. 
//
STDMETHODIMP CImpIRowsetScroll::GetApproximatePosition 
	(
	HCHAPTER        hChapter,		//@parm IN	| The chapter handle
	ULONG           cbBookmark,		//@parm IN	| The length in bytes of the bookmark
	const BYTE *    pBookmark,		//@parm IN	| A pointer to a bookmark  
	ULONG *         pulPosition,	//@parm IN	| The position of the row 
	ULONG *         pcRows			//@parm IN	| The total number of rows 
	)
{
	INTERFACE_METHOD_START( "IRowsetScroll::GetApproximatePosition" );

	// default: The method succeeded. 
	HRESULT hr = S_OK;

	ULONG  ulTnRows = 0;	// The total number of rows
	double dfPos = 0;		// The position of the row (fractional)

	// Check for invalid arguments
	if ((cbBookmark != 0) && (pBookmark == NULL))
		return (E_INVALIDARG);
	
	// We can make rough check for DB_E_BADBOOKMARK, because our bookmark is 
	// always of PHYSICAL_POS_BUF_LEN length
	// Bookmark length can be either "our custom" or standart -- or to be ignored
	if ( !(cbBookmark == PHYSICAL_POS_BUF_LEN * m_pObj->m_pData->bookmarks() || cbBookmark == 1 || cbBookmark == 0 ))
		return DB_E_BADBOOKMARK;

	// Count the total number of rows 
	hr = m_pObj->m_pData->GetRowCnt(&ulTnRows);
	if (hr != S_OK)
		return hr;

	if (pcRows) 
		*pcRows = ulTnRows;
	
	if (ulTnRows == 0)
	{
		if (pulPosition) 
			*pulPosition = 0;
		return (S_OK);
	}

	if (cbBookmark == 0)
	{
		//we just requested to return number of rows
		//we do it already
		return S_OK;
	}

	// nothing to return in pulPosition and we already know the total number of rows
	if (pulPosition == NULL) 
		return (S_OK);	

	// special case: DBBMK_FIRST
	if ((cbBookmark == 1) && (*((BYTE*)pBookmark) == DBBMK_FIRST))
	{
		*pulPosition = 1;
		return (S_OK);
	}

	// special case: DBBMK_LAST
	if ((cbBookmark == 1) && (*((BYTE*)pBookmark) == DBBMK_LAST))
	{
		// Get actual number of rows
		*pulPosition = ulTnRows;
		return S_OK;
	}

	// get approximate position of the row
	hr = m_pObj->m_pData->GetPercentage((ULONG*)pBookmark, &dfPos);

	if (hr == S_OK)
		*pulPosition = (int)(dfPos * ulTnRows);

	return hr;

	INTERFACE_METHOD_END();
}



// CImpIRowsetScroll::GetRowsAtRatio --------------------------------------------
//
// @cmember GetRowsAtRatio Method 
// @mfunc Fetches rows starting from a fractional position in the rowset
//
// @rdesc HRESULT
//		Return Code 
//      @flag S_OK					| The method succeeded. 
//      @flag DB_S_ENDOFROWSET		| GetRowsAtRatio reached the start or the end of the rowset.
//      @flag DB_S_ROWLIMITEXCEEDED | Fetching the number of rows specified in cRows would have 
// exceeded the total number of active rows supported by the rowset. 
//      @flag DB_S_STOPLIMITREACHED | Fetching rows required further execution of the command, 
// such as when the rowset uses a server-side cursor. Execution was stopped because a resource 
// limit was reached. The number of rows that were actually fetched is returned in *pcRowsObtained. 
//      @flag E_FAIL				| A provider-specific error occurred. 
//      @flag E_INVALIDARG			| pcRowsObtained or prghRows was a null pointer. 
//      @flag E_OUTOFMEMORY			| The provider was unable to allocate sufficient memory in
// which to instantiate the rows or return the row handles. 
//      @flag E_UNEXPECTED			| ITransaction::Commit or ITransaction::Abort was called and
// the object is in a zombie state. 
//      @flag DB_E_BADCHAPTER		| The rowset was chaptered and hChapter was invalid. 
//									| The rowset was single-chaptered and the specified chapter
// was not the currently open chapter. 
//      @flag DB_E_BADRATIO			| ulNumerator was greater than ulDenominator. 
//									| ulDenominator was zero.
//      @flag DB_E_CANTFETCHBACKWARDS | cRows was negative and the rowset cannot fetch backward.
//      @flag DB_E_NOTREENTRANT		| The consumer called this method while it was processing a 
// notification, and it is an error to call this method while processing the specified DBREASON
// value. 
//      @flag DB_E_ROWSNOTRELEASED	| The provider requires release of existing rows before new
// ones can be fetched. 
//      @flag DB_SEC_E_PERMISSIONDENIED | The consumer did not have sufficient permission to 
// fetch any of the rows; no rows were fetched. 
//
STDMETHODIMP CImpIRowsetScroll::GetRowsAtRatio 
	(
	HWATCHREGION    hReserved1,		//@parm IN | Reserved for future use. 
	HCHAPTER        hChapter,		//@parm IN | The chapter handle. 
	ULONG           ulNumerator,	//@parm IN | ulNumerator.
	ULONG           ulDenominator,	//@parm IN | ulDenominator.
	LONG            cRows,			//@parm IN | The number of rows to fetch. 
	ULONG *         pcRowsObtained,	//@parm IN | the number of rows fetched. 
	HROW **         prghRows		//@parm IN | an array of handles of the fetched rows. 
	)
{
	INTERFACE_METHOD_START( "IRowsetScroll::GetRowsAtRatio" );

	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

	// default: The method succeeded. 
	HRESULT hr = S_OK;
    
	BOOL	bCanFetchBackwards;		// We can support DBPROP_CANFETCHBACKWARDS 
	BOOL	bCanScrollBackwards;	// We can support DBPROP_CANSCROLLBACKWARDS 
	ULONG	ulPropIndex;			// Number of property
	
	short int siStartFrom;			// Start from begin of File
	ULONG	ulStartRow;				// # of starting row
	ULONG	cTnRows;				// Total number of rows

    // Check validity of arguments.
    if (pcRowsObtained == NULL || prghRows == NULL)
        return E_INVALIDARG;

	if (ulDenominator == 0 || ulDenominator < ulNumerator)
		return DB_E_BADRATIO;

    // init out-params
    *pcRowsObtained = 0;

    // Look if consumer requested supporting scrolling and fetching backward.
	if (m_pObj->m_pUtilProp ->GetPropIndex(DBPROP_CANFETCHBACKWARDS, &ulPropIndex) == TRUE)
		bCanFetchBackwards = m_pObj->m_pUtilProp -> m_rgproperties[ulPropIndex].boolVal;

	if (m_pObj->m_pUtilProp -> GetPropIndex(DBPROP_CANSCROLLBACKWARDS, &ulPropIndex) == TRUE)
		bCanScrollBackwards = m_pObj->m_pUtilProp ->m_rgproperties[ulPropIndex].boolVal;
    
	// Look if given parameters correspond with se properties
	if ((cRows < 0) && (!bCanFetchBackwards))  
        return DB_E_CANTFETCHBACKWARDS;
	
    // Fetch Data
    //
	// allocate memory for temporary storage  record position 	
	// GetRowsAtRatio Specific
	
	// Calculate the number of skipped rows
	// default
	siStartFrom = START_FROM_NUM;

	// Check for margin conditions
	if (ulNumerator == 0 && cRows > 0) 
		siStartFrom = START_FROM_BOF;
	else if (ulNumerator == 0 && cRows < 0)
		return DB_S_ENDOFROWSET;
	else if (ulNumerator == ulDenominator && cRows > 0)
		return DB_S_ENDOFROWSET;
	else if (ulNumerator == ulDenominator && cRows < 0)
		siStartFrom = START_FROM_EOF;
	
	BYTE btBookmark;

	// make decision
	switch (siStartFrom)
	{
	case START_FROM_BOF:
		// Set position at the first row in the table
		btBookmark = DBBMK_FIRST;
		hr = m_pObj->MoveToBookmark(&btBookmark, 1);
		if (hr != S_OK)
			return hr;
		break;
	
	case START_FROM_EOF:
		// Set position at the last row in the table
		btBookmark = DBBMK_LAST;
		hr = m_pObj->MoveToBookmark(&btBookmark, 1);
		if (hr != S_OK)
			return hr;
		break;
	
	case START_FROM_NUM:
		// calculate the # of row to begin with
		hr = m_pObj->m_pData->GetRowCnt(&cTnRows);
		if (hr != S_OK)
			return hr;

		ulStartRow = (ULONG) ((double(ulNumerator) / double(ulDenominator)) * double(cTnRows));

		// SetPosition to the row with # ulStartRow
		// restart position
		// Set position at the first row in the table
		btBookmark = DBBMK_FIRST;
		hr = m_pObj->MoveToBookmark(&btBookmark, 1);
		if (hr != S_OK)
			return hr;
	
		// skip ulStartRow rows
		if (ulStartRow > 0)
		{
			hr = m_pObj->m_pData->Skip(ulStartRow, FALSE);
			if (hr != S_OK)
				return hr;
		}

		break;
	}

	// now retreive rows from data source with GetNextRows
	return m_pObj->m_pIRowset->GetNextRows(false, NULL, 0, cRows, pcRowsObtained, prghRows);

	INTERFACE_METHOD_END();
}


// CImpIRowsetScroll::Compare
//
// @mfunc	Compares two bookmarks.
//
// @desc	HRESULT
//		@flag	S_OK			| The method succeeded. 
//		@flag	E_FAIL			| A provider-specific error occurred. 
//		@flag	E_INVALIDARG	| cbBookmark1 or cbBookmark2 was zero. 
//		@flag	DB_E_BADBOOKMARK| pBookmark1 or pBookmark2 was zero
//		@flag	E_POINTER		| pComparison was zero

//	DBCOMPARE_LT				The first bookmark is before the second.
//	DBCOMPARE_EQ				The two bookmarks are equal.
//	DBCOMPARE_GT				The first bookmark is after the second.
//	DBCOMPARE_NE				The bookmarks are not equal and not ordered.
//	DBCOMPARE_NOTCOMPARABLE		The two bookmarks cannot be compared.

STDMETHODIMP	CImpIRowsetScroll::Compare(
		HCHAPTER	hReserved, 
		ULONG		cbBookmark1, 
		const BYTE*	pBookmark1,
		ULONG		cbBookmark2,
		const BYTE*	pBookmark2,
		DBCOMPARE*	pComparison)
{
	INTERFACE_METHOD_START( "IRowsetScroll::Compare" );
	
	// Default is run CImpIRowsetLocate::Compare
	return (m_pObj->m_pIRowsetLocate->Compare(
										hReserved, 
										cbBookmark1, 
										pBookmark1,
										cbBookmark2,
										pBookmark2,
										pComparison));

	INTERFACE_METHOD_END();
}


// CImpIRowsetScroll::GetRowsAt
//
// @mfunc	Fetches rows starting with the row specified by an 
//			offset from a bookmark.
//
// @desc	HRESULT
//		@flag	S_OK			| The method succeeded. 
//		@flag	E_FAIL			| A provider-specific error occurred. 
//		@flag	E_INVALIDARG	| cbBookmark was zero. 
//		@flag	E_POINTER		| pcRowsObtained or prghRows was zero
//		@flag	DB_E_BADBOOKMARK| pBookmark was zero

STDMETHODIMP	CImpIRowsetScroll::GetRowsAt(
			HWATCHREGION	hReserved1,		// IN
			HCHAPTER		hReserved2,		// IN
			ULONG			cbBookmark,		// IN
			const BYTE*		pBookmark,		// IN
			LONG			lRowsOffset,	// IN
			LONG			cRows,			// IN
			ULONG*			pcRowsObtained,	// OUT
			HROW**			prghRows)		// OUT
{
	INTERFACE_METHOD_START( "IRowsetScroll::GetRowsAt" )
	
	// Default is run CImpIRowsetLocate::GetRowsAt
	return (m_pObj->m_pIRowsetLocate->GetRowsAt(
											hReserved1,		// IN
											hReserved2,		// IN
											cbBookmark,		// IN
											pBookmark,		// IN
											lRowsOffset,	// IN
											cRows,			// IN
											pcRowsObtained,	// OUT
											prghRows));		// OUT

	INTERFACE_METHOD_END();
}


//------------------------------------------------------------------------------
// CImpIRowsetScroll::GetRowsByBookmark
//
// @cmember Fetches the rows that match the specified bookmarks.
//
// @desc
//		@flag	S_OK			| The method succeeded. 
//		@flag	E_FAIL			| A provider-specific error occurred. 
//		@flag	E_INVALIDARG	| cbBookmark or rghRows was zero. 
//
STDMETHODIMP	CImpIRowsetScroll::GetRowsByBookmark (
			HCHAPTER		hReserved,
			ULONG			cRows,
			const ULONG		rgcbBookmarks[],
			const BYTE*		rgpBookmarks[],
			HROW			rghRows[],
			DBROWSTATUS		rgRowStatus[]
)
{
	INTERFACE_METHOD_START( "IRowsetScroll::GetRowsByBookmark" );
	
	// Default is run CImpIRowsetLocate::GetRowsByBookmark
	return (m_pObj->m_pIRowsetLocate->GetRowsByBookmark(
										hReserved,
										cRows,
										rgcbBookmarks,
										rgpBookmarks,
										rghRows,
										rgRowStatus));

	INTERFACE_METHOD_END();
}



// CImpIRowsetScroll::Hash
//
// @mfunc Returns hash values for the specified bookmarks.
//
// @desc
//

STDMETHODIMP	CImpIRowsetScroll::Hash (
			HCHAPTER		hReserved,
			ULONG			cBookmarks,
			const ULONG		rgcbBookmarks[],
			const BYTE*		rgpBookmarks[],
			DWORD			rgHashedValues[],
			DBROWSTATUS		rgBookmarkStatus[])
{
	INTERFACE_METHOD_START( "IRowsetScroll::Hash" );
	
	// Default is run CImpIRowsetLocate::Hash
	return (m_pObj->m_pIRowsetLocate->Hash(
										hReserved,
										cBookmarks,
										rgcbBookmarks,
										rgpBookmarks,
										rgHashedValues,
										rgBookmarkStatus));

	INTERFACE_METHOD_END();
}


// CImpIRowsetScroll::GetData
//
// @mfunc Simply run IRowset::GetData implementation.
//
// @desc
//
STDMETHODIMP	CImpIRowsetScroll::GetData 
    (
    HROW        hRow,       //@parm IN | Row Handle
    HACCESSOR   hAccessor,  //@parm IN | Accessor to use
    void       *pData       //@parm OUT | Pointer to buffer where data should go.
    )
{
	INTERFACE_METHOD_START( "IRowsetScroll::GetData" );
	
	return m_pObj->m_pIRowset -> GetData(hRow, hAccessor, pData);

	INTERFACE_METHOD_END();
}


// CImpIRowsetScroll::GetNextRows
//
// @mfunc Simply run IRowset::GetNextRows implementation.
//
// @desc
//
STDMETHODIMP	CImpIRowsetScroll::GetNextRows
    (
    HCHAPTER   hReserved,       //@parm IN | Reserved for future use. Ingored.
    LONG       cRowsToSkip,     //@parm IN | Rows to skip before reading
    LONG       cRows,           //@parm IN | Number of rows to fetch
    ULONG      *pcRowsObtained, //@parm OUT | Number of rows obtained
    HROW       **prghRows       //@parm OUT | Array of Hrows obtained
    )
{
	INTERFACE_METHOD_START( "IRowsetScroll:GetNextRows" );
	
	return m_pObj-> m_pIRowset-> GetNextRows(hReserved, cRowsToSkip, cRows, pcRowsObtained, prghRows);

	INTERFACE_METHOD_END();
}

// CImpIRowsetScroll::ReleaseRows
// @mfunc Simply run IRowset::ReleaseRows implementation.
//
// @desc
//
STDMETHODIMP	CImpIRowsetScroll::ReleaseRows
    (
    ULONG			cRows,			//@parm IN  | Number of rows to release
    const HROW		rghRows[],		//@parm IN  | Array of handles of rows to be released
	DBROWOPTIONS	rgRowOptions[],	//@parm IN  | Additional Options
    ULONG			rgRefCounts[],	//@parm OUT | Array of refcnts for the rows
	DBROWSTATUS     rgRowStatus[]	//@parm OUT | Array of row status
    )
{
	INTERFACE_METHOD_START( "IRowsetScroll::ReleaseRows" );
		
	return m_pObj->m_pIRowset->ReleaseRows(cRows, rghRows, rgRowOptions, rgRefCounts, rgRowStatus);

	INTERFACE_METHOD_END();
}


// CImpIRowsetScroll::RestartPosition
//
// @mfunc Simply run IRowset::RestartPosition implementation.
//
// @desc
//
STDMETHODIMP	CImpIRowsetScroll::RestartPosition
    (
    HCHAPTER    hReserved        //@parm IN | Reserved for future use.  Ignored.
    )
{
	INTERFACE_METHOD_START( "IRowsetScroll::RestertPosition" );
	
	return m_pObj->m_pIRowset -> RestartPosition(hReserved);

	INTERFACE_METHOD_END();
}


// CImpIRowsetScroll::AddRefRows
//
// @mfunc Simply run IRowset::AddRefRows implementation.
//
// @desc
//
STDMETHODIMP	CImpIRowsetScroll::AddRefRows
    (
    ULONG           cRows,          // @parm IN     | Number of rows to refcount
    const HROW      rghRows[],      // @parm IN     | Array of row handles to refcount
    ULONG           rgRefCounts[],  // @parm OUT    | Array of refcounts
    DBROWSTATUS     rgRowStatus[]   // @parm OUT    | Array of row status
	)
{
	INTERFACE_METHOD_START( "IRowsetScroll::AddRefRows" );
	
	return m_pObj->m_pIRowset->AddRefRows(cRows, rghRows, rgRefCounts, rgRowStatus);

	INTERFACE_METHOD_END();
}