/*
 * default font
 *
 * $Id: fontset.c,v 1.6 2002/03/26 10:46:59 taka Exp $
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>

#include "aime.h"
#include "queue.h"
#include "fontset.h"

struct fontsetentry
{
	Display*		display;
	XFontSet		xfontset;
	XFontSetExtents*	xfontsetextents;
	char*			fontsetname;
	int			ref_count;
	int			is_removable;

	SLIST_ENTRY (fontsetentry)	entry;
};
static SLIST_HEAD (,fontsetentry) fontset_head = SLIST_HEAD_INITIALIZER(0);

static struct fontsetentry*
register_fontsetentry (Display* d, const char* fontsetname)
{
	char**	missing;
	int	count;
	char*	defstr;
	struct fontsetentry* f;

	f = malloc (sizeof (struct fontsetentry));
	if (f == NULL) {
		NOMEMORY ("can't allocate fontsetentry");
		return NULL;
	}
	memset (f, 0, sizeof (f));

	f->display = d;
	f->fontsetname = strdup (fontsetname);
	if (f->fontsetname == NULL) {
		NOMEMORY ("can't allocate fontsetname for fontsetentry");
		free (f);
		return NULL;
	}
	f->xfontset = XCreateFontSet (f->display, f->fontsetname, &missing,
				      &count, &defstr);
	if (f->xfontset == NULL) {
		error_printf ("XCreateFontSet failed");
		free (f->fontsetname);
		free (f);
		return NULL;
	}
	f->xfontsetextents = XExtentsOfFontSet (f->xfontset);
	f->ref_count = 1;
	f->is_removable = 1;

	XFreeStringList (missing);
	SLIST_INSERT_HEAD (&fontset_head, f, entry);
	return f;
}
static void
unregister_fontsetentry (struct fontsetentry* f)
{
	SLIST_REMOVE (&fontset_head, f, fontsetentry, entry);
	
	XFreeFontSet (f->display, f->xfontset);
	free (f->fontsetname);
	free (f);
}
static struct fontsetentry*
search_fontsetentry_byname (const char* fontsetname)
{
	struct fontsetentry* f;
	SLIST_FOREACH (f, &fontset_head, entry) {
		if (strcmp (f->fontsetname, fontsetname) == 0) {
			return f;
		}
	}
	return NULL;
}

struct fontsetentry*
fontset_get_byname (Display* d, const char* fontsetname)
{
	struct fontsetentry* f;
	f = search_fontsetentry_byname (fontsetname);

	if (f == NULL) {
		return register_fontsetentry (d, fontsetname);
	} else {
		f->ref_count++;
		return f;
	}
}
void
fontset_release (struct fontsetentry* f)
{
	if (!f) return;
	f->ref_count --;

	if (f->ref_count == 0 && f->is_removable == 1) {
		unregister_fontsetentry (f);
	}
}
struct fontsetentry*
fontset_duplicate (struct fontsetentry* f)
{
	f->ref_count ++;
	return f;
}

XFontSet
fontset_get_xfontset (struct fontsetentry* f)
{
	return f->xfontset;
}
XFontSetExtents*
fontset_get_xfontsetextents (struct fontsetentry* f)
{
	return f->xfontsetextents;
}
char*
fontset_get_fontsetname (struct fontsetentry* f)
{
	return f->fontsetname;
}

int
fontset_initialize_module (Display* d)
{
	struct fontsetentry* f;
	f = register_fontsetentry (d, DEFAULT_FONTSETNAME);
	if (f == NULL) {
		fprintf (stderr, "XCreateFontSet Failed.\n");
		return -1;
	}

	/*
	 * this fontsetentry is used for default value.
	 * so can't remove thins entry;
	 */
	f->is_removable = 0;
	f->ref_count ++;
	return 0;
}
void
fontset_destroy_module (void)
{
	struct fontsetentry* f;

	f = SLIST_FIRST (&fontset_head);
	while (f) {
		struct fontsetentry* next = SLIST_NEXT (f, entry);

		unregister_fontsetentry (f);
		
		f = next;
	}
}
