/*
 * xim protocol procedure
 *
 * $Id: ximproc.c,v 1.31 2002/03/26 10:47:00 taka Exp $
 */
#include <stdio.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xproto.h>

#include "ximpacket.h"
#include "imconnection.h"
#include "ximic.h"
#include "queue.h"
#include "byteorder.h"
#include "xim.h"
#include "conversion.h"
#include "aime.h"
#include "pewindow.h"
#include "conversion.h"
#include "ximproc.h"
#include "charcode.h"
#include "control.h"

#define PUT8(imc,x,v) ximpacket_put8 ((x), (v), (imc)->byte_order)
#define PUT16(imc,x,v) ximpacket_put16 ((x), (v), (imc)->byte_order)
#define PUT32(imc,x,v) ximpacket_put32 ((x), (v), (imc)->byte_order)
#define PUTBYTES(imc,x,v,len) \
	ximpacket_putbytes ((x), (v), (len), (imc)->byte_order)

static int
send_error_packet (struct imconnection* imc, int errorcode, int imid,
		   int icid, const char* msg)
{
	int ret;
	struct ximpacket* x;

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = ximpacket_error_packet (x, errorcode, imid, icid, msg,
				      imc->byte_order);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
static int
send_bad_length (struct imconnection* imc, int imid, int icid)
{
	return send_error_packet (imc, XIM_ERROR_BADSOMETHING, imid, icid,
				  "bad length");
}
int
ximp_bad_length (struct ximpacket* x, int imid, int icid, int border)
{
	return ximpacket_error_packet (x, XIM_ERROR_BADSOMETHING,
				       imid, icid, "bad length", border);
}
static int
send_invalid_imid (struct imconnection* imc, int imid, int icid)
{
	return send_error_packet (imc, XIM_ERROR_BADSOMETHING, imid, icid,
				  "invalid input-method ID");
}
static struct ximim*
search_ximim (struct imconnection* imc, int imid)
{
	struct ximim* xim;
	SLIST_FOREACH (xim, &imc->imlist, entry) {
		if (xim->imid == imid) {
			return xim;
		}
	}
	return NULL;
}
static struct ximic*
search_ximic (struct imconnection* imc, int imid, int icid)
{
	struct ximim* xim;
	xim = search_ximim (imc, imid);
	if (xim) {
		return ximim_search_ximic (xim, icid);
	} else {
		return NULL;
	}
}

int
ximp_bad_protocol (struct imconnection* imc, int major, int minor, int len)
{
	return send_error_packet (imc, XIM_ERROR_BADPROTOCOL, 0, 0,
				  "Clients should not send this request");
}
int
ximp_connect (struct imconnection* imc, int major, int minor, int len)
{
	return send_error_packet (imc, XIM_ERROR_BADPROTOCOL, 0, 0,
				  "duplicate XIM_CONNECT message");
}
int
ximp_disconnect (struct imconnection* imc, int major, int minor, int len)
{
	int ret;
	struct ximpacket* x;

	if (len != 0) send_bad_length (imc, 0, 0);

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = ximpacket_simple_request (x, XIM_DISCONNECT_REPLY, 0);

	if (ret != 0) return -1;

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	imc->state = IMC_STATE_CLOSE;
	return 0;
}
int
ximp_error (struct imconnection* imc, int major, int minor, int len)
{
	int l;
	char error_message[128];

	l = get16 (imc->buffer + 8, imc->byte_order);
	l = sizeof (error_message) - 1 > l ? l : sizeof (error_message) - 1;
	memcpy (error_message, imc->buffer + 12, l);
	error_message[l] = '\0';
	DPRINTF ("xim_error: len=%d %s\n", l, error_message);
	return 0;
}
int
ximp_open (struct imconnection* imc, int major, int minor, int len)
{
	struct ximpacket* x1;
	struct ximpacket* x2;
	struct ximim*     xim;
	char localename[256];
	int locale_len;
	int ret;

	if (len < 1) return send_bad_length (imc, 0, 0);

	locale_len = get8 (imc->buffer, imc->byte_order);
	memcpy (localename, imc->buffer + 1, locale_len);
	localename[locale_len] = '\0';
	DPRINTF (" xim open with locale[%s]\n", localename);

	x1 = ximpacket_allocate ();
	x2 = ximpacket_allocate ();
	if (x1 == NULL || x2 == NULL) {
		ximpacket_destroy (x1);
		ximpacket_destroy (x2);
		return -1;
	}
	xim = ximim_allocate ();
	if (xim == NULL) {
		ximpacket_destroy (x1);
		ximpacket_destroy (x2);
		return -1;
	}

	/* make XIM_OPEN_REPLY */
	ret = 0;
	ret |= ximpacket_putheader (x1, XIM_OPEN_REPLY, 0, 0, imc->byte_order);
	ret |= PUT16 (imc, x1, xim->imid);
	ret |= ximim_im_putattributes (x1, imc->byte_order);
	ret |= ximic_ic_putattributes (x1, imc->byte_order);
	if (ret != 0) {
		ximpacket_destroy (x1);
		ximpacket_destroy (x2);
		ximim_destroy (xim);
		return -1;
	}
	ximpacket_rewritelen (x1, imc->byte_order);

	/* make XIM_SET_EVENT_MASK */
	ret = 0;
	ret |= ximpacket_putheader (x2, XIM_SET_EVENT_MASK, 0, 0,
				    imc->byte_order);
	ret |= PUT16 (imc, x2, xim->imid);
	ret |= PUT16 (imc, x2, 0);
	ret |= PUT32 (imc, x2, KeyPressMask);
	ret |= PUT32 (imc, x2, KeyPressMask);
	if (ret != 0) {
		ximpacket_destroy (x1);
		ximpacket_destroy (x2);
		ximim_destroy (xim);
		return -1;
	}
	ximpacket_rewritelen (x2, imc->byte_order);


	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x1, entry);
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x2, entry);
	/* add ximic */
	SLIST_INSERT_HEAD (&imc->imlist, xim, entry);
	return 0;
}
int
ximp_close (struct imconnection* imc, int major, int minor, int len)
{
	int ret;
	int imid;
	struct ximpacket* x;
	struct ximim* xim;

	if (len < 2) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer, imc->byte_order);
	xim = search_ximim (imc, imid);
	if (xim == NULL) return send_invalid_imid (imc, imid, 0);

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_CLOSE_REPLY, 0, 4, imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= ximpacket_putpad (x, 2);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	SLIST_REMOVE (&imc->imlist, xim, ximim, entry);
	ximim_destroy (xim);
	return 0;
}
int
ximp_encoding_negotiation (struct imconnection* imc, int major, int minor,
			   int len)
{
	int imid;
	int slen;
	int offset = 0;
	int strindex = 0;
	int found = 0;
	int ret;
	struct ximpacket* x;
	struct ximim* xim;

	if (len < 4) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer, imc->byte_order);
	xim = search_ximim (imc, imid);
	if (xim == NULL) return send_invalid_imid (imc, imid, 0);

	slen = get16 (imc->buffer + 2, imc->byte_order);
	offset = 4;

	if (offset + slen > len) return send_bad_length (imc, imid, 0);

	while (offset < slen + 4) {
		char string[255 + 1];
		int l = get8 (imc->buffer + offset, imc->byte_order);

		offset += 1;
		if (offset >= len) return send_bad_length (imc, imid, 0);

		memcpy (string, imc->buffer + offset, l);
		string[l] = '\0';

		DPRINTF (" encoding negotiation [%s] comes\n", string);
		if (strcmp ("COMPOUND_TEXT", string) == 0) {
			found = 1;
			break;
		}
		offset += l;
		strindex ++;
	}

	if (found == 0) {
		return send_error_packet (imc, XIM_ERROR_BADSOMETHING,
					  imid, 0, "error argument");
	}

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_ENCODING_NEGOTIATION_REPLY, 0, 8,
				    imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= PUT16 (imc, x, 0);	/* encoding determined by name */
	ret |= PUT16 (imc, x, strindex);
	ret |= PUT16 (imc, x, 0);	/* padding */
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_query_extension (struct imconnection* imc, int major, int minor, int len)
{
	struct ximpacket* x;
	int ret;
	int imid;
	int extlen;
	int start, end;

	if (len < 4) return send_bad_length (imc, 0, 0);

	imid   = get16 (imc->buffer,     imc->byte_order);
	extlen = get16 (imc->buffer + 2, imc->byte_order);
	
	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_QUERY_EXTENSION_REPLY, 0, 0,
				    imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	start = ximpacket_position (x);
	ret |= PUT16 (imc, x, 0);

	if (extlen == 0) {
		struct imrequest* table;
		table = get_extrequest_table ();
		for (; table->name != NULL; table++) {
			int l;
			l = strlen (table->name);
			ret |= PUT8  (imc, x, table->major);
			ret |= PUT8  (imc, x, table->minor);
			ret |= PUT16 (imc, x, l);
			ret |= ximpacket_putpad (x, l);
		}
	} else {
		struct imrequest* table;
		int offset = 4;
		table = get_extrequest_table ();

		while (extlen > 1) {
			struct imrequest* t;
			int slen;
			char str[129];

			slen = get8 (imc->buffer + offset, imc->byte_order);
			offset ++;
			if (slen + 1 > extlen) {
				ximpacket_destroy (x);
				return send_bad_length (imc, imid, 0);
			}

			memcpy (str, imc->buffer + offset, slen);
			str[slen] = '\0';

			DPRINTF (" query extension [%s] comes\n", str);
			t = table;
			while (t->name != NULL) {
				if (strcmp (t->name, str)) {
					int l;
					l = strlen (t->name);
					ret |= PUT8  (imc, x, table->major);
					ret |= PUT8  (imc, x, table->minor);
					ret |= PUT16 (imc, x, l);
					ret |= ximpacket_putpad (x, l);
					break;
				}
				t++;
			}
			offset += slen;
			extlen -= slen + 1;
		}
	}
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	end = ximpacket_position (x);
	ximpacket_reput16 (x, end - start, start, imc->byte_order);
	ximpacket_rewritelen (x, imc->byte_order);
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_get_imvalues (struct imconnection* imc, int major, int minor, int len)
{
	int ret;
	int imid;
	int l;
	struct ximpacket* x;

	if (len < 4) return send_bad_length (imc, 0, 0);

	imid  = get16 (imc->buffer,     imc->byte_order);
	l     = get16 (imc->buffer + 2, imc->byte_order);

	if (len != 4 + l + PAD4(l)) return send_bad_length (imc, imid, 0);
	
	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_GET_IM_VALUES_REPLY, 0, 0,
				    imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= ximim_im_get_values (x, imc->buffer + 4, l, imc->byte_order);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	ximpacket_rewritelen (x, imc->byte_order);

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_createic (struct imconnection* imc, int major, int minor, int len)
{
	int imid;
	int ret;
	int l;
	struct ximpacket* x;
	struct ximim* xim;
	struct ximic* xic;

	if (len < 2) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer, imc->byte_order);
	xim = search_ximim (imc, imid);
	if (xim == NULL) return send_invalid_imid (imc, imid, 0);

	l    = get16 (imc->buffer + 2, imc->byte_order);
	if (l + 4 != len) return send_bad_length (imc, imid, 0);

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	xic = ximim_allocate_ic (xim, imc);
	if (xic == NULL) {
		goto ERROR;
	}

	xic->conv = conversion_allocate (xic);
	if (xic->conv == NULL) {
		ximim_destroy_ic (xim, xic);
		goto ERROR;
	}

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_CREATE_IC_REPLY, 0, 4,
				    imc->byte_order);
	ret |= PUT16 (imc, x, xic->imid);
	ret |= PUT16 (imc, x, xic->icid);
	if (ret != 0) {
		ximpacket_destroy (x);
		ximim_destroy_ic (xim, xic);
		return -1;
	}

	ret = ximic_ic_set_attributes (xic, x, imc->buffer + 4, l,
				       imc->byte_order);
	if (ret != 0) {
		ximim_destroy_ic (xim, xic);
		/* x is filled with XIM_ERROR */
		goto END;
	}

	xic->pew = pewindow_create (xic, imc->display,
				    xic->common_attr.input_style);
	if (xic->pew == NULL) {
		ximim_destroy_ic (xim, xic);
		goto ERROR;
	}

	DPRINTF ("createIC imid=%d icid=%d\n", xic->imid, xic->icid);
	
	goto END;

 ERROR:
	ximpacket_init_buffer (x);
	ximpacket_error_packet (x, XIM_ERROR_BADALLOC, imid, 0, "can't alloc",
				imc->byte_order);
 END:
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_destroyic (struct imconnection* imc, int major, int minor, int len)
{
	int imid, icid;
	struct ximim* xim;
	struct ximic* xic;
	struct ximpacket* x;
	int ret;

	if (len < 4) return send_bad_length (imc, 0, 0);
	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);

	xim = search_ximim (imc, imid);
	if (xim == NULL) return send_invalid_imid (imc, imid, icid);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_DESTROY_IC_REPLY, 0, 4,
				    imc->byte_order);
	ret |= PUT16 (imc, x, xim->imid);
	ret |= PUT16 (imc, x, xic->icid);

	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	DPRINTF ("destroyIC imid=%d icid=%d\n", xic->imid, xic->icid);
	ximim_destroy_ic (xim, xic);
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_set_icvalues (struct imconnection* imc, int major, int minor, int len)
{
	struct ximpacket* x;
	struct ximic* xic;
	int l;
	int imid;
	int icid;
	int ret;

	if (len < 8) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	l    = get16 (imc->buffer + 4, imc->byte_order);
	if (l + 8 != len) return send_bad_length (imc, imid, icid);

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = ximic_ic_set_attributes (xic, x, imc->buffer + 8, l,
				       imc->byte_order);
	if (ret == -1) {
		ximpacket_destroy (x);
		return -1;
	}

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_SET_IC_VALUES_REPLY, 0, 4,
				    imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= PUT16 (imc, x, icid);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_get_icvalues (struct imconnection* imc, int major, int minor, int len)
{
	struct ximpacket* x;
	struct ximic* xic;
	int l;
	int imid, icid;
	int ret;

	if (len < 6) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	l    = get16 (imc->buffer + 4, imc->byte_order);
	if (len != 6 + l + PAD4 (2 + l)) {
		return send_bad_length (imc, imid, icid);
	}

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_GET_IC_VALUES_REPLY, 0, 0,
				    imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= PUT16 (imc, x, icid);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	ret = ximic_ic_get_attributes (xic, x, imc->buffer + 6, l,
				       imc->byte_order);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	ximpacket_rewritelen (x, imc->byte_order);

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_set_icfocus (struct imconnection* imc, int major, int minor, int len)
{
	struct ximic* xic;
	int imid, icid;

	if (len < 4) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	conversion_focus_changed (xic->conv, 1);
	
	return 0;
}
int
ximp_unset_icfocus (struct imconnection* imc, int major, int minor, int len)
{
	struct ximic* xic;
	int imid, icid;

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	conversion_focus_changed (xic->conv, 0);

	return 0;
}
int
ximp_forward_event (struct imconnection* imc, int major, int minor, int len)
{
	struct ximic* xic;
	int imid, icid;
	int synchronous;
	unsigned int serial;
	int ret;
	int val;
	XEvent e;

	if (len != 8 + sizeof (xEvent)) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	val         = get16 (imc->buffer + 4, imc->byte_order);
	synchronous = val & 0x0001;
	serial      = get16 (imc->buffer + 6, imc->byte_order);

	/* convert xEvent to XEvent */
	ret = xim_convert_xEvent_to_XEvent (imc->display, imc->buffer + 8,
					    serial, &e, imc->byte_order);
	if (ret == -1) {
		return send_error_packet (imc, XIM_ERROR_BADSOMETHING,
					  imid, icid,
					  "can't conver xEvent to XEvent");
	}

	return invoke_key_press (xic->conv, &e, serial, imc->buffer + 8, 1);
}
int
ximp_sync (struct imconnection* imc, int major, int minor, int len)
{
	struct ximic* xic;
	int imid, icid;
	struct ximpacket* x;
	int ret;

	if (len != 4) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_SYNC_REPLY, 0, 4, imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= PUT16 (imc, x, icid);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}

	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);
	return 0;
}
int
ximp_resetic (struct imconnection* imc, int major, int minor, int len)
{
	struct ximic* xic;
	int imid, icid;
	struct ximpacket* x;
	int ret;

	if (len != 4) return send_bad_length (imc, 0, 0);

	imid = get16 (imc->buffer,     imc->byte_order);
	icid = get16 (imc->buffer + 2, imc->byte_order);
	xic = search_ximic (imc, imid, icid);
	if (xic == NULL) return send_invalid_imid (imc, imid, icid);

	ret = conversion_reset (xic->conv);
	if (ret == -1) return send_error_packet (imc, XIM_ERROR_BADSOMETHING,
						 0, 0, "can't reset ic");

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_RESET_IC_REPLY, 0, 8,
				    imc->byte_order);
	ret |= PUT16 (imc, x, imid);
	ret |= PUT16 (imc, x, icid);
	ret |= PUT16 (imc, x, 0);	/* skip ;p */
	ret |= ximpacket_putpad (x, 2);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);

	pewindow_clear (xic->pew);
	pewindow_update (xic->pew);

	return 0;
}

static int
send_sync (struct ximic* xic, int sync)
{
	struct ximpacket* x;
	int border = xic->imc->byte_order;
	int ret;

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	ret = 0;
	ret |= ximpacket_putheader (x, sync ? XIM_SYNC : XIM_SYNC_REPLY,
				    0, 4, border);
	ret |= ximpacket_put16 (x, xic->imid, border);
	ret |= ximpacket_put16 (x, xic->icid, border);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	SIMPLEQ_INSERT_TAIL (&xic->imc->packetlist, x, entry);
	return 0;
}
int
invoke_key_press (struct conversion* conv, XEvent* e, unsigned int serial,
		  char* xEventbuffer, int fromclient)
{
	enum {
		SEND_SYNC,
		SEND_SYNC_REPLY,
		SEND_NOTHING,
	};
	struct ximic* xic;
	struct imconnection* imc;
	int ret;
	int imid;
	int icid;
	int border;
	int type = SEND_NOTHING;

	xic  = conversion_get_xic (conv);
	imc = xic->imc;
	border = xic->imc->byte_order;
	imid = xic->imid;
	icid = xic->icid;


	ret = conversion_pushkey (xic->conv, (XKeyEvent*)e);

	switch (ret) {
	case CV_SEND_KEY_EVENT:
	{
		struct ximpacket* x;
		xEvent real_xe;
		char* xev;
		unsigned short sync = 0;

		if (xEventbuffer) {
			xev = xEventbuffer;
		} else {
			xev = (char*)&real_xe;
			xim_convert_XEvent_to_xEvent (imc->display, e,
						      &serial, &real_xe,
						      border);
		}
		
		x = ximpacket_allocate ();
		if (x == NULL) return -1;

		ret = 0;
		ret |= ximpacket_putheader (x, XIM_FORWARD_EVENT, 0, 0,
					    border);
		ret |= ximpacket_put16 (x, imid, border);
		ret |= ximpacket_put16 (x, icid, border);
		ret |= ximpacket_put16 (x, sync, border);
		ret |= ximpacket_put16 (x, serial, border);
		ret |= ximpacket_putbytes (x, xev, sizeof (xEvent), border);
		if (ret != 0) {
			ximpacket_destroy (x);
			return -1;
		}
		ximpacket_rewritelen (x, border);

		SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);

		if (fromclient) type = SEND_SYNC_REPLY;
		else            type = SEND_NOTHING;
		break;
	}
	case CV_SEND_COMMIT:
	{
		struct ximpacket* x;
		char* ctext;
		int clen;
		unsigned short commit_flag;
#define SYNCHRONOUS  0x0001
#define XLOOKUPCHARS 0x0002

		ctext = conversion_get_commited_ctext (xic->conv);
		if (ctext == NULL) return -1;
		x = ximpacket_allocate ();
		if (x == NULL) {
			conversion_free_commited_ctext (xic->conv);
			return -1;
		}

		clen = strlen (ctext);

		if (xic->common_attr.input_style == ROOT_WINDOW) {
			if (fromclient) {
				commit_flag = XLOOKUPCHARS;
				type = SEND_SYNC_REPLY;
			} else {
				commit_flag = XLOOKUPCHARS | SYNCHRONOUS;
				commit_flag = XLOOKUPCHARS;
				type = SEND_NOTHING;
			}
		} else {
			commit_flag = XLOOKUPCHARS | SYNCHRONOUS;
			type = SEND_SYNC_REPLY;
		}
		
		ret = 0;
		ret |= ximpacket_putheader (x, XIM_COMMIT, 0, 0, border);
		ret |= ximpacket_put16 (x, imid, border);
		ret |= ximpacket_put16 (x, icid, border);
		ret |= ximpacket_put16 (x, commit_flag, border);
		ret |= ximpacket_put16 (x, clen, border);
		ret |= ximpacket_putbytes (x, ctext, clen, border);
		ret |= ximpacket_putpad (x, clen);

		conversion_free_commited_ctext (xic->conv);

		if (ret != 0) {
			ximpacket_destroy (x);
			return -1;
		}
		ximpacket_rewritelen (x, border);
		SIMPLEQ_INSERT_TAIL (&imc->packetlist, x, entry);

		pewindow_commit (xic->pew);

		break;
	}
	default:
		if (fromclient) type = SEND_SYNC_REPLY;
		break;
	}

#if 1
	switch (type) {
	case SEND_SYNC_REPLY:
	{
		/* send sync-reply packet */
		ret = send_sync (xic, 0);
		if (ret != 0) return -1;
		break;
	}
	case SEND_SYNC:
	{
		/* send sync packet */
		ret = send_sync (xic, 1);
		if (ret != 0) return -1;
		break;
	}
	default:
		break;
	}
#endif

	return 0;
}

static int previous_len = 0;
int
ximp_preedit_start (struct conversion* conv)
{
	int ret;
	struct ximpacket* x;
	int imid, icid;
	struct ximic* xic;
	int border;

	previous_len = 0;
	
	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	xic = conversion_get_xic (conv);
	imid = xic->imid;
	icid = xic->icid;
	border = xic->imc->byte_order;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_PREEDIT_START, 0, 4, border);
	ret |= ximpacket_put16 (x, imid, border);
	ret |= ximpacket_put16 (x, icid, border);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	SIMPLEQ_INSERT_TAIL (&xic->imc->packetlist, x, entry);
	return 0;
}
int
ximp_preedit_stop (struct conversion* conv)
{
	int ret;
	struct ximpacket* x;
	int imid, icid;
	struct ximic* xic;
	int border;

	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	xic = conversion_get_xic (conv);
	imid = xic->imid;
	icid = xic->icid;
	border = xic->imc->byte_order;

	ret = 0;
	ret |= ximpacket_putheader (x, XIM_PREEDIT_DONE, 0, 4, border);
	ret |= ximpacket_put16 (x, imid, border);
	ret |= ximpacket_put16 (x, icid, border);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	SIMPLEQ_INSERT_TAIL (&xic->imc->packetlist, x, entry);
	return 0;
}
int
ximp_preedit_draw (struct conversion* conv, int sendnull)
{
	int ret;
	struct ximpacket* x;
	int imid, icid;
	struct ximic* xic;
	int border;
	char* ctext;
	unsigned short clen;
	int status = 0;
	struct conversion_string* cstr;
	struct conversion_string* c;
	int prevlen = previous_len;
	int stringlen = 0;

	ctext = conversion_get_current_ctext (conv);
	if (ctext == NULL) {
		/* there is no preedit string */
		clen = 0;
		status |= 0x1; /* no string */
	} else {
		clen = strlen (ctext);
	}

	cstr = conversion_get_conversion_string (conv);

	/* compute stringlen */
	c = cstr;
	while (c) {
		char* str = c->buf;
		if (!str) goto NEXT1;
		stringlen += eucjp_len (str);
	NEXT1:
		c = TAILQ_NEXT (c, entry);
	}

	if (sendnull) {
		status |= 0x1; /* no string */
		clen = 0;
		stringlen = 0;
	}
	
	x = ximpacket_allocate ();
	if (x == NULL) return -1;

	xic = conversion_get_xic (conv);
	imid = xic->imid;
	icid = xic->icid;
	border = xic->imc->byte_order;

#if 0
	fprintf (stderr, "%d %d %d %d\n", prevlen, status, clen, stringlen);
#endif
	
	ret = 0;
	ret |= ximpacket_putheader (x, XIM_PREEDIT_DRAW, 0, 0, border);
	ret |= ximpacket_put16 (x, imid, border);
	ret |= ximpacket_put16 (x, icid, border);
	ret |= ximpacket_put32 (x, 0, border);
	ret |= ximpacket_put32 (x, 0, border);
	ret |= ximpacket_put32 (x, prevlen, border);
	ret |= ximpacket_put32 (x, status, border);
	ret |= ximpacket_put16 (x, clen, border);
	if (clen) {
		ret |= ximpacket_putbytes (x, ctext, clen, border);
	}
	ret |= ximpacket_putpad (x, clen + 2);

	/* feedback array len */
	ret |= ximpacket_put16 (x, stringlen * 4, border);
	ret |= ximpacket_put16 (x, 0, border);		/* padding */

	if (stringlen == 0) goto SEND_END;

	/* put attr */
	c = cstr;
	while (c) {
		char* str = c->buf;
		int len;
		int i;
		int attr = 0;
		if (!str || strlen (str) == 0) goto NEXT2;

#if 1
		if (c->stat & CONV_STAT_UNDERLINE) attr |= XIMUnderline;
		if (c->stat & CONV_STAT_REVERSE)   attr |= XIMReverse;
		if (c->stat & CONV_STAT_HILIGHT)   attr |= XIMHighlight;
#endif
		len = eucjp_len (str);
		for (i = 0; i < len; i++) {
			ret |= ximpacket_put32 (x, attr, border);
		}
	NEXT2:
		c = TAILQ_NEXT (c, entry);
	}

 SEND_END:
	conversion_free_current_ctext (conv);
	if (ret != 0) {
		ximpacket_destroy (x);
		return -1;
	}
	ximpacket_rewritelen (x, border);
	SIMPLEQ_INSERT_TAIL (&xic->imc->packetlist, x, entry);

	if (sendnull) send_sync (xic, 0);
	previous_len = stringlen;

	return 0;
}
