/* -*- c++ -*- */
#include "document.h"
#include "document_factory.h"
#include "model_check.h"

#include "../util/string_funcs.h"

namespace aka2 {
  document_factory g_doc_factory_;
}

using namespace aka2;

document::~document() {
}

bool aka2::document_of(const aka2::document &i, const std::string &tagname) {
  return aka2::qname(tagname) == i.get_name();
}

bool operator==(const document &lhs, const document &rhs) {
  if (&lhs.get_op() != &rhs.get_op())
    return false;
  const_node lnd = lhs.get_root();
  const_node rnd = rhs.get_root();
  if (lnd.empty() && rnd.empty())
    return true;
  if (lnd.empty() || rnd.empty())
    return false;

  return lnd.op().equals(lnd.ptr(), rnd.ptr());
}



element_props &document_factory::register_document_type(const qname &docname, 
							const element_op& op){
  element_props props(op);
  props.set_name(docname);
  std::pair<doctype_map::iterator, bool> res = 
    doctype_map_.insert(doctype_map::value_type(docname, props));
  if (!res.second) {
    std::string message = "Document " + quote(docname.qualified()) + " is already registered.";
    throw error(message, __FILE__, __LINE__);
  }
  res.first->second.check_emptiable();
  return res.first->second;
}


void document_factory::register_document_type(const qname &docname, 
					      const element_op& op,
					      const occurrence &occ){
  element_props &props = register_document_type(docname, op);
  if (!check_occurrence(op.get_schematype(), occ)) {
    std::string message;
    message = message + "Document " + quote(docname.qualified()) + " has wrong occurrence.";
    throw error(message, __FILE__, __LINE__);
  }
  props.set_occurrence(occ);
  props.check_emptiable();
}

void document_factory::register_document_type(const qname &docname, 
					      const element_op& op,
					      default_op *defop,
					      const std::string &defval){
  element_props &props = register_document_type(docname, op);
  if (!check_has_default(op.get_schematype())) {
    std::string message = "Document " + quote(docname.qualified())
      + " must be simpleType to have default value.";
    throw error(message, __FILE__, __LINE__);
  }
  props.set_default_op(defop);
  props.setup_default_value(defval);
  props.check_emptiable();
}

bool document_factory::is_document_type_registered(const qname &docname) const {
  doctype_map::const_iterator it = doctype_map_.find(docname);
  return it != doctype_map_.end();
}

document document_factory::create_named_document(const qname &docname){
  doctype_map::iterator it = doctype_map_.find(docname);
  if (it == doctype_map_.end()) {
    std::string errmsg = "Document(" + docname.qualified() + ") not found.";
    throw error(errmsg.c_str(), __FILE__, __LINE__);
  }
  void* root = it->second.op().create();

  return document(root, it->second);
}

const element_props& document_factory::get_props(const qname &docname) const {
  doctype_map::const_iterator it = doctype_map_.find(docname);
  if (it == doctype_map_.end()) {
    std::string message = "Document" + quote(docname.qualified()) + " not found.";
    throw error(message, __FILE__, __LINE__);
  }
  return it->second;
}

