/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_DOCUMENT_H__
#define AKAXISO2_FRAMEWORK_DOCUMENT_H__

/**
 * @file akaxiso2/framework/document.h 
 * @brief document class
 */

#include <akaxiso2/framework/qname.h>
#include <akaxiso2/framework/element_props.h>
#include <akaxiso2/framework/node_ptr.h>

namespace aka2 {

  /** 
   * @brief stores a pair of a document tag name and 
   * a pointer to the root instance.
   *
   * Mainly used by aka2::deserializer derived classes to return deseirlized documents.\n
   * Methods of this class is not intended for use of users.\n
   * To access contents, use aka2::document_of(), aka2::adopt_root<>(), aka2::root_cast<>().\n
   */
  struct document {
    document() : props_(0) {}
    document(void* root, const element_props &def)
      : root_(node(root, def.op())), props_(&def) {}
    ~document();

    node get_root() { return root_.get_node(); }
    node adopt() { return root_.adopt(); }
    const_node get_root() const { return root_.get_node(); }
    const qname &get_name() const { return props_->get_name(); }
    const element_op& get_op() const { return props_->op(); }
    const element_props &get_props() const { return *props_; }
    bool empty() const { return root_.get_node().ptr() == 0; }
  private:
    node_ptr root_;
    const element_props *props_;
  };


  /**
   * @brief Get root node instance as class T.
   *
   * Obtained instance is owned and deleted by document instance.
   * @param doc document instance.
   * @return pointer to root node instance.
   */
  template<class T>
  T* root_cast(aka2::document &doc) {
    return static_cast<T*>(doc.get_root().ptr());
  } 

  /**
   * @brief Const version of function to get root node instance as class T.
   *
   * Obtained instance is owned and deleted by document instance.
   * @param doc document instance.
   * @return pointer to root node instance.
   */
  template<class T>
  const T* root_cast(const aka2::document &doc) {
    return static_cast<const T*>(doc.get_root().ptr());
  }

  /**
   * @brief Adopt root node instance as class T.
   *
   * Obtained instance is owned by user. User is responsible for deleting root instance.
   * @param doc document instance.
   * @return pointer to root node instance.
   */
  template<class T>
  T* adopt_root(aka2::document &doc) {
    return static_cast<T*>(doc.adopt().ptr());
  }

  /**
   * @brief Check document tagname.
   * @param doc document instance returned by deserializer.
   * @param tagname tag name of document.
   * @return true if the tag name of deserialized document is the same as the specified tagname.
   */
  bool document_of(const aka2::document &doc, const std::string &tagname);

} // namespace aka2

bool operator==(const aka2::document &lhs, const aka2::document &rhs);

#endif
