#include "generator.h"
#include <akaxiso2/util/string_funcs.h>
#include <iostream>

using namespace osx;


namespace {

  void write_occurrence(const aka::occurrence &occ, std::ostream &ostm) {
    ostm << occ.minOccurs_ << ", ";
    if (occ.maxOccurs_ == aka::unbounded)
      ostm << "aka::unbounded";
    else
      ostm << occ.maxOccurs_;
  }

  std::string tagname(const element_type &et) {
    if (et.is_group_)
      return aka::quote("&" + et.name_.local());
    return "\"" + et.name_.qualified() + "\"";
  }
  std::string tagname(const attribute_type &et) {
    return aka::quote(et.name_.qualified());
  }

  std::string quote(const std::string &value) {
    std::string ret;
    for (std::string::size_type index = 0; index < value.size(); ++index) {
      if (value[index] == '\"') //!!!!!!!!!!!!!!!!!!
	ret += "\\\"";
      else if (value[index] == '\\')
	ret += "\\\\";
      else
	ret += value[index];
    }
    return aka::quote(ret);
  }

  std::string quote(const aka::qname &value) {
    return aka::quote(value.qualified());
  }

}

void output_file::open(const std::string &filename) {
  std::string fullpath = outputdir_ + filename;
  ostm_.open(fullpath.c_str());
  if (!ostm_.is_open())
    throw fatal_error("Failed to open file, " + aka::quote(filename) + ".",
		      __FILE__, __LINE__);
  filename_ = filename;
}

void output_file::close() {
  ostm_.close();
}

void output_file::write_include(const std::string &include_name) {
  ostm_ << "#include " << aka::quote(include_name) << std::endl;
}

void output_file::write_system_include(const std::string &include_name) {
  ostm_ << "#include " << aka::tag(include_name) << std::endl;
}

void output_file::write_header() {

  std::string escaped = escape_filename(filename_);

  ostm_ << "/* -*- c++ -*- */" << std::endl
	<< "#ifndef " << escaped << "__" << std::endl
	<< "#define " << escaped << "__" << std::endl
	<< std::endl;
}


void output_file::write_footer() {

  std::string escaped = escape_filename(filename_);
  ostm_ << "#endif // #ifndef " << escaped << "__" << std::endl;
}


std::string output_file::escape_filename(const std::string &fn) {
  std::string escaped;
  for (std::string::size_type pos = 0; pos < fn.size(); ++pos) {
    std::string::value_type ch = fn[pos];
    if (ch == '.')
      ch = '_';
    else if (ch == '-')
      ch = '_';
    escaped += ch;
  }
  return escaped;
}

void output_file::newline() {
  ostm_ << std::endl;
}





void generator_base::set_unit(const unit_props &unit) {
  regunit_ = &registry_[unit];
  target_ns_id_ = unit.target_ns_id_;
}


std::string generator_base::cppname(const aka::qname &name) const {

  aka::qname tyname = registry_.rename_type(name);
  std::string localname = registry_.escape(tyname.local());
  std::string to_escape;
  if (tyname.is_qualified())
    to_escape = std::string("::") + tyname.prefix() + "::" + localname;
  else if (registry_.is_builtin(name))
    to_escape = localname;
  else
    to_escape = std::string("::") + localname;
  return registry_.escape(to_escape);
}

std::string generator_base::get_prefix(aka::id_type nsid) const {
  const std::string &prefix = aka::get_prefix(nsid);
  std::string escaped_prefix = registry_.escape(prefix);
  return escaped_prefix;
}



std::string generator_base::cppname(const std::string &name) const {
  return cppname(aka::qname(name));
}

std::string generator_base::declname(const aka::qname &name) const {
  aka::qname tyname = registry_.rename_type(name);
  return registry_.escape(tyname.local());
}

std::string generator_base::declname(const std::string &name) const {
  aka::qname qn(name);
  return declname(qn);
}

std::string generator_base::leafdecl(const aka::qname &name) const {
  aka::qname tyname = registry_.rename_type(name);
  return registry_.escape(tyname.local()) + "_leaf";
}

std::string generator_base::leafname(const aka::qname &type) const {
  if (registry_.is_predefined(type)) {
    return registry_.get_predefined_leafname(type);
  }
  else {
    aka::qname tyname = registry_.rename_type(type);
    std::string to_escape;
    if (tyname.is_qualified())
      to_escape = "::" + tyname.prefix() + "::" + tyname.local() + "_leaf";
    else
      to_escape = tyname.local() + "_leaf";
    return registry_.escape(to_escape);
  }
}

std::string generator_base::membername(const aka::qname &name) const {
  if ((target_ns_id_ == name.get_namespace_id()) || (name.prefix().empty()))
    return membername(name.local());
  else {
    std::string qualified = name.prefix() + "_" + name.local();
    return membername(qualified);
  }
}

std::string generator_base::membername(const std::string &name) const {
  return registry_.get_member_prefix() 
    + registry_.escape(name) 
    + registry_.get_member_suffix();
}



void generator::begin_namespace() {
  assert(target_ns_id_ != aka::empty_token);

  elm_.ostm_ << std::endl
	     << "namespace " << get_prefix(target_ns_id_) << " {" << std::endl
	     << std::endl;
  xiso_.ostm_ << std::endl
	      << "namespace " << get_prefix(target_ns_id_) << " {" << std::endl
	      << std::endl;
  ximpl_.ostm_ << std::endl
	       << "namespace " << get_prefix(target_ns_id_) << " {" << std::endl
	       << std::endl;
}    

void generator::end_namespace() {
  assert(target_ns_id_ != aka::empty_token);
  elm_.ostm_ << std::endl
	     << "}" << std::endl
	     << std::endl;
  xiso_.ostm_ << std::endl
	      << "}" << std::endl
	      << std::endl;
  ximpl_.ostm_ << std::endl
	       << "}" << std::endl
	       << std::endl;
}




void generator::begin_model_method(const aka::qname &name) const {
  ximpl_.ostm_ << "  void " << leafdecl(name) << "::model() {" << std::endl;
}

void generator::end_model_method() const {
  ximpl_.ostm_ << "  }" << std::endl
	       << std::endl;
}



std::string generator::get_ns_list(const std::string &ns_list) const {
  if (aka::trim(ns_list) == "##other")
    return quote("##other:" + aka::get_namespace_uri(target_ns_id_));
  return quote(ns_list);
}




bool generator::is_leaf_declared(const aka::qname &name) const {
  if (registry_.is_predefined(name))
    return true;
  return declared_leafs_.find(name) != declared_leafs_.end();
}

void generator::leaf_declared(const aka::qname &name) {
  if (is_leaf_declared(name))
    return;
  declared_leafs_.insert(name);

}


void generator::prepare() {
  registry_.get_predefinedTypes(resolved_);
  declared_leafs_.clear();
}


void generator::resolve(const unit_props &unit, bool generate) {

  set_unit(unit);

  try {

    qname_set ns_toplevels;
    qname_set ns_simpleTypes;

    for (class_defs::const_iterator cit = regunit_->classes_.begin(); 
	 cit != regunit_->classes_.end(); ++cit) {
      const class_def *def = cit->second;
      if (registry_.is_predefined(def->get_name()))
	continue;
      if (def->get_name().get_namespace_id() != target_ns_id_)
	continue;
      if (def->id_ == simpletype_id)
	ns_simpleTypes.insert(def->get_name());
    }

    elm_.newline();
    xiso_.newline();
    ximpl_.newline();

    if (generate && (target_ns_id_ != aka::empty_token)) {
      begin_namespace();
    }    

    for (qname_set::const_iterator nit = regunit_->toplevels_.begin(); 
	 nit != regunit_->toplevels_.end(); ++nit) {
      if (registry_.is_predefined(*nit))
	continue;
      if (nit->get_namespace_id() != target_ns_id_)
	continue;
      const class_def &def = registry_.classdef_get(*nit);
      if (def.id_ != simpletype_id)
	ns_toplevels.insert(*nit);
    }
    resolve_one_unit(ns_simpleTypes, ns_toplevels, generate);

    if (generate && (target_ns_id_ != aka::empty_token)) {
      end_namespace();
    }
  }
  catch ( ... ) {
    elm_.ostm_.close();
    xiso_.ostm_.close();
    ximpl_.ostm_.close();
    throw;
  }
}



void generator::write_leaf_decl(const class_def &def, const std::string &leaftmp) const {
  xiso_.ostm_ << "  struct " << leafdecl(def.get_name()) 
	      << " : aka::" << leaftmp << "< " << cppname(def.get_name())
	      << ", " << leafdecl(def.get_name()) << "> {" << std::endl
	      << "    void model();" << std::endl
	      << "  };" << std::endl
	      << std::endl;
}


void generator::write_member_decls(const element_types &etypes) const {

  if (!etypes.empty())
    elm_.ostm_ << "    /** members */" << std::endl;
  for (element_types::const_iterator eit = etypes.begin();
       eit != etypes.end(); ++eit) {
    if (eit->is_ptrmember_)
      elm_.ostm_ << "    aka::deep_ptr< " << cppname(eit->type_) << "> " << membername(eit->member_name_) 
		 << ";" << std::endl;
    else if (!eit->is_fixed_)
      elm_.ostm_ << "    " << cppname(eit->type_) << " " << membername(eit->member_name_) << ";" << std::endl;
  }
}

void generator::write_attribute_decls(const attribute_defs &attrs) const {

  if (attrs.has_attributes())
    elm_.ostm_ << "    /** attributes */" << std::endl;
  for (attribute_types::const_iterator ait = attrs.attributes_.begin();
       ait != attrs.attributes_.end(); ++ait) {
    if (!ait->is_fixed_)
      elm_.ostm_ << "    " << cppname(ait->type_) << " " << membername(ait->name_) << ";" << std::endl;
  }
  if (attrs.has_anyAttribute_) {
    elm_.ostm_ << "    aka::wc_attributes " << membername("attributes") << ";" << std::endl;
  }
}

void generator::write_member_leaves(const aka::qname &name, const element_types &etypes) const {

  if (etypes.empty())
    return;

  ximpl_.ostm_ << "    /** element member definitions */" << std::endl;

  for (element_types::const_iterator eit = etypes.begin();
       eit != etypes.end(); ++eit) {
    if (eit->is_fixed_) {
      ximpl_.ostm_ << "    fixed_member(" << tagname(*eit) << ", " 
		   << quote(eit->default_) << ", " << leafname(eit->type_) << ");";
      continue;
    }

    if (eit->type_ == aka::qname("aka:any")) {
      if (eit->is_ptrmember_)
	ximpl_.ostm_ << "    any_ptrmember(" << tagname(*eit) << ", &"
		     << cppname(name) << "::" << membername(eit->member_name_) << ", "
		     << get_ns_list(eit->namespace_list_) << ");" << std::endl;
      else
	ximpl_.ostm_ << "    any(" << tagname(*eit) << ", &"
		     << cppname(name) << "::" << membername(eit->member_name_) << ", "
		     << get_ns_list(eit->namespace_list_) << ");" << std::endl;
      continue;
    }
    if (eit->type_ == aka::qname("aka:any_array")) {
      ximpl_.ostm_ << "    any(" << tagname(*eit) << ", &"
		   << cppname(name) << "::" << membername(eit->member_name_) << ", ";
      write_occurrence(eit->occ_, ximpl_.ostm_);
      ximpl_.ostm_ << ", " << get_ns_list(eit->namespace_list_)
		   << ");" << std::endl;
      continue;
    }

    if (eit->is_ptrmember_) {
      ximpl_.ostm_ << "    ptrmember(" << tagname(*eit) << ", &"
		   << cppname(name) << "::" << membername(eit->member_name_) << ", "
		   << leafname(eit->type_)
		   << "());" << std::endl;
      continue;
    }

    ximpl_.ostm_ << "    member(" << tagname(*eit) << ", &"
		 << cppname(name) << "::" << membername(eit->member_name_) << ", "
		 << leafname(eit->type_) << "()";
    if (eit->occ_required_) {
      ximpl_.ostm_ << ", ";
      write_occurrence(eit->occ_, ximpl_.ostm_);
    }
    ximpl_.ostm_ << ");" << std::endl;
  }
}


void generator::write_item_leaves(const aka::qname &name, const element_types &etypes) const {

  if (etypes.empty())
    return;

  ximpl_.ostm_ << "    /** element member definitions */" << std::endl;

  for (element_types::const_iterator eit = etypes.begin();
       eit != etypes.end(); ++eit) {
    if (eit->is_fixed_) {
      ximpl_.ostm_ << "    fixed_item(" << tagname(*eit) << ", " 
		   << quote(eit->default_) << ", " << leafname(eit->type_) << ");";
      continue;
    }

    if (registry_.is_aka_any(eit->type_)) {
      ximpl_.ostm_ << "    any(" << tagname(*eit);
      if (eit->occ_required_) {
	ximpl_.ostm_ << ", ";
	write_occurrence(eit->occ_, ximpl_.ostm_);
      }
      ximpl_.ostm_ << ", " << get_ns_list(eit->namespace_list_)
		   << ");" << std::endl;
      continue;
    }

    if (registry_.is_predefined(eit->type_)) {
      ximpl_.ostm_ << "    item(" << tagname(*eit) << ", "
		   << leafname(eit->type_) << "());" << std::endl;
      continue;
    }

    ximpl_.ostm_ << "    item(" << tagname(*eit) << ", "
		 << leafname(eit->type_) << "()";
    if (eit->occ_required_) {
      ximpl_.ostm_ << ", ";
      write_occurrence(eit->occ_, ximpl_.ostm_);
    }
    ximpl_.ostm_ << ");" << std::endl;
  }
}

void generator::write_attribute_leaves(const aka::qname &name, 
				       const attribute_defs &attrs) const {

  if (!attrs.has_attributes())
    return;
  
  ximpl_.ostm_ << "    /** attribute member definition. */" << std::endl;

  for (attribute_types::const_iterator ait = attrs.attributes_.begin();
       ait != attrs.attributes_.end(); ++ait) {
    if (ait->is_fixed_) {
      ximpl_.ostm_ << "    fixed_attribute< " << leafname(ait->type_) << " >(" << tagname(*ait) << ", " 
		   << quote(ait->default_) << ", "
		   << leafname(ait->type_) << "())";
      if (ait->required_)
	ximpl_.ostm_ << ".required(true)";
      ximpl_.ostm_ << ";";
      continue;
    }
    ximpl_.ostm_ << "    attribute(" << tagname(*ait) << ", &" 
		 << cppname(name) << "::" << membername(ait->name_) << ", "
		 << leafname(ait->type_) << "())";
    if (!ait->default_.empty()) {
      ximpl_.ostm_ << ".set_default(" << quote(ait->default_) << ")";
      assert(!ait->required_);
    }
    if (ait->required_)
      ximpl_.ostm_ << ".required(true)";

    ximpl_.ostm_ << ";" << std::endl;
  }
  if (attrs.has_anyAttribute_)
    ximpl_.ostm_ << "    any_attribute(&" << cppname(name) << "::attributes_, "
		 << get_ns_list(attrs.namespace_list_) << ");" << std::endl;
}


void generator::generate_sequence(const class_def &def) {
  write_ptrmember_decl(def.etypes_);
  elm_.ostm_ << "  struct " << declname(def.get_name()) << " {" << std::endl;
  write_member_decls(def.etypes_);
  write_attribute_decls(def.adefs_);
  elm_.ostm_ << "  };" << std::endl
	     << std::endl;

  write_leaf_decl(def, "sequence");
  leaf_declared(def.get_name());

  begin_model_method(def.get_name());
  write_member_leaves(def.get_name(), def.etypes_);
  write_attribute_leaves(def.get_name(), def.adefs_);
  end_model_method();
}


void generator::generate_all(const class_def &def) {
  write_ptrmember_decl(def.etypes_);
  elm_.ostm_ << "  struct " << declname(def.get_name()) << " {" << std::endl;
  write_member_decls(def.etypes_);
  write_attribute_decls(def.adefs_);
  elm_.ostm_ << "  };" << std::endl
	     << std::endl;

  write_leaf_decl(def, "all");
  leaf_declared(def.get_name());

  begin_model_method(def.get_name());
  write_member_leaves(def.get_name(), def.etypes_);
  write_attribute_leaves(def.get_name(), def.adefs_);
  end_model_method();
}


void generator::generate_choice(const class_def &def) {
  // element.h
  std::ostringstream forwarded;

  elm_.ostm_ << "  typedef " << registry_.choice_container_type() << "<aka::item> " 
	     << declname(def.get_name()) << ';' << std::endl
	     << std::endl;

  write_leaf_decl(def, "sequential_choice");
  leaf_declared(def.get_name());

  begin_model_method(def.get_name());
  write_item_leaves(def.get_name(), def.etypes_);
  end_model_method();
}


void generator::generate_simplecontent(const class_def &def) {

  elm_.ostm_ << "  struct " << declname(def.get_name()) << " {" << std::endl;
  elm_.ostm_ << "    " << cppname(def.get_value_type()) << " " 
	     << membername(registry_.simplecontent_value_name()) << ";" << std::endl; 
  write_attribute_decls(def.adefs_);
  elm_.ostm_ << "  };" << std::endl
	     << std::endl;

  write_leaf_decl(def, "simplecontent");
  leaf_declared(def.get_name());

  begin_model_method(def.get_name());

  ximpl_.ostm_ << "    value(&" << cppname(def.get_name()) << "::" 
	       << membername(registry_.simplecontent_value_name()) << ");" << std::endl;
  // write default !!!!!

  write_attribute_leaves(def.get_name(), def.adefs_);
  end_model_method();

}



void generator::generate_array(const class_def &def) {
  if (resolved_.find(def.get_value_type()) == resolved_.end()) {
    elm_.ostm_ << "  /** forwarded declartion */" << std::endl
	       << "  struct " << declname(def.get_value_type()) << ";" << std::endl;
  }
  elm_.ostm_ << "  typedef " 
	     << registry_.array_container_type() << "< " << cppname(def.get_value_type()) << "> " 
	     << declname(def.get_name()) << ";" << std::endl
	     << std::endl;

  // Forwareded declaration of leaf type of array value type.
  if (!is_leaf_declared(def.get_value_type()))
    xiso_.ostm_ << "  struct " << leafdecl(def.get_value_type()) << ";" << std::endl;

  xiso_.ostm_ << "  typedef aka::sequential_array< " << cppname(def.get_name())
	      << ", " << leafname(def.get_value_type()) << "> " 
	      << leafdecl(def.get_name()) << ";" << std::endl
	      << std::endl;

  leaf_declared(def.get_name());
}

void generator::generate_simpletype(const class_def &def) {
  elm_.ostm_ << "  typedef " << cppname(def.get_value_type()) << " " 
	     << declname(def.get_name()) << ";" << std::endl
	     << std::endl;

  if (!registry_.is_predefined(def.get_name())) {
    xiso_.ostm_ << "  typedef xiso::leaf< " << cppname(def.get_value_type()) << "> " 
		<< leafdecl(def.get_name()) << ";" << std::endl; 
    leaf_declared(def.get_name());
  }
}


void generator::write_ptrmember_decl(const element_types &etypes) const {
  for (element_types::const_iterator it = etypes.begin(); it != etypes.end(); ++it) {
    if (!it->is_ptrmember_)
      continue;
    if (resolved_.find(it->type_) == resolved_.end())
      elm_.ostm_ << "  /** forwareded declaration for ptrmember value class. */" << std::endl
		 << "  struct " << declname(it->type_) << ";" << std::endl;
  }
}



void generator::resolve_one_unit(qname_set &simpleTypes, qname_set &toplevels, bool generate) {

  if (!simpleTypes.empty() && generate)
    elm_.ostm_ << "  /** simpleType definitions */" << std::endl
	       << std::endl;
  
  while (!simpleTypes.empty()) {
    qname_set::iterator it = simpleTypes.begin();
    for (; it != simpleTypes.end(); ++it) {
      if (resolved_.find(*it) != resolved_.end()) {
	// already resolved.  Nothing to generate.
	simpleTypes.erase(it);
	break;
      }
      else if (resolve_simpleType(*it, generate)) {
	resolved_.insert(*it);
	simpleTypes.erase(it);
	break;
      }
    }
    if (it == simpleTypes.end()) {
      std::ostringstream ostm;
      ostm << "Unresolved simpleTypes: ";
      for (it = simpleTypes.begin(); it != simpleTypes.end(); ++it)
	ostm << it->qualified() << " ";
      throw fatal_error(ostm.str(), __FILE__, __LINE__);
    }
  }
  
  if (!toplevels.empty() && generate)
    elm_.ostm_ << "  /** complexType definitions */" << std::endl
	       << std::endl;

  while (!toplevels.empty()) {
    qname_set::iterator it = toplevels.begin();
    for (; it != toplevels.end(); ++it) {
      if (resolved_.find(*it) != resolved_.end()) {
	toplevels.erase(it); // already resolved.
	break;
      }
      //       std::cerr << "Resolving " << *it << "...";

      // Force hard resolve only.
      if (resolve_complexType(*it, toplevels, true, generate)) {
	// 	std::cerr << "done." << std::endl;
	toplevels.erase(it);
	break;
      }
    }

    if (it == toplevels.end()) {
      for (it = toplevels.begin(); it != toplevels.end(); ++it) {
	if (resolve_complexType(*it, toplevels, false, generate)) {
	  // 	std::cerr << "done." << std::endl;
	  toplevels.erase(it);
	  break;
	}
      }
    }

    if (it == toplevels.end()) {
      throw fatal_error("Could not resolve dependencies.", __FILE__, __LINE__);
    }
  }
}

bool generator::resolve_simpleType(const aka::qname &to_resolve, 
				   bool generate_source) {

  class_def *def = regunit_->classes_.get(to_resolve);
  if (resolved_.find(def->get_value_type()) == resolved_.end())
    return false;

  assert(def->id_ == simpletype_id);

  if (generate_source)
    generate_simpletype(*def);
  return true;
}


bool generator::resolve_complexType(const aka::qname &to_resolve, 
				    qname_set &toplevels, 
				    bool force_hard_resolve,
				    bool generate_source) {

  qname_set resolving(resolved_);

  class_def_array deps;
  qname_set forwarded;

  if (force_hard_resolve) {
    if (!hard_resolve(resolving, to_resolve, deps, forwarded))
      return false;
  }
  else {
    if (!soft_resolve(resolving, to_resolve, deps, forwarded))
      return false;
  }

  class_def &def = *regunit_->classes_.get(to_resolve);
  deps.push_back(&def);

  for (class_def_array::iterator it = deps.begin(); it != deps.end(); ++it) {
    class_def *def = *it;
    if (resolved_.find(def->get_name()) != resolved_.end())
      continue;

    //     logstm_ << def->get_name() << " resolved." << std::endl;

    if (generate_source && (def->get_name().get_namespace_id() == target_ns_id_)) {
      switch (def->id_) {
      case sequence_id:
	generate_sequence(*def);
	break;
      case choice_id:
	generate_choice(*def);
	break;
      case all_id:
	generate_all(*def);
	break;
      case array_id:
	generate_array(*def);
	break;
      case simplecontent_id:
	generate_simplecontent(*def);
	break;
      case simpletype_id:
      default:
	assert(!"Must not reach here.");
	break;
      }
    }
    resolved_.insert(def->get_name());
  }
  
  // generation of forwarded classes. 
  for (qname_set::iterator fit = forwarded.begin(); fit != forwarded.end(); ++fit) {
    if (resolved_.find(*fit) == resolved_.end())
      toplevels.insert(*fit);
  }
  return true;
}


bool generator::soft_resolve(qname_set &resolving, const aka::qname &name, 
			     class_def_array &deps, qname_set &forwarded) {

  if (resolving.find(name) != resolving.end()) // already resolved.
    return true;

  class_def_array resolved_classes;
  qname_set resolved_here(resolving);
  class_def *def = &registry_.classdef_get(name);
  
  qname_set processing;
  if (hard_resolve(resolved_here, name, resolved_classes, processing)) {
    deps.insert(deps.end(), resolved_classes.begin(), resolved_classes.end());
    deps.push_back(def);
    resolving = resolved_here;
    resolving.insert(name);
    return true;
  }
  
  if (def->id_ == choice_id) {
    deps.push_back(def);
    for (element_types::const_iterator it = def->etypes_.begin(); 
	 it != def->etypes_.end(); ++it)
      forwarded.insert(it->type_);
    return true;
  }

  if (def->id_ == array_id) {
    if (!use_soft_array_) {
      return hard_resolve(resolving, def->get_value_type(),
			  resolved_classes, processing);
    }
    else {
      deps.push_back(def);
      forwarded.insert(def->get_value_type());
      return true;
    }
  }

  qname_set forwarded_here;
  
  for (element_types::const_iterator eit = def->etypes_.begin();
       eit != def->etypes_.end(); ++eit) {
    const element_type& childtype = *eit;
    if (def->get_name() == childtype.type_) {
      if (childtype.occ_required_ || childtype.is_ptrmember_) // consider is_model()!!!!!
	continue;
    }
    else if (eit->is_ptrmember_) {
      if (resolved_.find(eit->type_) != resolved_.end())
	continue;
      forwarded.insert(eit->type_);
      continue;
    }
    else if (soft_resolve(resolved_here, childtype.type_, 
			  resolved_classes, forwarded_here))
      continue;
    return false;
  }
  
  for (attribute_types::const_iterator ait = def->adefs_.attributes_.begin();
       ait != def->adefs_.attributes_.end(); ++ait) {
    if (hard_resolve(resolving, ait->type_, resolved_classes, processing))
      continue;
    return false;
  }
  
  resolving = resolved_here;
  deps.insert(deps.end(), resolved_classes.begin(), resolved_classes.end());
  for (qname_set::iterator it = forwarded_here.begin(); it != forwarded_here.end(); ++it)
    forwarded.insert(*it);
  
  deps.push_back(def);
  resolving.insert(name);
  
  return true;
}


bool generator::hard_resolve(qname_set &resolving, const aka::qname &name, 
			     class_def_array &deps, qname_set &processing) {
  
  if (resolving.find(name) != resolving.end())
    return true;
  
  if (processing.find(name) != processing.end())
    return false;
  
  processing.insert(name);
  
  class_def_array resolved_classes;
  qname_set resolved_here(resolving);
  const class_def *def = &registry_.classdef_get(name);
 
  if (def->id_ == array_id) {
    if (resolved_here.find(def->get_value_type()) == resolved_here.end())
      if (!hard_resolve(resolved_here, def->get_value_type(), deps, processing))
	return false;
  }
  else {
    for (element_types::const_iterator eit = def->etypes_.begin();
	 eit != def->etypes_.end(); ++eit) {
      if (resolved_here.find(eit->type_) == resolved_here.end())
	if (!hard_resolve(resolved_here, eit->type_, deps, processing))
	  return false;
    }
  }

  for (attribute_types::const_iterator ait = def->adefs_.attributes_.begin();
       ait != def->adefs_.attributes_.end(); ++ait) {
    if ((resolved_here.find(ait->type_) == resolved_here.end()) &&
	!hard_resolve(resolved_here, ait->type_, deps, processing))
      return false;
  }

  resolving = resolved_here;
  resolving.insert(name);

  deps.insert(deps.end(), resolved_classes.begin(), resolved_classes.end());
  class_def *child = &registry_.classdef_get(name);
  deps.push_back(child);
  return true;
}







void generator::open(const std::string &basename) {
  elm_.open(basename + ".h");
  xiso_.open(basename + "_xiso.h");
  ximpl_.open(basename + "_xiso.cpp");

  elm_.write_header();
  xiso_.write_header();

  elm_.write_system_include("akaxiso2/builtin/builtin.h");
  elm_.write_system_include("akaxiso2/builtin/schema_builtin.h");
  elm_.newline();

  xiso_.write_include(elm_.get_filename());
  xiso_.write_system_include("akaxiso2/content_model.h");
  xiso_.newline();

  ximpl_.write_include(xiso_.get_filename());
  ximpl_.write_system_include("akaxiso2/akaxiso2.h");
  ximpl_.newline();

}


void generator::close() {
  elm_.write_footer();
  xiso_.write_footer();

  elm_.close();
  xiso_.close();
  ximpl_.close();
}


void generator::write_include(const std::string &basename) const {
  elm_.write_include(basename + ".h");
  xiso_.write_include(basename + "_xiso.h");
}


namespace {
  struct doc_tag {
    doc_tag(const aka::qname &name,
	    const aka::qname &type)
      : name_(name), 
	type_(type){}
    aka::qname name_;
    aka::qname type_;
  };
}

void func_generator::generate_serialize_methods() {

  typedef std::vector<doc_tag> doc_tags;
  typedef std::map<aka::qname, doc_tags, aka::qname_less> element_map;
  element_map emap;

  for (unit_array::iterator uit = registry_.units_.begin();
       uit != registry_.units_.end(); ++uit) {

    set_unit(*uit);

    for (element_defs::iterator eit = regunit_->elements_.begin();
	 eit != regunit_->elements_.end(); ++eit) {
      aka::qname realname = eit->second.type_;

      while (true) {
	if (registry_.is_builtin(realname))
	  break;
	
	if (registry_.is_predefined(realname)) {
	  aka::qname saved = realname;
	  realname = registry_.resolve_predefinedType(realname);
	  if (realname != saved)
	    continue;
	  else
	    break;
	}

	assert(registry_.classdef_exists(realname));
	class_def &def = registry_.classdef_get(realname);
	if (def.id_ == simpletype_id) {
	  realname = def.get_value_type();
	  continue;
	}
	
	assert(def.id_ != simpletype_id);
	break;
      }
      doc_tag dt(eit->first, eit->second.type_);
      emap[realname].push_back(dt);
    }
  }    
  
  for (element_map::iterator eit = emap.begin(); eit != emap.end(); ++eit) {
    const aka::qname &type = eit->first;
    const doc_tags &tags = eit->second;

    if (tags.size() == 1) {
      const doc_tag &dtag = tags.front();
      header_.ostm_ << "/** serialization of " << quote(dtag.name_) << ". */" << std::endl
		    << "void serialize(const " << cppname(eit->first) 
		    << " &root, std::ostream &ostm);" << std::endl;
      
      impl_.ostm_ << "void serialize(const " << cppname(type) 
		  << "& root, std::ostream &ostm) {" << std::endl
		  << "  aka::xml_serializer ser;" << std::endl
		  << "  ser.serialize(root, " << quote(dtag.name_) << ", ostm);" << std::endl
		  << "}" << std::endl
		  << std::endl;
    }
    else {

      header_.ostm_ << "/** serialization of ";
      
      doc_tags::const_iterator tit = tags.begin();
      header_.ostm_ << quote(tit->name_) << "(" << cppname(tit->type_) << ")";
      for (++tit; tit != tags.end(); ++tit) {
	header_.ostm_ << ", " << quote(tit->name_) << "(" << cppname(tit->type_) << ")";
      }
      header_.ostm_ << ". */" << std::endl
		    << "void serialize(const " << cppname(type)
		    << "& root, const std::string &tagname, std::ostream &ostm);" << std::endl;
      
      impl_.ostm_ << "void serialize(const " << cppname(type) 
		  << "& root, const std::string &tagname, std::ostream &ostm) {" << std::endl
		  << "  aka::xml_serializer ser;" << std::endl
		  << "  ser.serialize(root, tagname, ostm);" << std::endl
		  << "}" << std::endl
		  << std::endl;
    }
  }
}


void func_generator::begin_instantiate_xiso() const {

  header_.ostm_ << "void instantiate_xiso();" << std::endl
		<< std::endl;
  impl_.ostm_ << "void instantiate_xiso() {" << std::endl;
}


void func_generator::generate_instantiate_xiso() {

  for (unit_array::const_iterator uit = registry_.units_.begin();
       uit != registry_.units_.end(); ++uit) {

    set_unit(*uit);
    if ((target_ns_id_ != aka::empty_token) && (target_ns_id_ != aka::xml_ns_token))
      impl_.ostm_ << "  aka::xmlns(" << quote(aka::get_prefix(target_ns_id_))
		  << ", " << quote(aka::get_namespace_uri(target_ns_id_)) << ");" << std::endl;
    
    for (element_defs::iterator nit = regunit_->elements_.begin();
	 nit != regunit_->elements_.end(); ++nit) {
      const aka::qname &name = nit->first;
      if (name.get_namespace_id() == target_ns_id_) {
	if (nit->second.type_ == aka::qname("aka:wildcard")) {
	  impl_.ostm_ << "  aka::doctype(" << quote(nit->first) << ");" << std::endl;
	}
	else if (nit->second.is_fixed_) {
	  impl_.ostm_ << "  aka::doctype(" << quote(nit->first) << ", " 
		      << leafname(nit->second.type_) << ", "
		      << quote(nit->second.default_)
		      << ");" << std::endl;
	}
	else {
	  impl_.ostm_ << "  aka::doctype(" << quote(nit->first) << ", "
		      << leafname(nit->second.type_) << "());" << std::endl;
	}
      }
    }
  }
}

void func_generator::write_includes() {
  for (unit_array::const_iterator uit = registry_.units_.begin();
       uit != registry_.units_.end(); ++uit) {
    header_.write_include(uit->basename_ + ".h");
    impl_.write_include(uit->basename_ + "_xiso.h");
  }
}


void func_generator::end_instantiate_xiso() const {
  impl_.ostm_ << "}" << std::endl
	      << std::endl;
}

void func_generator::generate() {
  begin_instantiate_xiso();
  generate_instantiate_xiso();
  end_instantiate_xiso();
  generate_serialize_methods();
}

void func_generator::open(const std::string &basename) {

  header_.open(basename + ".h");
  header_.write_header();
  header_.newline();

  impl_.open(basename + ".cpp");
  impl_.write_include(header_.get_filename());
  impl_.write_system_include("akaxiso2/akaxiso2.h");
  impl_.newline();
}


void func_generator::close() {
  header_.newline();
  header_.write_footer();
  header_.close();

  impl_.newline();
  impl_.close();
}
