#include "osixaka2.h"
#include "preprocessor.h"
#include "processor.h"
#include "dtd_processor.h"
#include "util.h"
#include "generator.h"
#include "akaxisonizer.h"
#include "dot_generator.h"

#include "serializer.h"
#include <fstream>

using namespace osx;

void osixaka2::initialize() {
  aka::xmlns("aka", "http://akaxiso.sourceforge.jp/akaxiso2");
  aka::xmlns("std", "http://akaxiso.sourceforge.jp/osixaka2/c++-std");
  aka::xmlns("osx", "http://akaxiso.sourceforge.jp/osixaka2");
  instantiate_xiso();
}

int osixaka2::preprocess(const std::string &filename, bool generate_deps) {
  try {
    preprocessor pre(registry_, generate_deps, ostm_, verbose_);
    pre.preprocess(filename);
    return 0;
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    ostm_ << "Failed to generate files. Aborting..." << std::endl;
    return 1;
  }
  catch ( ... ) {
    ostm_ << "Unknown exception caught. Aborting..." << std::endl;
    return 1;
  }
}  

int osixaka2::process() {
  try {
    registry_.insert_simpleTypes();
    processor prc(registry_, ostm_, verbose_);

    unit_array::iterator it;
    for (it = registry_.units_.begin();
	 it != registry_.units_.end(); ++it)
      prc.scan(*it);
    for (it = registry_.units_.begin();
	 it != registry_.units_.end(); ++it)
      prc.process(*it);

    return 0;
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    ostm_ << "Failed to process Schema. Aborting..." << std::endl;
    return 1;
  }
  catch ( ... ) {
    ostm_ << "Unknown exception caught. Aborting..." << std::endl;
    return 1;
  }
}  

void osixaka2::load_defaults() {
  registry_.load_defaults();
  registry_.register_ns();
}

void osixaka2::load_osixaka1_defaults() {
  registry_.load_osixaka1_defaults();
  registry_.register_ns();
}

void osixaka2::load_preference(const std::string &filename) {
  registry_.load_preference(filename);
  registry_.register_ns();
}

void osixaka2::save_preference(std::ostream &file) {
  registry_.save_preference(file);
}

#ifdef AKAXISO2_USE_EXPAT
int osixaka2::process_dtd(const std::string &filepath) {
  dtd_processor processor(registry_, ostm_, verbose_, true);
  return processor.process(filepath);
}
#else

int osixaka2::process_dtd(const std::string &filepath) {
  ostm_ << "To use DTD processor, enable expat." << std::endl;
  return 1;
}
#endif


void osixaka2::generate_all_in_one(const std::string &outputdir) {

  output_file elm(outputdir);
  output_file xiso(outputdir);
  output_file ximpl(outputdir);

  const unit_props &unit = registry_.units_.back();
  
  generator gen(registry_, elm, xiso, ximpl, use_soft_array_);
  gen.open(unit.basename_);

  gen.prepare();
  
  for (unit_array::iterator it = registry_.units_.begin(); 
       it != registry_.units_.end(); ++it) {
    gen.resolve(*it, true);
  }

  func_generator fgen(registry_, elm, ximpl);
  fgen.generate(); 

  gen.close();
}


void osixaka2::generate_target_ns(const std::string &outputdir) {

  output_file elm(outputdir);
  output_file xiso(outputdir);
  output_file ximpl(outputdir);

  const unit_props &unit = registry_.units_.back();
  
  generator gen(registry_, elm, xiso, ximpl, use_soft_array_);
  gen.open(unit.basename_);

  gen.prepare();
  
  for (unit_array::iterator it = registry_.units_.begin(); 
       unit.target_ns_id_ != it->target_ns_id_; ++it) {
    gen.write_include(it->basename_);
    gen.resolve(*it, false);
  }

  gen.resolve(unit, true);

  gen.close();
}



int osixaka2::generate(const std::string &outputdir, bool one_ns) {

  try {
    if (one_ns) {
      generate_target_ns(outputdir);
    }
    else {
      generate_all_in_one(outputdir);
    }
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Code generation failed." << std::endl;
    return 1;
  }
  return 0;
}


int osixaka2::generate_functions(const std::string &outputdir, const std::string &filename) {
  
  try {
    output_file header(outputdir), impl(outputdir);
    func_generator fgen(registry_, header, impl);
    fgen.open(filename);
    fgen.write_includes();
    fgen.generate();
    fgen.close();
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Code generation failed." << std::endl;
    return 1;
  }
  return 0;
}

int osixaka2::generate_dot(const std::string &outputdir) {

  const unit_props &unit = registry_.units_.back();
  std::string filepath = outputdir + unit.basename_ + ".dot";
  std::ofstream ostm(filepath.c_str());
  if (!ostm.is_open()) {
    ostm_ << "Failed to open file, " + aka::quote(filepath) + "." << std::endl;
    return 1;
  }

  try {
    dot_generator gen(registry_, ostm);
    gen.generate();
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Failed to generate dot file." << std::endl;
    return 1;
  }
  ostm.close();
  return 0;
}


int osixaka2::akaxisonize(int level) {
  try {
    for (unit_array::const_iterator uit = registry_.units_.begin();
	 uit != registry_.units_.end(); ++uit) {
      akaxisonizer akx(registry_, registry_[*uit], verbose_);
      akx.add_missings();
      if (level == 1)
	akx.optimize();
      akx.akaxisonize();
    }
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Failed during akaxisonization." << std::endl;
    return 1;
  }
  return 0;
}

int osixaka2::dump_schemas(std::ostream &ostm) const {
  try {
    for (unit_array::const_iterator it = registry_.units_.begin();
	 it != registry_.units_.end(); ++it)
      serialize(*it->schema_, ostm);
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    return 1;
  }
  return 0;
}
