#include "preprocessor.h"
#include "util.h"
#include "xschemadeserializer.h"

using namespace osx;

xs::schema* preprocessor::deserialize(const std::string &filename) {

  xschemadeserializer deserializer(ostm_, verbose_);
  xs::schema *target_schema = 
    deserializer.deserialize(replace_path_seperator(filename));
  
  if (target_schema == 0)
    throw fatal_error(aka::quote(filename) + " : Failed to parse."
		      , __FILE__, __LINE__);
  return target_schema;
}

void preprocessor::preprocess(const std::string &filepath) {

  std::string simplified_path = simplify_path(filepath);
  if (registry_.is_processed(simplified_path))
    return;

  basepath_ = get_dirpath(simplified_path);
  xs::schema *schema = deserialize(simplified_path);

  target_ns_id_ = aka::get_namespace_id(schema->targetNamespace_);
  /** targetNamespace */
  const xs::schema_sequence_c0 &dirs = schema->c0_;
  for (xs::schema_sequence_c0::const_iterator it = dirs.begin();
       it != dirs.end(); ++it) {
    const aka::item &item = *it;
    const aka::qname &tagname = item.get_name();

    if (tagname == aka::qname("xs:include")) {
      const xs::include &inc = aka::item_cast<const xs::include>(item);
      process_include(inc, *schema);
    }
    else if (tagname == aka::qname("xs:import")) {
      process_import(aka::item_cast<xs::import>(item));
    }
    else if (tagname == aka::qname("xs:redefine")) {
      process_redefine(aka::item_cast<xs::redefine>(item));
    }
    else if (tagname == aka::qname("xs:annotation")) {
      // Nothing to do.
    }
  }

  std::string basename = get_basename(filepath);
  unit_props unit(schema, target_ns_id_, basename, simplified_path);
  registry_.units_.push_back(unit);
  // finalDefault, blockDefault, id, xml:lang, version is not processed. !!!!!

}

void preprocessor::process_include(const xs::include &include, 
				   xs::schema &schema) {

  if (include.schemaLocation_.empty()) {
    ostm_ << "xs:include@schemaLocation is empty.  Ignore xs:include." << std::endl;
    return;
  }

  preprocessor pre(registry_, 
		   generate_deps_,
		   ostm_, 
		   verbose_);
  pre.preprocess(basepath_ + include.schemaLocation_);

  if (pre.get_target_ns_id() != target_ns_id_)
    throw fatal_error(include.schemaLocation_ + ": Included document has different namespace.",
		      __FILE__, __LINE__); // The included NS differs from the target one.

  merge_schema(schema, *registry_.units_.back().schema_);
  registry_.units_.pop_back();
}

void preprocessor::process_import(const xs::import &import) {

  if (import.schemaLocation_.empty()) {
    ostm_ << "xs:import@schemaLocation is empty.  Ignore xs:import." << std::endl;
    return;
  }

  preprocessor pre(registry_, 
		   generate_deps_,
		   ostm_, 
		   verbose_);
  pre.preprocess(basepath_ + import.schemaLocation_);
  if (pre.get_target_ns_id() == target_ns_id_)
    throw fatal_error(aka::quote(import.schemaLocation_) + 
		      " : namespace of imported document should not be the same.", 
		      __FILE__, __LINE__); // Importing NS is the same as the target schema doc. 
}

void preprocessor::process_redefine(const xs::redefine &redefine) {
  // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  throw not_implemented(__FILE__, __LINE__);
}

void preprocessor::merge_schema(xs::schema &schema, const xs::schema &to_merge) {
  schema.array0_.insert(schema.array0_.end(),
		    to_merge.array0_.begin(),
		    to_merge.array0_.end());

}
