#include "all_handler.h"

#include "sequence_handler.h"
#include "choice_handler.h"
#include "simpletype_handler.h"
#include "simplecontent_handler.h"
#include "ptrmember_handler.h"
#include "array_handler.h"
#include "fixed_handler.h"
#include "../../framework/construct.h"

#include <cassert>

using namespace aka2;

all_handler::all_handler(bool emptiable, 
			 void *all, const all_op &aop,
			 int depth,
			 const element_props &props,
			 parser_context &context)  
  : handler(emptiable, depth, props, context) , all_(all), aop_(aop) {
  mmap_ = aop_.get_member_map();
  all_construct(all, aop_);
}


validation_result all_handler::query_element(const qname &tagname, const attribute_values &attrs){
  
  current_ = find_element(tagname);
  if (current_ == mmap_.end()) {
    // element specified by tagname not found.
    if (can_skip()) 
      return skip;
    context_.report_no_element(tagname, __FILE__, __LINE__);
    return invalid;
  }

  emptiable_ = false;

  const member_type &mtype = current_->second;
  assert(mtype.is_element());
  validation_result res = parse_element(mtype, attrs);
  return res;
}

validation_result all_handler::query_next(const qname &tagname, const attribute_values &attrs){

  current_ = find_element(tagname);
  if (current_ == mmap_.end()) {
    // element specified by tagname not found.
    return can_skip() ? skip : invalid;
  }
  return ok;
}


// Element is already found.  emptiable = false.
validation_result all_handler::parse_element(const member_type &mtype, const attribute_values &attrs) {

  const element_op &memberop = mtype.op();
  cache_.set(all_, mtype.op(), mtype.getter());
  cache_.prepare(true); // members or members in members may have defaults.
  void *memelm = cache_.value();

  // Branch according to the member type.
  switch (mtype.get_schematype()) {
  case sequence_id:
  case choice_id:
  case all_id:  {
    create_particle_handler(false, memelm, memberop, 
			    depth_ + 1, 
			    mtype.get_name(), mtype);
    break;
  }
  case simpletype_id:
  case simplecontent_id: {
    create_simple_handler(memelm, memberop, 
			  depth_ + 1, 
			  mtype.get_name(), mtype);
    break;
  }
  case ptrmember_id: {
    handler *h = create_ptrmember_handler(false, memelm, depth_, 
					  mtype.get_name(), mtype);
    validation_result res = h->query_element(mtype.get_name(), attrs);
    assert(res == ok);
    break;
  }
  case array_id: {
    handler *h = create_array_handler(false, memelm, 
				      depth_, 
				      mtype.get_name(), mtype);
    validation_result res = h->query_element(mtype.get_name(), attrs);
    assert(res == ok);
    break;
  }
  case fixed_id: { // @fixed is always simpletype. (spec. restriction of akaxiso.)
    create_fixed_handler(0, static_cast<const fixed_op&>(memberop), 
			 depth_ + 1, 
			 mtype.get_name(), mtype);
    break;
  }
  case any_id: { // anyType.
    handler *h = create_any_handler(*static_cast<any*>(memelm), 
				    depth_ + 1,
				    mtype.get_name(), mtype);
    validation_result res = h->query_element(mtype.get_name(), attrs);
    assert(res == ok);
    break;
  }
  case any_array_id: {
    handler *h = create_any_array_handler(false,
					  *static_cast<any_array*>(memelm), 
					  depth_, mtype);
    validation_result res = h->query_element(mtype.get_name(), attrs);
    assert(res == ok);
    break;
  }
  case enclose_id: // use of enclose/disclose is restricted for sequence.
  case disclose_id: 
  case wildcard_attribute_id: // attribute is not handled here.
  case wildcard_id:
    assert(!"Internal error.  Must not reach here.");
  }
  return ok;
}


validation_result all_handler::end_element(const qname &tagname){
  // No element left.  OK to end parse. 
  if (can_skip())
    return ok;
  context_.report_no_element(tagname, __FILE__, __LINE__);
  return invalid;
}


bool all_handler::parse_entity(const std::string &entity) {
  return true;
}


node all_handler::get_node() {
  return node(all_, aop_);
}

void all_handler::receive_child(const node &child) {
  assert(&child.op() == &current_->second.op());
  cache_.flush();
  mmap_.erase(current_->first);
  current_ = mmap_.end();
}

bool all_handler::can_skip() {

  if (emptiable_)
    return true;

  for (member_map::const_iterator it = mmap_.begin();
       it != mmap_.end(); ++it) {
    const member_type &mtype = it->second;
    if (!mtype.get_occurrence().in_range(0))
      return false;
  }
  return true;
}

void all_handler::abort() {
  cache_.clear();
}

member_map::const_iterator all_handler::find_element(const qname &tagname) const {
  member_map::const_iterator it = mmap_.find(tagname);
  if (it != mmap_.end())
    return it;
  for (it = mmap_.begin(); it != mmap_.end(); ++it) {
    if (context_.compare_tagname(tagname, it->first))
      return it;
  }
  return mmap_.end();
}
