/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_ANY_H__
#define AKAXISO2_FRAMEWORK_ANY_H__

/**
 * @file akaxiso2/framework/any.h
 * @brief builtin ur-type classes for akaxiso2 datamodel.
 */

#include <akaxiso2/framework/qname.h>
#include <akaxiso2/framework/document.h>
#include <vector>

namespace aka2 {

  /** 
   * @brief akaxiso builtin type to represent any attribute entity like xs:anyAttribute.
   */
  struct wc_attribute {
    wc_attribute(){}
    wc_attribute(const qname &name, const std::string &value) 
      : name_(name), value_(value) {}
    /** tag name of attribute */
    qname name_;
    /** value of attribute (xs:anySimpleType) */
    std::string value_;
  };

  /**
   * @brief array of wildcard attribute.
   */
  typedef std::vector<wc_attribute> wc_attributes;

  /**
   * @brief akaxiso builtin type to represent xs:anyType in XML Schema.
   *
   * akaxiso uses aka2::any class to store wildcard \(like xs:anyType\) values.\n
   * Text nodes in mixed content will be ignored.\n
   * If wildcard element has child elements, children are put to the children_ member. \n
   * If wildcard element has text entity like \<element\>TEST\</element\>, 
   * text entity is put to the value_ member.\n
   * Both children_ and value_ member should not have their contents at the same time.
   */
  struct wildcard {
    /** @brief tag name */
    qname name_;
    /** @brief any attributes */
    wc_attributes attributes_;
    /** @brief value of this element if this wildcard element has value of xs:anySimpleType. */
    std::string value_;
    /** @brief array of any element if this wildcard element has child elements. */
    std::vector<wildcard> children_; 
    bool empty() const { return value_.empty() && attributes_.empty() && children_.empty(); }
  };


  /**
   * @brief akaxiso2-builtin type to represent any element (like xs:any).
   *
   * aka2::any is the top-node for any-type element.
   * aka2::any can have one of two types of entity.
   * If the entity is a root node of a document that is registered to akaxiso2,
   * document_ member will have deserialized document instance.
   * If the entity is just an wildcard element, aka2::any will have wildcard entities
   * in wc_ member.
   */
  struct any {
    /** wildcard child entity */
    wildcard wc_;
    /** deserialized document instance. */  
    document document_;

    /** 
     * tells this element is empty.
     * @return true neigher wc_ is empty nor document_ is empty.
     */
    bool empty() const { return (document_.empty() == 0) && wc_.empty(); }

    /**
     * tells this element is an external document.
     * @return true if entity is an external document. 
     */
    bool has_document() const { return !document_.empty(); }
  };

  /**
   * @brief array of aka2::wildcard.
   */
  typedef std::vector<wildcard> wc_array; 

  /**
   * @brief array of aka2::any.
   */
  typedef std::vector<any> any_array; 
}

bool operator==(const aka2::wildcard &lhs, const aka2::wildcard &rhs);

#endif
