#include "iconv_transcoder.h"

#ifdef AKAXISO2_USE_ICONV

#include <iostream>
#include <sstream>
#include <errno.h>

#include "../unicode/utf8_traits.h"
#include "encoding_name.h"
#include "../util/string_funcs.h"

using namespace aka2;

iconv_transcoder::~iconv_transcoder() {
  iconv_close(cd_);
}

void iconv_transcoder::reset() {
  size_t inbytesleft = 0;
  char outbuf[16];
  size_t outbytesleft = 16;
  char *outpos = outbuf;
  iconv(cd_, 0, &inbytesleft, &outpos, &outbytesleft);
}

bool iconv_transcoder::transcode(const char *to_convert, size_t length) {
  inbuf_.append(to_convert, length);

  const char *inbuf_current = inbuf_.get_ptr();
  unsigned int inbytesleft = inbuf_.get_content_length();

  while (inbytesleft > 0) {

    char *outbuf = outbuf_.get_end_ptr(inbytesleft);
    unsigned int outbytesleft = outbuf_.get_remain_length();

    int res = iconv(cd_, 
		    const_cast<char**>(&inbuf_current), &inbytesleft, 
		    &outbuf, &outbytesleft);
    outbuf_.commit_current_end(outbuf);

    if (res >= 0) // Conversion OK.
      break;
    
    assert(res == -1);
    int err = errno;
    
    if (err == E2BIG) {
      continue;
    }
    else if (err == EINVAL) {// Input is not complete string.
      inbuf_.erase_by(inbuf_current);
      return false;
    }    
    else if (err == EILSEQ) { // ill_sequence
      switch (from_type_) {
      case utf8_type: {
	size_t length = utf8_escape(inbuf_current, inbuf_.get_end_ptr());
	inbuf_current += length;
	inbytesleft -= length;
	break;
      }
      case ucs2_type: {
	assert((inbuf_.get_end_ptr(0) - inbuf_current) >= 2);
	ucs2_escape(*reinterpret_cast<const aka2::uchar_t*>(inbuf_current));
	inbuf_current += 2;
	inbytesleft -= 2;
	break;
      }
      case other_types:
	throw error("ill sequence found.", __FILE__, __LINE__);
      }
    }
    else {
      std::ostringstream ostm;
      ostm << "iconv error : " << err;
      throw error(ostm.str(), __FILE__, __LINE__);
    }
  }
  inbuf_.clear();
  return true;
}


transcoder *
iconv_transcoder::create(const std::string &tocode, const std::string &fromcode) {

  iconv_t cd = iconv_open(tocode.c_str(), fromcode.c_str());
  if (cd == iconv_t(-1)) {
    throw error("encoding " + quote(fromcode) + " or " + quote(tocode) + " not found.",
		__FILE__, __LINE__);
  }
  return new iconv_transcoder(cd);
}



#endif
