/* -*- c++ -*- */
#ifndef AKAXISO2_TRANSCODERS_TRANSCODER_H__
#define AKAXISO2_TRANSCODERS_TRANSCODER_H__

/**
 * @file akaxiso2/transcoders/transcoder.h
 * @brief transcoder interface and base classes. 
 */

#include <string>
#include <akaxiso2/unicode/string_buffer.h>
#include <akaxiso2/transcoders/encoding_name.h>

namespace aka2 {

  class transcoder {
    friend class stateful_transcoder;
    friend class string_transcoder;
    friend class pivot_transcoder;
  public:
    /**
     * @brief transcoding result.
     */
  protected:
    enum encoding_type {
      utf8_type =0,
      ucs2_type =1,
      other_types = 2
    };
    
    transcoder() { }
    string_buffer outbuf_;
    string_buffer inbuf_;
    encoding_type to_type_;
    encoding_type from_type_;
    
    /**
     * Escape ill_sequence for UTF-8.
     * Character encoding of parameter-passed char is assumed to be UTF-8.
     * Escape one character and return the utf8 char length.
     * @return length of escaped UTF-8 char length.
     */
    size_t utf8_escape(const char *utf8, const char *utf8end);

    /**
     * Escape ill_sequence for UCS2.
     * Character encoding of parameter-passed uchar is assumed to be UCS2.
     * Escape one character, and write the result to outbuf_.
     */
    void ucs2_escape(const aka2::uchar_t uch);

  public:
    /** reset transcoder */
    virtual void reset() = 0;

    /**
     * @brief write BOM if in outbuf_.
     */
    bool write_bom(const struct char_mark *bom);

    /**
     * @brief clear BOM if in outbuf_.
     */
    bool clear_bom(const struct char_mark *bom);

    /**
     * @brief set encoding type.
     * Get the encoding type to determine algorithmic builtin-transcoders.
     */
    void set_encoding_types(const std::string &tocode, 
			    const std::string &fromcode);

  public:
    virtual ~transcoder() { }

    virtual bool transcode(const char *to_convert, size_t length) = 0;

    /** clear buffers and reset transcoder */
    void clear();

    /** clear transcoded buffer. */
    void clear_transcoded_buffer();
  };


  class ubuffered_transcoder : public transcoder {
  protected:
    ubuffered_transcoder() {}

    ustring_buffer ubuffer_;

    virtual bool from_inbuf() = 0;
    virtual bool to_outbuf(const aka2::uchar_t *pivot) = 0;

  public:
    virtual ~ubuffered_transcoder();

    virtual bool transcode(const char *to_convert, size_t length);

    //virtual void reset() = 0;
    static transcoder *create(const std::string &tocode, const std::string &fromcode,
			      bool is_stateful);
  };

  typedef transcoder *(*transcoder_factory)(const std::string &tocode, 
					    const std::string &fromcode);

  transcoder *create_transcoder(const std::string &tocode, 
				const std::string &fromcode,
				transcoder_factory tf);
}

#endif
