//CUPPA:include=+
//CUPPA:include=-
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestAssert.h>

#include <akaxiso2/akaxiso2.h>
#include <akaxiso2/content_model.h>

#include <iostream>

//CUPPA:namespace=+
//CUPPA:namespace=-


namespace {

  typedef std::list<std::string> fixed_array;
  typedef aka2::fixed_sequential_array<fixed_array, xiso::leaf<std::string> > fixed_array_leaf;
  

  struct SeqSample {
    SeqSample() : ptrlong_(0){}
    double dbl_;
    float float_;
    long long_;
    short short_;
    std::string string_;
    aka2::deep_ptr<long> ptrlong_;
    fixed_array fixed_array_;
  };


  class SeqSampleLeaf : public aka::sequence<SeqSample, SeqSampleLeaf> {
  public:
    void model() {
      xmltype("sequence_sample");
      member("double", &SeqSample::dbl_);
      member("float", &SeqSample::float_);
      member("long", &SeqSample::long_);
      member("short", &SeqSample::short_);
      member("string", &SeqSample::string_);
      ptrmember("ptrlong", &SeqSample::ptrlong_); // minOccurs = 0, maxOccurs = 1.
      fixed_member<std::string>("fixed_string", "fixed_value");
      fixed_member<std::string>("fixed_string2", "fixed_value2", xiso::leaf<std::string>());
      fixed_array("fixed_array", "array_value", 
		  &SeqSample::fixed_array_, fixed_array_leaf(),
		  0, aka2::unbounded);
    }
  };

  struct SeqSample1 {  };

  typedef std::list<long> dummyList;
  struct dummyListLeaf : public aka::sequential_array<dummyList, xiso::leaf<long> > {  };
  struct EmptySequence {
    dummyList list_;
  };
  struct EmptySequenceLeaf : public aka::sequence<EmptySequence, EmptySequenceLeaf> {
    void model() {
      member("dummy", &EmptySequence::list_, dummyListLeaf(), 0, aka::unbounded);
    }
  };

}


class sequenceTest : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE(sequenceTest);
//CUPPA:suite=+
  CPPUNIT_TEST(test_equality0);
  CPPUNIT_TEST(test_equality1);
  CPPUNIT_TEST(test_serialize);
  CPPUNIT_TEST(test_serialize_empty_member_array);
  CPPUNIT_TEST(test_parse);
  CPPUNIT_TEST(test_copy);
  CPPUNIT_TEST(test_replicate);
  CPPUNIT_TEST(test_parse_empty_member_array);
  CPPUNIT_TEST(test_wrongSequence0);
  CPPUNIT_TEST(test_wrongSequence1);
//CUPPA:suite=-
  CPPUNIT_TEST_SUITE_END();
private:

//    // your stuff...

public:

  virtual void setUp() { 
    aka::initialize();
    aka::doctype("sequence_test", SeqSampleLeaf());
    aka::doctype("empty_array", EmptySequenceLeaf());
  }

  virtual void tearDown(){
    aka::uninitialize();
  }


  static void setValues(SeqSample &seq){
    seq.dbl_ = 0.2253;
    seq.float_ = 0.225f;
    seq.long_ = 1638453;
    seq.short_ = 32600;
    seq.string_ = "Test String for a member in sequence.";
  }

//CUPPA:decl=+
  void test_equality0(){
    SeqSample seq;
    SeqSample seq1;

    setValues(seq);
    setValues(seq1);

    bool equal = aka::equals(seq, seq1, SeqSampleLeaf());
    CPPUNIT_ASSERT_MESSAGE("Equality test failed.", equal);
  }

  void test_equality1(){
    SeqSample seq;
    SeqSample seq1;

    setValues(seq);
    setValues(seq1);
    seq.dbl_ = 0;

    bool equal = aka::equals(seq, seq1, SeqSampleLeaf());
    CPPUNIT_ASSERT_MESSAGE("Equality test failed.", !equal);
  }


  void test_copy() {
    SeqSample seq;
    SeqSample seq1;

    setValues(seq);
    aka::copy_element(seq1, seq, SeqSampleLeaf());
    bool is_equal = aka::equals(seq, seq1, SeqSampleLeaf());
    CPPUNIT_ASSERT_MESSAGE("Copy failed.", is_equal);
  }

  void test_replicate(){
    SeqSample seq;

    setValues(seq);
    SeqSample *seq1 = aka::replicate_element(seq, SeqSampleLeaf());
    bool is_equal = aka::equals(seq, *seq1, SeqSampleLeaf());
    delete seq1;

    CPPUNIT_ASSERT_MESSAGE("Replicate failed.", is_equal);
  }

//   void test_compare_different_type(){
//     SeqSample seq;
//     SeqSample1 seq1;
    
//     bool exception_caught = false;
    
//     try {
//       element_equals(&seq, &seq1, SeqSampleLeaf::dispatcher_);
//     }
//     catch (... ) {
//       exception_caught = true;
//     }
//     CPPUNIT_ASSERT_MESSAGE("Exception 'invalid_argument' not thrown", exception_caught);
//   }


  void test_serialize() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    SeqSample root;
    setValues(root);
    ser.serialize(root, "sequence_test", ostm);
//      std::cout << ostm.rdbuf()->str() << std::endl;
  }


  void test_serialize_empty_member_array() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    EmptySequence root;
    ser.serialize(root, "empty_array", ostm);
//      std::cout << ostm.rdbuf()->str() << std::endl;
  }

  void test_parse() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    SeqSample root;
    setValues(root);
    root.fixed_array_.push_back(std::string());

    ser.serialize(root, "sequence_test", ostm);
    //    std::cout << ostm.rdbuf()->str() << std::endl;

    aka::expat_deserializer deserializer;

    std::string errMsg;
    aka::document deserialized;
    try {
      deserialized = deserializer.deserialize(ostm.rdbuf()->str());
    }
    catch (const std::exception &e) {
      errMsg = e.what();
    }

    CPPUNIT_ASSERT_MESSAGE(errMsg.c_str(), errMsg.empty());

    CPPUNIT_ASSERT_MESSAGE("wrong document name. ", 
			   aka::document_of(deserialized, "sequence_test"));


    SeqSample *seq = aka::root_cast<SeqSample>(deserialized);
    bool is_equal = aka::equals(*seq, root, SeqSampleLeaf());
    if (!is_equal)
      ser.serialize(*seq, "sequence_test", std::cout);
    CPPUNIT_ASSERT_MESSAGE("Deserialized document is not the same as original.", is_equal);

  }


  void test_parse_empty_member_array() {
    std::ostringstream ostm;
    aka::xml_serializer ser;

    EmptySequence root;
    ser.serialize(root, "empty_array", ostm);

    aka::document doc = aka::deserialize(ostm.rdbuf()->str());
    CPPUNIT_ASSERT_MESSAGE("parsed document has wrong name.", 
			   aka::document_of(doc, "empty_array"));
    
    EmptySequence *parsed = aka::root_cast<EmptySequence>(doc);
    CPPUNIT_ASSERT_MESSAGE("parsed document has wrong-sized array", parsed->list_.empty());
  }


   void test_wrongSequence0() {
    
     std::string doc = 
       "<?xml version=\"1.0\"?>\n"
       "<sequence_test>\n"
       "  <double>0.2</double>\n"
       "  <float>3.0</float>\n"
       "  <long>2</long>\n"
       // should be short.
       "  <string>Test</string>\n"
       "  <short>2</short>\n"
       "  <ptrlong>11</ptrlong>\n"
       "  <fixed_string>fixed_value</fixed_string>\n"
       "  <fixed_string2>fixed_value2</fixed_string2>\n"
       "  <fixed_array>array_value</fixed_array>\n"
       "</sequence_test>";
     
     bool err = false;
     try {
       aka::deserialize(doc);
     }
     catch (const std::exception &) {
       err = true;
     }
     CPPUNIT_ASSERT_MESSAGE("parse error not reported.", err);
   }

   void test_wrongSequence1() {
    
     std::string doc = 
       "<?xml version=\"1.0\"?>\n"
       "<sequence_test>\n"
       "  <double>0.2</double>\n"
       "  <float>3.0</float>\n"
       "  <long>2</long>\n"
       "  <short>2</short>\n"
       "  <string>Test</string>\n"
       "  <ptrlong>11</ptrlong>\n"
       "  <fixed_string>fixed_value</fixed_string>\n"
       // Lack of fixed_string.
//        "  <fixed_string2>fixed_value2</fixed_string2>\n"
       "  <fixed_array>array_value</fixed_array>\n"
       "</sequence_test>";

     bool err = false;
     try {
       aka::deserialize(doc);
     }
     catch (const std::exception &) {
       err = true;
     }
     CPPUNIT_ASSERT_MESSAGE("parse error not reported.", err);
   }


//CUPPA:decl=-
};

//CUPPA:impl=+
//CUPPA:impl=-

CPPUNIT_TEST_SUITE_REGISTRATION(sequenceTest);
