#include "registry.h"

using namespace osx;

registry::registry() {
  registry_unit regunit(0, aka::unregistered_token, "", "");
  units_.push_back(regunit);
}

void registry::set_preference(const preference &pref) {
  pref_ = &pref;
  units_.front().set_preference(pref);
}


registry_unit &registry::operator[](const aka::id_type target_ns_id) {
  for (registry_units::iterator it = units_.begin();
       it != units_.end(); ++it) {
    if (it->target_ns_id_ == target_ns_id)
      return *it;
  }
  assert(!"Must not reach here.");
  return *static_cast<registry_unit*>(0); // supress warning.
}

units registry::get_units() {
  units uts;
  registry_units::iterator it = units_.begin();
  ++it;
  for (;it != units_.end(); ++it) {
    uts.push_back(&*it);
  }
  return uts;
}

const_units registry::get_units() const {
  const_units uts;
  registry_units::const_iterator it = units_.begin();
  ++it;
  for (;it != units_.end(); ++it) {
    uts.push_back(&*it);
  }
  return uts;
}

void registry::add_unit(const registry_unit &unit) {
  units_.push_back(unit);
}


namespace {
  // type search will be done in reversed order for regstry_unit,
  // in order to enable overriding preference.

  template<class I, class E>
  TYPENAME E::value_type *find_def(const aka::qname &name, I beg, I end, const E &e) {
    typedef TYPENAME E::value_type value_type;
    assert(beg != end);

    I it = end;
    do {
      --it;
      value_type *v = e.get(name, *it, static_cast<value_type*>(0));
      if (v != 0)
	return v;
    } while (it != beg);
    return 0;
  }

  template<class I, class E>
  const TYPENAME E::value_type *const_find_def(const aka::qname &name,
					       I beg, I end, const E & e) {
    typedef TYPENAME E::value_type value_type;
    assert(beg != end);

    I it = end;
    do {
      --it;
      const value_type *v = e.get(name, *it, static_cast<const value_type*>(0));
      if (v != 0)
	return v;
    } while (it != beg);
    return 0;
  }

  struct globalAttribute_ex {
    typedef gattribute_def value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.gattributes_.get(name);
    }
  };

  struct attributeGroup_ex {
    typedef attributeGroup_def value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.attributeGroups_.get(name);
    }
  };

  struct complexType_ex {
    typedef user_class value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.complexTypes_.get(name);
    }
  };

  struct simpleType_ex {
    typedef user_class value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.simpleTypes_.get(name);
    }
  };

  struct group_ex {
    typedef user_class value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.groups_.get(name);
    }
  };

  struct array_ex {
    typedef user_class value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.arrays_.get(name);
    }
  };

  struct element_ex {
    typedef element_def value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.elements_.get(name);
    }
  };

  struct regunit_type_ex {
    typedef registry_unit value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.type_exists(name) ? &regunit : 0;
    }
  };

  struct find_imported_type_ex {
    typedef imported_type value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.imports_.find_import(name);
    }
  };

  struct resolve_imported_type_ex {
    typedef imported_type value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.imports_.resolve(name);
    }
  };

  struct find_builtin_type_ex {
    typedef imported_type value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.imports_.find_builtin(name);
    }
  };

  struct type_get_ex {
    typedef user_class value_type;
    template<class D, class R>
    D *get(const aka::qname &name, R &regunit, D*) const {
      return regunit.type_get(name);
    }
  };
}

const registry_unit *registry::find_unit_by_type(const type_ref &type) const {
  for (registry_units::const_iterator it = units_.begin();
       it != units_.end(); ++it) {
    if (it->type_exists(type))
      return &*it;
  }
  return 0;
}

bool registry::type_exists(const type_ref &type) const {
  const registry_unit *unit = find_unit_by_type(type);
  return unit != 0;
}

const gattribute_def *registry::globalAttribute_get(const aka::qname &name) const {
  return const_find_def(name, units_.begin(), units_.end(), globalAttribute_ex());
}

const attributeGroup_def *registry::attributeGroup_get(const aka::qname &name) const {
  return const_find_def(name, units_.begin(), units_.end(), attributeGroup_ex());
}

user_class *registry::complexType_get(const aka::qname &name) {
  return find_def(name, units_.begin(), units_.end(), complexType_ex());
}

const user_class *registry::simpleType_get(const aka::qname &name) const {
  return const_find_def(name, units_.begin(), units_.end(), simpleType_ex());
}

user_class *registry::group_get(const aka::qname &name) {
  return find_def(name, units_.begin(), units_.end(), group_ex());
}


element_def *registry::element_get(const aka::qname &name) {
  return find_def(name, units_.begin(), units_.end(), element_ex());
}


bool registry::erasable(const type_ref &type) const {
  if (type.get_name() != type.get_classname())
    return false;
  const registry_unit *unit = find_unit_by_type(type);
  assert(unit != 0);
  return unit->erasable(type);
}


import_id registry::get_import_id(const aka::qname &type) const {
  const imported_type *imported = get_imported_type(type);
  assert(imported != 0);
  return imported->import_id_;
}

const imported_type* registry::get_imported_type(const aka::qname &name) const {
  return const_find_def(name, units_.begin(), units_.end(), find_imported_type_ex());
}

const imported_type* registry::get_builtin_type(const aka::qname &name) const {
  return const_find_def(name, units_.begin(), units_.end(), find_builtin_type_ex());
}

const imported_type *registry::resolve_imported_type(const imported_type *imported) const {
  if (imported->resolve_.empty())
    return imported;
  return const_find_def(imported->resolve_, 
			units_.begin(), units_.end(), resolve_imported_type_ex());
}

type_ref registry::resolve_type(const type_ref &type) const {

  assert(!type.empty());

  if (type.is_imported()) {
    const imported_type *imported = resolve_imported_type(type.imported_);
    assert(imported != 0);
    return type_ref(imported);
  }
  else {
    const user_class *us = type.us_;
    assert(us != 0);
    if (us->id_ != simpletype_id)
      return type_ref(us);
    else
      return resolve_typedef(us->get_value_type());
  }
}


type_ref registry::resolve_typedef(const type_ref &type) const {

  assert(!type.empty());

  const user_class *us = type.us_;
  if (us != 0) {
    if (us->id_ != simpletype_id)
      return type_ref(us);
    else
      return resolve_typedef(us->get_value_type());
  }

  const imported_type *imported = resolve_imported_type(type.imported_);
  assert(imported != 0);
  return type_ref(imported);
}


bool registry::is_xs_any(const aka::qname &type) const {
  return (type == aka::qname("xs:anyType")) 
    || (type == aka::qname("xs:anySimpleType"));
}

bool registry::is_aka_any(const aka::qname &type) const {
  return (type == aka::qname("aka2:any"))
    || (type == aka::qname("aka2:any_array"))
    || (type == aka::qname("aka2:wildcard"));
}


const imported_type *registry::get_imported_array_type(const imported_type *imported) const {
  const imported_type *resolved = resolve_imported_type(imported);
  assert(resolved != 0);

  if (resolved->import_id_ == import_array) {
    return imported;
  }

  assert(!resolved->array_name_.empty());
  const imported_type *ret = get_imported_type(resolved->array_name_);
  if (ret != 0)
    return ret;
  return get_builtin_type(resolved->array_name_);
}

aka::qname registry::create_array_type_name(const aka::qname &name) const {
  aka::qname arrayname(name.get_namespace_id(), 
		       pref_->format_.array_.prefix_ 
		       + name.local() 
		       + pref_->format_.array_.suffix_);
    return arrayname;
}

const user_class *registry::find_array_type(const user_class *us) const {

  const registry_unit &regunit = *find_unit_by_type(type_ref(us));
  const aka::qname &name = us->get_name();

  const user_classes *uss = 0;
  if (regunit.simpleTypes_.get(name) != 0)
    uss = &regunit.simpleTypes_;
  else if (regunit.complexTypes_.get(name) != 0)
    uss = &regunit.complexTypes_;
  else if (regunit.groups_.get(name) != 0)
    uss = &regunit.groups_;
  else if (regunit.local_complexTypes_.get(name) != 0)
    uss = &regunit.local_complexTypes_;
  else if (regunit.local_simpleTypes_.get(name) != 0)
    uss = &regunit.local_simpleTypes_;
  else if (regunit.local_classes_.get(name) != 0)
    uss = &regunit.local_classes_;
  else {
    return 0;
  }

  aka::qname array_name = create_array_type_name(us->get_name());
  if (uss->get(array_name) != 0)
    return uss->get(array_name);
  return 0;
}

type_ref registry::resolve_type_ref(const type_ref &type) {
  const aka::qname &refer = type.referred_name_;

  const registry_unit *regunit = 0;

  for (registry_units::const_iterator it = units_.begin();
       it != units_.end(); ++it) {
    if (it->target_ns_id_ == refer.get_namespace_id()) {
      regunit = &*it;
      break;
    }
  }

  if (regunit != 0) {
    if (type.is_element_) {
      const element_def *edef = regunit->elements_.get(refer);
      if (edef == 0) {
	raise_name_error("xs:element", refer, "not defined.", __FILE__, __LINE__);
      }
      return edef->type_;
    }
    else {
      if (type.is_local_) {
	if (regunit->local_simpleTypes_.get(refer) != 0)
	  return type_ref(regunit->local_simpleTypes_.get(refer));
	else if (regunit->local_complexTypes_.get(refer) != 0)
	  return type_ref(regunit->local_complexTypes_.get(refer));
      }
      else {
	if (regunit->simpleTypes_.get(refer) != 0)
	  return type_ref(regunit->simpleTypes_.get(refer));
	else if (regunit->complexTypes_.get(refer) != 0)
	  return type_ref(regunit->complexTypes_.get(refer));
      }
    }
  }

  if (get_imported_unit().type_exists(type)) {
    const imported_type *imported = 
      get_imported_unit().imports_.find_import(refer);
    return type_ref(imported);
  }
  
  raise_name_error("type", refer, "not found.", __FILE__, __LINE__);
  return type_ref();
}

type_ref registry::get_aka_any() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_builtin(aka::qname("aka2:any"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::get_aka_any_array() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_builtin(aka::qname("aka2:any_array"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::get_std_string() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_builtin(aka::qname("std:string"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::get_std_string_array() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_builtin(aka::qname("std:string_array"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::get_aka_wildcard() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_builtin(aka::qname("aka2:wildcard"));
  assert(imported != 0);
  return type_ref(imported);
}


type_ref registry::get_anyType() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_import(aka::qname("xs:anyType"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::get_anySimpleType() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_import(aka::qname("xs:anySimpleType"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::get_nill_type() const {
  const imported_type *imported = 
    get_imported_unit().imports_.find_builtin(aka::qname("aka2:nill"));
  assert(imported != 0);
  return type_ref(imported);
}

type_ref registry::create_array(const type_ref &value_type, user_classes &arrays) {
  if (!value_type.is_imported()) {
    assert(value_type.us_->id_ != array_id);
  }
  aka::qname array_name = create_array_type_name(value_type.get_name());
  aka::qname array_classname = create_array_type_name(value_type.get_classname());
  user_class *ardef = 0;
  if (arrays.get(array_name) != 0)
    ardef = arrays.get(array_name);
  else
    ardef = user_class::user_array(array_name, array_classname, 
				   aka::occurrence(), value_type, arrays);
  return type_ref(ardef);
}

aka::qname registry::create_particle_item_name(const aka::qname &name) const {
  aka::qname itemname(name.get_namespace_id(), 
		      pref_->format_.particle_item_.prefix_
		      + name.local()
		      + pref_->format_.particle_item_.suffix_);
  return itemname;
}
									    
