#include "XMLSchema_xiso.h"
#include <akaxiso2/util/string_funcs.h>
#include <akaxiso2/util/tokenizer.h>
#include <akaxiso2/builtin/validator.h>

#include "schema_st.h"

namespace xs {

  void allNNI_leaf::write_text(const void *e, std::ostream &ostm,
			       aka::entity_complements &ecomp) {
    const long &v = *static_cast<const long*>(e);
    if (v == aka::unbounded)
      ostm << "unbounded";
    else
      ostm << v;
  }
  
  void allNNI_leaf::read_text(void *e, const std::string &entity, 
			      aka::entity_complements &ecomp) {
    long &v = *static_cast<long*>(e);
    std::string val = aka::trim(entity);
    if (val == "unbounded") {
      v = aka::unbounded;
      return;
    }
    
    aka::isstream istm(entity);
    istm >> v;
    if (istm.fail())
      throw aka::error(std::string("Failed to parse allNNI(maxOccurs) value.(")
		       + istm.rdbuf()->str() + ")",
		       __FILE__, __LINE__);

    if (v < 0) {  // allNNI is nonNegativeInteger.
      throw aka::error(std::string("Wrong allNNI(maxOccurs) value.(")
		       + istm.rdbuf()->str() + ")",
		       __FILE__, __LINE__);

    }
  }



  void attribute_use_leaf::write_text(const void *e, std::ostream &ostm,
				      aka::entity_complements &ecomp) {
    const attribute_use &use = *static_cast<const attribute_use*>(e);
    switch (use) {
    case use_required:
      ostm << "required";
      break;
    case use_optional:
      ostm << "optional";
      break;
    case use_prohibited:
      ostm << "ptohibited";
      break;
    default:
      assert(!"Must not reach here.");
    }
  }
  
  void attribute_use_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
					     aka::entity_complements &ecomps) {
    attribute_use &use = *static_cast<attribute_use*>(e);
    std::string lcp = xs::to_NMTOKEN(entity, "attribute_use", ecomps);;

    if (lcp == "required")
      use = use_required;
    else if (lcp == "optional")
      use = use_optional;
    else if (lcp == "prohibited")
      use = use_prohibited;
    else {
      throw aka::error(std::string("Wrong attribute_use value, ") + aka::quote(lcp) + ".",
		       __FILE__, __LINE__);
    }
  }


  bool is_use_required(const xs::attribute_use &val) {
    return val == xs::use_required;
  }

  bool is_use_prohibited(const xs::attribute_use &val) {
    return val == xs::use_prohibited;
  }

  bool is_use_optional(const xs::attribute_use &val) {
    return val == xs::use_optional;
  }



  void blockSet_leaf::write_text(const void *e, std::ostream &ostm,
				 aka::entity_complements &ecomp) {
    const blockSet &bs = *static_cast<const blockSet*>(e);

    if (bs == blockSet_none) {
      return;
    }
    if (bs == blockSet_all) {
      ostm << "#all";
      return;
    }

    bool add_space = false;
    if (bs == blockSet_extension) {
      ostm << "extension";
      add_space = true;
    }

    if (bs == blockSet_restriction) {
      if (add_space)
	ostm << " ";
      else
	add_space = true;
      ostm << "restriction";
    }
    
    if (bs == blockSet_substitution) {
      if (add_space)
	ostm << " ";
      else 
	add_space = true;
      ostm << "substitution";
    }
  }
  
  void blockSet_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
				aka::entity_complements &ecomps) {
    blockSet &bs = *static_cast<blockSet*>(e);
    std::string tokens = to_token(entity, ecomps);

    if (tokens.empty()) {
      bs = blockSet_none;
      return;
    }

    if (tokens == "#all") {
      bs = blockSet_all;
      return;
    }

    bs = blockSet_none;
    cuppa::tokenizer<char> tokenizer(tokens);

    while (tokenizer.has_more_token(" ")) {
      std::string token = tokenizer.next_token(" ");
      if (token == "extension") {
	if (bs & blockSet_extension) {
	  goto wrong_blockSet_value;
	}
	bs |= (int)blockSet_extension;
      }
      else if (token == "restriction") {
	if (bs & blockSet_restriction) {
	  goto wrong_blockSet_value;
	}
	bs |= blockSet_restriction;
      }
      else if (token == "substitution") {
	if (bs & blockSet_substitution) {
	  goto wrong_blockSet_value;
	}
	bs |= blockSet_substitution;
      }
      else {
	goto wrong_blockSet_value;
      }
    }

    return;

  wrong_blockSet_value:
    throw aka::error(std::string("Wrong blockSet value, ") + aka::quote(tokens) + ".",
		     __FILE__, __LINE__);
    
  }
  
  void all_minOccurs_leaf::write_text(const void *e, std::ostream &ostm,
				      aka::entity_complements &ecomp) {
    const long &v = *static_cast<const long*>(e);
    ostm << v;
  }
  
  void all_minOccurs_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
				     aka::entity_complements &ecomps) {
    long &v = *static_cast<long*>(e);
    std::string number = to_NMTOKEN(entity, "all@minOccurs", ecomps);
    aka::isstream istm(number);
    istm >> v;
    if (istm.fail())
      throw aka::error(std::string("Failed to parse all@minOccurs.(")
		       + istm.rdbuf()->str() + ")",
		       __FILE__, __LINE__);

    if ((v != 0) && (v != 1)) { 
      throw aka::error(std::string("Wrong all@minOccurs value.(")
		       + istm.rdbuf()->str() + ")",
		       __FILE__, __LINE__);

    }
  }

  void all_maxOccurs_leaf::write_text(const void *e, std::ostream &ostm,
				      aka::entity_complements &ecomp) {
    const long &v = *static_cast<const long*>(e);
    ostm << v;
  }
  
  void all_maxOccurs_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
				     aka::entity_complements &ecomps) {
    long &v = *static_cast<long*>(e);
    std::string digits = to_NMTOKEN(entity, "all@maxOccurs", ecomps);
    aka::isstream istm(digits);
    istm >> v;
    if (istm.fail())
      throw aka::error(std::string("Failed to parse all@maxOccurs.(")
		       + istm.rdbuf()->str() + ")",
		       __FILE__, __LINE__);

    if (v != 1) { 
      throw aka::error(std::string("Wrong all@maxOccurs value.(")
		       + istm.rdbuf()->str() + ")",
		       __FILE__, __LINE__);

    }
  }
  
  
  
  void namespaceList_leaf::write_text(const void *e, std::ostream &ostm,
				      aka::entity_complements &ecomp) {
    const namespaceList &ns_list = *static_cast<const namespaceList*>(e);
    std::string list_str;
    if (ns_list.size() > 0) {
      list_str = ns_list.front();
    }
    for (size_t index = 1; index < ns_list.size(); ++index) {
      list_str += ' ';
      list_str += ns_list[index];
    }
    ostm << list_str;
  }
  
  void namespaceList_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
				     aka::entity_complements &ecomps) {
    namespaceList &ns_list = *static_cast<namespaceList*>(e);
    ns_list.clear();

    aka::ustring tokens = to_token(aka::pivot_to_ucs2(entity));
    
    if (tokens.empty())
      return; // entity is consist of spaces, just return.


    std::string lcp_tokens = ecomps.to_lcp(aka::ucs2_to_pivot(tokens));
    if (lcp_tokens == "##any") {
      ns_list.push_back("##any");
      return;
    }
    else if (lcp_tokens == "##other") {
      ns_list.push_back("##other");
      return;
    }

    cuppa::tokenizer<aka::uchar_t> tokenizer(tokens);
    std::vector<aka::ustring> ns_tokens;

    aka::ustring space;
    space += aka::uchar_t(' ');

    while (tokenizer.has_more_token(space)) {
      ns_tokens.push_back(tokenizer.next_token(space));
    }

    for (size_t index = 0; index < ns_tokens.size(); ++index) {
      std::string token = ecomps.to_lcp(aka::ucs2_to_pivot(ns_tokens[index]));
      if ((token == "##any") || (token == "##other")) {
	throw aka::error("Wrong namespace list, " + aka::quote(lcp_tokens) + ".",
			 __FILE__, __LINE__);
      }
      ns_list.push_back(token);
    }
  }
  
  
  void formChoice_leaf::write_text(const void *e, std::ostream &ostm,
				   aka::entity_complements &ecomp) {
    const formChoice &form = *static_cast<const formChoice*>(e);
    switch (form) {
    case qualified:
      ostm << "qualified";
      break;
    case unqualified:
      ostm << "unqualified";
      break;
    default:
      assert(!"Must not reach here.");
    }
    
  }
  
  void formChoice_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
					  aka::entity_complements &ecomps) {
    formChoice &form = *static_cast<formChoice*>(e);
    std::string value = to_NMTOKEN(entity, "formChoice", ecomps);

    if (value == "qualified") {
      form = qualified;
    }
    else if (value == "unqualified") {
      form = unqualified;
    }
    else {
      throw aka::error("Wrong formChoice, " + aka::quote(value) + ".",
		       __FILE__, __LINE__);
    }
  }

  bool is_form_qualified(const formChoice &val) {
    return val == qualified;
  }



  void derivationSet_leaf::write_text(const void *e, std::ostream &ostm,
				      aka::entity_complements &ecomp) {
    std::vector<std::string> tokens;
    const derivationSet &dset = *static_cast<const derivationSet*>(e);
    if (dset == derivationSet_extension)
      ostm << "extension";
    else if (dset == derivationSet_restriction)
      ostm << "restriction";
    else if (dset == (derivationSet_extension | derivationSet_restriction))
      ostm << "extension restriction";
    else {
      assert(!"Must not reach here.");
    }
  }
  
  void derivationSet_leaf::read_unicode_text(void *e, const aka2::pstring &entity, 
				     aka::entity_complements &ecomps) {
    derivationSet &form = *static_cast<derivationSet*>(e);
    std::string token = to_token(entity, ecomps);

    if (token.empty()) {
      form = derivationSet_none;
      return;
    }

    if (token == "extension") {
      form = derivationSet_extension;
    }
    else if (token == "restriction") {
      form = derivationSet_restriction;
    }
    else if ((token == "extension restriction") || (token == "restriction extension")) {
      form = derivationSet_extension | derivationSet_restriction;
    }
    else {
      throw aka::error("Wrong derivationSet, " + aka::quote(token) + ".",
		       __FILE__, __LINE__);
    }
  }

}

