#include "platform.h"
#include "classgen.h"
#include <iostream>
#include "exception.h"
#include <assert.h>

namespace xsd = xml_schema;
namespace aka = akaxiso;
namespace osi = osixaka;

using namespace osi;


std::string generator::resolve_typename(const akaxiso::name &name) {
  std::string cpptype = type_registory_.get_type(name);
  if (cpptype.empty())
    throw fatal_error(__FILE__, __LINE__, "name of type not found.");
  return cpptype;
}


int generator::generate_classes(xsd::SchemaDoc *doc) {

  xsd::Schema *sc = doc->get_root();

  ComplexTypeList cel;
  GlobalElementList gel;

  sc->get_typesafe_elementlist(cel, "xsd:complexType");
  sc->get_typesafe_elementlist(gel, "xsd:element");
  
  for (ComplexTypeList::iterator cit = cel.begin(); cit != cel.end(); ++cit) 
    type_registory_.insert_complexType(cit->name_, cit->name_.get_localname());
  for (GlobalElementList::iterator eit = gel.begin(); eit != gel.end(); ++eit)
    type_registory_.insert_element(eit->name_, eit->name_.get_localname());

  generate_global_complexTypes(cel);
  generate_global_elements(gel);

  return 0;
}

std::string generator::get_filesig(const std::string &filename) {
  std::string sig;
  for (std::string::const_iterator it = filename.begin();
       it != filename.end(); ++it) {
    if (*it == '.')
      sig += '_';
    else
      sig += toupper(*it);
  }
  return sig;
}


void generator::write_header(std::ostream &ostm, const std::string &filename) {
  std::string filesig = get_filesig(filename);

  ostm << "/* -*- c++ -*-  (Tell emacs for this file to be cpp-header.)*/" << std::endl
       << "#ifndef " << filesig << "__" << std::endl
       << "#define " << filesig << "__" << std::endl
       << std::endl;

  ostm << "#include <akaxiso/akaxiso.h>" << std::endl
       << std::endl;
  
  ostm << "/**" << std::endl
       << " * @file " << filename << std::endl
       << " * Document class declarations generated by osixaka." << std::endl
       << " */" << std::endl
       << std::endl;

}

void generator::write_footer(std::ostream &ostm, const std::string &filename) {
  std::string filesig = get_filesig(filename);
  ostm << std::endl
       << "#endif //" << filesig << "__" << std::endl;
}


void generator::output_setup_file(std::ostream &header, std::ostream &impl) {

  std::cerr << "Generating setup.h/.cpp..." << std::flush;

  write_header(header, setup_header_);
  header << std::endl
	 << "void initialize();" << std::endl
	 << "void uninitialize();" << std::endl
	 << std::endl;
  write_footer(header, setup_header_);
  
  
  impl << "#include \"" << setup_header_ << "\"" << std::endl
       << std::endl;

  impl << std::endl 
       << "#include \"" << element_fn_ << "\"" << std::endl
       << "#include \"" << document_fn_ << "\"" << std::endl
       << std::endl;

  class_contents_container::iterator it;

  for (it = contents_.begin(); it != contents_.end(); ++it) 
    it->second->generate_implementation(impl);
  for (it = documents_.begin(); it != documents_.end(); ++it) 
    it->second->generate_implementation(impl);

  impl << std::endl
       << "void initialize() {" << std::endl
       << std::endl;


  for (it = contents_.begin(); it != contents_.end(); ++it) 
    impl << "  " << it->second->name_.get_name() << "::initialize();" << std::endl;
  for (it = documents_.begin(); it != documents_.end(); ++it) 
    impl << "  " << it->second->name_.get_name() << "::initialize();" << std::endl;



  impl << std::endl
       << "}" << std::endl
       << std::endl;

  impl << "void uninitialize() {" << std::endl
       << std::endl;

  for (it = contents_.begin(); it != contents_.end(); ++it) 
    impl << "  " << it->second->name_.get_name() << "::uninitialize();" << std::endl;
  for (it = documents_.begin(); it != documents_.end(); ++it) 
    impl << "  " << it->second->name_.get_name() << "::uninitialize();" << std::endl;

  impl << std::endl
       << "}" << std::endl
       << std::endl;

  std::cerr << "done." << std::endl;

}


/* Document */
void generator::output_document_file(std::ostream &ostm) {

  std::cerr << "Generating document.h..." << std::flush;

  // Document classes
  write_header(ostm, document_fn_);
  ostm << std::endl 
       << "#include \"" << element_fn_ << "\"" << std::endl
       << std::endl;
  
  for (class_contents_container::iterator it = documents_.begin(); it != documents_.end(); ++it)
    it->second->generate_header(ostm);

  ostm << std::endl;

  write_footer(ostm, document_fn_);

  std::cerr << "done." << std::endl;
  
}


void generator::output_element_file(std::ostream &ostm) {

  std::cerr << "Generating element.h..." << std::flush;

  // Document classes
  write_header(ostm, element_fn_);
  ostm << std::endl 
       << std::endl;
  
  while (!contents_.empty()) {
    std::vector<class_contents*> output_classes;
    bool no_output = true;
    for (class_contents_container::iterator it = contents_.begin(); it != contents_.end(); ++it) 
      if (!it->second->has_dependent_classes()) {
	it->second->generate_header(ostm);
	output_classes.push_back(it->second);
	no_output = false;
      }

    if (no_output) 
      throw fatal_error(__FILE__, __LINE__, "Circular dependency detected.  Sorry, currently not supported.");

    for (std::vector<class_contents*>::iterator ccit = output_classes.begin(); ccit != output_classes.end(); ++ccit)
      contents_.remove_class_contents(*ccit);
  }


  ostm << std::endl;

  write_footer(ostm, element_fn_);

  std::cerr << "done." << std::endl;
  
}


void generator::generate_global_elements(const GlobalElementList &elms) {

  for (GlobalElementList::const_iterator it = elms.begin(); it != elms.end(); ++it) {
    document_class *doc = new document_class;
    doc->name_.set_name(it->name_.get_name() + "_doc");
    documents_.add(doc->name_, doc);
    
    if (it->type_.empty() && it->complexTypes_.empty())
      throw fatal_error(__FILE__, __LINE__, "xsd:element does not specify its content model.");

    if (!it->complexTypes_.empty()) {
      type_registory_.insert_complexType(it->name_, it->name_.get_name());
      generate_complexType(*it->complexTypes_.begin(), it->name_);
      doc->root_name_ = it->name_.get_name();
    }
    else if (type_registory_.simpleType_exists(it->type_)) { // document root is simpleType.
      const std::string &cpptype = type_registory_.get_simpleType(it->type_);
      generate_simpletype_wrapper(it->name_, cpptype);
      doc->root_name_ = it->name_.get_name();
    }
    else if (type_registory_.complexType_exists(it->type_)) // document root is complexType.
      doc->root_name_= resolve_typename(it->type_);
    else {
      std::ostringstream ostm;
      ostm << "Type not found(" << it->type_.get_name() << ")";
      throw fatal_error(__FILE__, __LINE__, ostm.rdbuf()->str().c_str());
    }
  }
}



void generator::generate_global_complexTypes(const ComplexTypeList &cel) {
  for (ComplexTypeList::const_iterator it = cel.begin(); it != cel.end(); ++it) 
    generate_complexType(*it, it->name_);
}

void generator::generate_complexType(const xsd::ComplexType &ct, const xsd::name &name) {

  assert(!name.empty());
  
  typedef aka::const_elementlist_impl<xsd::Sequence> seqlist_t;
  typedef aka::const_elementlist_impl<xsd::Choice> cholist_t;

  seqlist_t seqlist;
  cholist_t cholist;

  ct.get_typesafe_elementlist(seqlist, "xsd:sequence");
  ct.get_typesafe_elementlist(cholist, "xsd:choice");
  
  if (seqlist.size() == 1)
    generate_sequence(*seqlist.begin(), name);
  else if (cholist.size() == 1)
    generate_choice(*cholist.begin(), name);
  else
    throw fatal_error(__FILE__, __LINE__, "Number of particles must be 1 under xsd:complexType.");
}


void generator::generate_sequence(const xsd::Sequence &seq, const akaxiso::name &name) {
  
  // Create class_contents. 
  sequence_class *cls = new sequence_class;
  contents_.add(name, cls);

  for (xsd::Elements::const_iterator it = seq.elements_.begin(); it != seq.elements_.end(); ++it) {
    
    sequence_member membertype;
    membertype.minOccurs_ = it->minOccurs_;
    membertype.maxOccurs_ = it->maxOccurs_;

    /** name and type specifies the content model  */
    /* <xsd:element name="hoge" type="hogetype"/> */
    if (!it->name_.empty() && !it->type_.empty()) {
      if (!it->complexTypes_.empty() || !it->ref_.empty()) // complxType and ref should be empty.
	throw fatal_error(__FILE__, __LINE__, "");

      // complexType and simpleType is acceptable.
      if (!type_registory_.type_exists(it->type_)) {
	std::ostringstream ostm;
	ostm << "name of type (" << it->type_ << ") not found.";
	throw fatal_error(__FILE__, __LINE__, ostm.rdbuf()->str().c_str());
      }

      membertype.name_ = it->name_.get_localname();
      membertype.type_ = resolve_typename(it->type_);

      cls->push_membertype(membertype, type_registory_.simpleType_exists(it->type_));
    }
    else if (!it->ref_.empty()) { // <xsd:element ref="toplevel_element">
      if (!it->complexTypes_.empty()) // complexType should not appear.
	throw fatal_error(__FILE__, __LINE__, "Both xsd:complexType and xsd:element@ref attributes are specified."); 

      if (!type_registory_.element_exists(it->ref_)) {
	std::ostringstream ostm;
	ostm << "Type of sequence member(" << it->ref_ << ") is not found.";
	throw fatal_error(__FILE__, __LINE__, ostm.rdbuf()->str().c_str());
      }

      membertype.name_ = it->ref_.get_localname();
      membertype.type_ = type_registory_.get_element(it->ref_);
      cls->push_membertype(membertype, false);
    }
    else {
      if (it->complexTypes_.empty())       // Anonymous complexType must specify the content model.
	throw fatal_error(__FILE__, __LINE__, "ComplexType declaration is invalid."); 
      generate_complexType(*it->complexTypes_.begin(), it->name_);
    }
  }
}

void generator::generate_choice(const xsd::Choice &cho, const akaxiso::name &name) {

  // Create class_contents. 
  choice_class *cls = new choice_class;
  contents_.add(name, cls);

  typedef akaxiso::const_elementlist_impl<xsd::Element> items_type;
  items_type items;
  cho.get_typesafe_elementlist(items, "xsd:element");
  
  for (items_type::iterator it = items.begin(); it != items.end(); ++it) {
    
    choice_item itemtype;
    itemtype.minOccurs_ = it->minOccurs_;
    itemtype.maxOccurs_ = it->maxOccurs_;

    /** name and type specifies the content model  */
    /* <xsd:element name="hoge" type="hogetype"/> */
    if (!it->name_.empty() && !it->type_.empty()) {
      if (!it->complexTypes_.empty() || !it->ref_.empty()) // complexType or ref will specify the model.
                                                           // Both of them should not be empty at the same time. 
	throw fatal_error(__FILE__, __LINE__, "Neither complexType nor ref not specified");

      // complexType and simpleType is acceptable.
      if (!type_registory_.type_exists(it->type_)) {
	std::ostringstream ostm;
	ostm << "name of type (" << it->type_ << ") not found.";
	throw fatal_error(__FILE__, __LINE__, ostm.rdbuf()->str().c_str());
      }

      itemtype.name_ = it->name_.get_localname();

      // Thanks Mr. chy to find a bug.  Fixed here.
      if (type_registory_.simpleType_exists(it->type_)) {
	std::string type = resolve_typename(it->type_);
	generate_simpletype_wrapper(it->name_, type);
	type_registory_.insert_complexType(it->name_, it->name_.get_localname());
      }
      itemtype.type_ = resolve_typename(it->name_);

      cls->push_itemtype(itemtype);
    }
    else if (!it->ref_.empty()) { // <xsd:element ref="toplevel_element">
      if (!it->complexTypes_.empty()) // complexType should not appear.
	throw fatal_error(__FILE__, __LINE__, "Both xsd:complexType and xsd:element@ref attributes are specified."); 

      if (!type_registory_.element_exists(it->ref_)) {
	std::ostringstream ostm;
	ostm << "Type of sequence member(" << it->ref_ << ") is not found.";
	throw fatal_error(__FILE__, __LINE__, ostm.rdbuf()->str().c_str());
      }

      itemtype.name_ = it->ref_.get_localname();
      itemtype.type_ = type_registory_.get_element(it->ref_);
      cls->push_itemtype(itemtype);
    }
    else {
      if (it->complexTypes_.empty())       // Anonymous complexType must specify the content model.
	throw fatal_error(__FILE__, __LINE__, "ComplexType declaration is invalid."); 
      generate_complexType(*it->complexTypes_.begin(), it->name_);
    }
  }
  
}



void generator::generate_simpletype_wrapper(const akaxiso::name &element_name, const std::string &cpptype) {

  simplecontent_class *cls = new simplecontent_class;
  contents_.add(element_name, cls);
  cls->value_type_ = cpptype;
  
}
