#include <config.h>

#include <akaxiso/akaxiso.h>
#include <akaxiso/xerces/parser.h>
#include <akaxiso/yggdrasil/ygg_parser.h>

#ifdef USE_XERCESC
#include <xercesc/util/PlatformUtils.hpp>
#endif

#include <iomanip>

#include "project.h"



static void usage() {
  std::cout << "Parse sample (" << PACKAGE << "-" << VERSION << ")" << std::endl
	    << " options" << std::endl
	    << "  --yggxml : Parsed with Yggdrasil-XML." << std::endl
	    << "  --yggsox : Parsed with Yggdrasil-SOX." << std::endl
	    << "  --xerces : Parsed with Xerces-C++-XML." << std::endl
	    << "  --help(-h) : Show this message." << std::endl;
}



int main(int argc, const char* argv[]){
  
  // Initialize Xerces.
#ifdef USE_XERCESC
  XMLPlatformUtils::Initialize();
#endif
  akaxiso::initialize();
  // Initialize Project classes defined in project.h.
  pj::initialize();
  
  // Parse
  akaxiso::parser *parser = NULL;

  if (argc == 2) {
    std::string arg(argv[1]);
    if (arg == "--yggsox")
      parser = new akaxiso::yggsox_parser;
    else if (arg == "--yggxml")
      parser = new akaxiso::yggxml_parser;
#ifdef USE_XERCESC
    else if (arg == "--xerces")
      parser = new akaxiso::xerces_parser;
#endif
    else if ((arg == "--help") || (arg == "-h")) {
      usage();
      exit(1);
    }
    else {
      std::cerr << "Unknown option(" << arg << ")" << std::endl;
      usage();
      exit(1);
    }
  }      

  if (parser == NULL)
    parser = new akaxiso::yggxml_parser; 

  akaxiso::document *doc = NULL;
  try {
    doc = parser->parse_stdin();
  }
  catch ( const std::exception &e) {
    std::cerr << e.what() << std::endl;
    exit(1);
  }
   catch ( ... ) {
     std::cerr << "Parse error detected." << std::endl;
     exit(1);
   }
    
  pj::Project *project = static_cast<pj::Project*>(doc);

  // Get the root node.
  pj::Team  *team = project->get_root();
  pj::Members &members = team->members_;


  // Get a list of 'person' element by get_typesafe_elementlist().
  typedef akaxiso::elementlist_impl<pj::Member> MemberList;
  MemberList memberlist;
  members.get_typesafe_elementlist(memberlist, "cuppa:member");
  
  std::cout << "Member list of '" << team->name_ << "'(" << team->url_ << ')' << std::endl;

  // Show values.
  // Iterator of PersonList behaves like a container for pj::Member&. 
  for (MemberList::iterator it = memberlist.begin();
       it != memberlist.end(); ++it) 
    std::cout << "name : " << std::setw(20) << std::left << it->name_
              << "Title : " << it->title_ << std::endl;
        
  delete project;

  // Uninitializers.
  akaxiso::uninitialize();
  pj::uninitialize();

#ifdef USE_XERCESC
  XMLPlatformUtils::Terminate();
#endif

  return 0;
}
