// ListBox.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _LISTBOX_H_
#define _LISTBOX_H_

#include "Window.h"


// CListBox class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CListBox : public CWindow {
	// \bh
public:
	bool	Create(bool bSubClass, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id);
	bool	CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id);
	int		InitStorage(int nItems, UINT nBytes);

	int		GetCount() const;
	int		GetHorizontalExtent() const;
	void	SetHorizontalExtent(int cxExtent);
	int		GetTopIndex() const;
	int		SetTopIndex(int nIndex);
	DWORD	GetItemData(int nIndex) const;
	void*	GetItemDataPtr(int nIndex) const;
	int		SetItemData(int nIndex, DWORD dwItemData);
	int		SetItemDataPtr(int nIndex, void* pData);
	int		GetItemRect(int nIndex, LPRECT lpRect) const;
	UINT	ItemFromPoint(POINT pt, bool& bOutSide) const;
	int		SetItemHeight(int nIndex, UINT cyItemHeight);
	int		GetItemHeight(int nIndex) const;
	int		GetSel(int nIndex) const;
	int		GetText(int nIndex, LPTSTR lpszBuffer) const;
	int		GetTextLen(int nIndex) const;
	void	SetColumnWidth(int cxWidth);
	void	SetTabStops();
	bool	SetTabStops(const int& cxEachStop);
	bool	SetTabStops(int nTabStops, LPINT rgTabStops);
	LCID	GetLocale() const;
	LCID	SetLocale(LCID nNewLocale);

	int		GetCurSel() const;
	int		SetCurSel(int nSelect);

	int		SetSel(int nIndex, bool bSelect = true);
	int		GetCaretIndex() const;
	int		SetCaretIndex(int nIndex, bool bScroll = true);
	int		GetSelCount() const;
	int		GetSelItems(int nMaxItems, LPINT rgIndex) const;
	int		SelItemRange(int nFirstItem, int nLastItem, bool bSelect = true);
	void	SetAnchorIndex(int nIndex);
	int		GetAnchorIndex() const;

	int		AddString(LPCTSTR lpszItem);
	int		DeleteString(UINT nIndex);
	int		InsertString(int nIndex, LPCTSTR lpszItem);
	void	ResetContent();
	int		Dir(UINT attr, LPCTSTR lpszWildCard);
	int		FindString(int nStartAfter, LPCTSTR lpszItem) const;
	int		FindStringExact(int nStartAfter, LPCTSTR lpszItem) const;
	int		SelectString(int nStartAfter, LPCTSTR lpszItem);

	// I[o[Ch
	bool	Attach(HWND hWnd, bool bSubClass = false);
protected:
	static LRESULT CALLBACK	WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
};


// CListBox class implementation
/////////////////////////////////////////////////////////////////////////////

inline bool CListBox::Attach(HWND hWnd, bool bSubClass /* = false */) {
	AssertValid();

	if(IsWindow() || !::IsWindow(hWnd) || m_OldProc != 0)
		return false;
	m_hWnd = hWnd;

	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CListBox::WndProc)));
		assert(m_OldProc != CListBox::WndProc);
		m_lOldUserData = GetWindowLong(GWL_USERDATA);
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}
	return true;
}

inline bool CListBox::Create(bool bSubClass, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindow(
		_T("LISTBOX"), _T(""), dwStyle,
		rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
		hwndParent, reinterpret_cast<HMENU>(id),
		reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc =
			reinterpret_cast<WNDPROC>(SetWindowLong(GWL_WNDPROC, reinterpret_cast<LPARAM>(CListBox::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}
	return true;
}

inline bool CListBox::CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindowEx(
		dwExStyle, _T("LISTBOX"), _T(""), dwStyle,
		rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
		hwndParent, reinterpret_cast<HMENU>(id),
		reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc =
			reinterpret_cast<WNDPROC>(SetWindowLong(GWL_WNDPROC, reinterpret_cast<LPARAM>(CListBox::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}
	return true;
}

inline int CListBox::InitStorage(int nItems, UINT nBytes) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_INITSTORAGE, nItems, static_cast<LPARAM>(nBytes)));
}

inline int CListBox::GetCount() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETCOUNT, 0, 0L));
}

inline int CListBox::GetHorizontalExtent() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETHORIZONTALEXTENT, 0, 0L));
}

inline void CListBox::SetHorizontalExtent(int cxExtent) {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, LB_SETHORIZONTALEXTENT, cxExtent, 0L);
}

inline int CListBox::GetTopIndex() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETTOPINDEX, 0, 0L));
}

inline int CListBox::SetTopIndex(int nIndex) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETTOPINDEX, nIndex, 0L));
}

inline DWORD CListBox::GetItemData(int nIndex) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETITEMDATA, nIndex, 0L));
}

inline void* CListBox::GetItemDataPtr(int nIndex) const {
	AssertValidAsWindow();
	return reinterpret_cast<void*>(::SendMessage(m_hWnd, LB_GETITEMDATA, nIndex, 0L));
}

inline int CListBox::SetItemData(int nIndex, DWORD dwItemData) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETITEMDATA, nIndex, static_cast<LPARAM>(dwItemData)));
}

inline int CListBox::SetItemDataPtr(int nIndex, void* pData) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETITEMDATA, nIndex, reinterpret_cast<LPARAM>(pData)));
}

inline int CListBox::GetItemRect(int nIndex, LPRECT lpRect) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETITEMRECT, nIndex, reinterpret_cast<LPARAM>(lpRect)));
}

inline UINT CListBox::ItemFromPoint(POINT pt, bool& bOutSide) const {
	AssertValidAsWindow();

	UINT nReturn;

	nReturn = static_cast<UINT>(::SendMessage(m_hWnd, LB_ITEMFROMPOINT, 0, MAKELPARAM(pt.x, pt.y)));
	bOutSide = toBoolean(HIWORD(nReturn));

	return nReturn;
}

inline int CListBox::SetItemHeight(int nIndex, UINT cyItemHeight) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETITEMHEIGHT, nIndex, static_cast<LPARAM>(cyItemHeight)));
}

inline int CListBox::GetItemHeight(int nIndex) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETITEMHEIGHT, nIndex, 0L));
}

inline int CListBox::GetSel(int nIndex) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETSEL, nIndex, 0L));
}

inline int CListBox::GetText(int nIndex, LPTSTR lpszBuffer) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETTEXT, nIndex, reinterpret_cast<LPARAM>(lpszBuffer)));
}

inline int CListBox::GetTextLen(int nIndex) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETTEXTLEN, nIndex, 0L));
}

inline void CListBox::SetColumnWidth(int cxWidth) {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, LB_SETCOLUMNWIDTH, cxWidth, 0L);
}

inline void CListBox::SetTabStops() {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, LB_GETSEL, 0, 0);
}

inline bool CListBox::SetTabStops(const int& cxEachTabStop) {
	AssertValidAsWindow();

	int rgTabStops[1];

	rgTabStops[0] = cxEachTabStop;
	return toBoolean(::SendMessage(m_hWnd, LB_GETSEL, 1, reinterpret_cast<LPARAM>(rgTabStops)));
}

inline bool CListBox::SetTabStops(int nTabStops, LPINT rgTabStops) {
	AssertValidAsWindow();
	return toBoolean(::SendMessage(m_hWnd, LB_GETSEL, nTabStops, reinterpret_cast<LPARAM>(rgTabStops)));
}

inline LCID CListBox::GetLocale() const {
	AssertValidAsWindow();
	return static_cast<LCID>(::SendMessage(m_hWnd, LB_GETLOCALE, 0, 0L));
}

inline LCID CListBox::SetLocale(LCID nNewLocale) {
	AssertValidAsWindow();
	return static_cast<LCID>(::SendMessage(m_hWnd, LB_SETLOCALE, nNewLocale, 0L));
}

inline int CListBox::GetCurSel() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETCURSEL, 0, 0L));
}

inline int CListBox::SetCurSel(int nSelect) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETCURSEL, nSelect, 0L));
}

inline int CListBox::SetSel(int nIndex, bool bSelect /* = true */) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETSEL, bSelect, nIndex));
}

inline int CListBox::GetCaretIndex() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETCARETINDEX, 0, 0L));
}

inline int CListBox::SetCaretIndex(int nIndex, bool bScroll /* = true */) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_SETCARETINDEX, nIndex, bScroll));
}

inline int CListBox::GetSelCount() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETSELCOUNT, 0, 0L));
}

inline int CListBox::GetSelItems(int nMaxItems, LPINT rgIndex) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETCURSEL, nMaxItems, reinterpret_cast<LPARAM>(rgIndex)));
}

inline int CListBox::SelItemRange(int nFirstItem, int nLastItem, bool bSelect /* = true */) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETCURSEL, bSelect, MAKELPARAM(nFirstItem, nLastItem)));
}

inline void CListBox::SetAnchorIndex(int nIndex) {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, LB_SETANCHORINDEX, nIndex, 0L);
}

inline int CListBox::GetAnchorIndex() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_GETANCHORINDEX, 0, 0L));
}

inline int CListBox::AddString(LPCTSTR lpszItem) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_ADDSTRING, 0, reinterpret_cast<LPARAM>(lpszItem)));
}

inline int CListBox::DeleteString(UINT nIndex) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_DELETESTRING, nIndex, 0L));
}

inline int CListBox::InsertString(int nIndex, LPCTSTR lpszItem) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_INSERTSTRING, nIndex, reinterpret_cast<LPARAM>(lpszItem)));
}

inline void CListBox::ResetContent() {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, LB_RESETCONTENT, 0, 0L);
}

inline int CListBox::Dir(UINT attr, LPCTSTR lpszWildCard) {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LB_DIR, attr, reinterpret_cast<LPARAM>(lpszWildCard)));
}

inline int CListBox::FindString(int nStartAfter, LPCTSTR lpszItem) const {
	AssertValidAsWindow();
	return static_cast<int>
		(::SendMessage(m_hWnd, LB_FINDSTRING, nStartAfter, reinterpret_cast<LPARAM>(lpszItem)));
}

inline int CListBox::FindStringExact(int nStartAfter, LPCTSTR lpszItem) const {
	AssertValidAsWindow();
	return static_cast<int>
		(::SendMessage(m_hWnd, LB_FINDSTRINGEXACT, nStartAfter, reinterpret_cast<LPARAM>(lpszItem)));
}

inline int CListBox::SelectString(int nStartAfter, LPCTSTR lpszItem) {
	AssertValidAsWindow();
	return static_cast<int>
		(::SendMessage(m_hWnd, LB_SELECTSTRING, nStartAfter, reinterpret_cast<LPARAM>(lpszItem)));
}

inline LRESULT CALLBACK CListBox::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	CListBox*	pListBox = 0;

	if(message == WM_MATTACH) {
		pListBox = reinterpret_cast<CListBox*>(lParam);
		pListBox->SetWindowLong(GWL_USERDATA, lParam);
		assert(hWnd == pListBox->m_hWnd);
		return pListBox->DispatchEvent(WM_MATTACH, 0, 0L);
	} else {
		pListBox = reinterpret_cast<CListBox*>(::GetWindowLong(hWnd, GWL_USERDATA));
		return (pListBox != 0) ? pListBox->DispatchEvent(message, wParam, lParam) : false;
	}
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _LIST_BOX_H_ */

/* [EOF] */