// ListCtrl.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _LIST_CTRL_H_
#define _LIST_CTRL_H_

#include "Window.h"
#include "Edit.h"
#include "ImageList.h"


// CListCtrl class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CListCtrl : public CWindow {
	// \bh
public:
	/* 쐬 */
	bool		Create(bool bSubClass, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id);
	bool		CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id);

	/*  */
	COLORREF	GetBkColor() const;
	bool		SetBkColor(COLORREF clr);
	CImageList*	GetImageList(int nImageListType) const;
	CImageList*	SetImageList(CImageList* pImageList, int nImageListType);
	int			GetItemCount() const;
	bool		GetItem(LVITEM* pItem) const;
	bool		SetItem(const LVITEM* pItem);
	bool		SetItem(int iItem, int iSubItem, UINT nMask, const TCHAR* lpszItem, int iImage, UINT nState, UINT nStateMask, LPARAM lParam);
	UINT		GetCallbackMask() const;
	bool		SetCallbackMask(UINT nMask);
	int			GetNextItem(int iItem, int nFlag) const;
	bool		GetItemRect(int iItem, LPRECT lpRect, UINT nCode) const;
	bool		GetItemPosition(int iItem, LPPOINT lpPoint) const;
	bool		SetItemPosition(int iItem, POINT pt);
	int			GetStringWidth(const TCHAR* lpszText) const;
	CEdit*		GetEditControl() const;
	bool		GetColumn(int iCol, LVCOLUMN* pColumn) const;
	bool		SetColumn(int iCol, const LVCOLUMN* pColumn);
	int			GetColumnWidth(int iCol) const;
	bool		SetColumnWidth(int iCol, int cx);
	bool		GetCheck(int iItem) const;
	bool		SetCheck(int iItem, bool fCheck = true);
	bool		GetViewRect(LPRECT lpRect) const;
	COLORREF	GetTextColor() const;
	bool		SetTextColor(COLORREF clr);
	COLORREF	GetTextBkColor() const;
	bool		SetTextBkColor(COLORREF clr);
	int			GetTopIndex() const;
	int			GetCountPerPage() const;
	bool		GetOrigin(LPPOINT lpPoint) const;
	UINT		GetItemState(int iItem, UINT nMask) const;
	bool		SetItemState(int iItem, LVITEM* pItem);
	bool		SetItemState(int iItem, UINT nState, UINT nMask);
	int			GetItemText(int iItem, int iSubItem, TCHAR* lpszText, int nLength) const;
	bool		SetItemText(int iItem, int iSubItem, const TCHAR* lpszText);
	void		SetItemCount(int nItems);
	void		SetItemCountEx(int iCount, DWORD dwFlags = LVSICF_NOINVALIDATEALL);
	DWORD		GetItemData(int iItem) const;
	bool		SetItemData(int iItem, DWORD dwData);
	UINT		GetSelectedCount() const;
	bool		GetColumnOrderArray(LPINT piArray, int iCount = -1) const;
	bool		SetColumnOrderArray(int iCount, LPINT piArray);
	SIZE		SetIconSpacing(int cx, int cy);
	SIZE		SetIconSpacing(SIZE size);
	HWND		GetHeaderControl() const;
	HCURSOR		GetHotCursor() const;
	HCURSOR		SetHotCursor(HCURSOR hc);
	bool		GetSubItemRect(int iItem, int iSubItem, int nArea, LPRECT lpRect) const;
	int			GetHotItem() const;
	int			SetHotItem(int iIndex);
	int			GetSelectionMark() const;
	int			SetSelectionMark(int iIndex);
	DWORD		GetExtendedStyle() const;
	DWORD		SetExtendedStyle(DWORD dwNewStyle);
	DWORD		SetExtendedStyleEx(DWORD dwExMask, DWORD dwExStyle);
	int			SubItemHitTest(LPLVHITTESTINFO pInfo);
	void		GetWorkAreas(int nWorkAreas, LPRECT lpRect) const;
	void		SetWorkAreas(int nWorkAreas, const LPRECT lpRect);
	SIZE		ApproximateViewRect(SIZE size, int iCount = -1) const;
	bool		GetBkImage(LVBKIMAGE* pImage) const;
	bool		SetBkImage(LVBKIMAGE* pImage);
	bool		SetBkImage(HBITMAP hBMP, bool fTile = true, int xOffsetPercent = 0, int yOffsetPercent = 0);
	bool		SetBkImage(const TCHAR* lpszURL, bool fTile = true, int xOffsetPercent = 0, int yOffsetPercent = 0);
	DWORD		GetHoverTime() const;
	DWORD		SetHoverTime(DWORD dwHoverTime = static_cast<DWORD>(-1));

	/*  */
	int			InsertItem(const LVITEM* pItem);
	int			InsertItem(int iItem, const TCHAR* lpszText);
	int			InsertItem(int iItem, const TCHAR* lpszText, int iImage);
	int			InsertItem(UINT nMask, int iItem, const TCHAR* lpszText, UINT nState, UINT nStateMask, int iImage, LPARAM lParam);
	bool		DeleteItem(int iItem);
	bool		DeleteAllItems();
	int			FindItem(LVFINDINFO* pFindInfo, int iStart = -1) const;
	bool		SortItems(PFNLVCOMPARE pfnCompare, DWORD dwData);
	int			HitTest(LVHITTESTINFO* pHitTestInfo) const;
	int			HitTest(POINT pt, UINT* pFlags = 0) const;
	bool		EnsureVisible(int iItem, bool bPartialOK);
	bool		Scroll(SIZE size);
	bool		RedrawItems(int iFirst, int iLast);
	bool		Update(int iItem);
	bool		Arrange(UINT nCode);
	CEdit*		EditLabel(int iItem);
	int			InsertColumn(int iCol, LVCOLUMN* pColumn);
	int			InsertColumn(int iCol, const TCHAR* lpszColumnHeading, int nFormat = LVCFMT_LEFT, int nWidth = -1, int iSubItem = -1);
	bool		DeleteColumn(int iCol);
	CImageList*	CreateDragImage(int iItem, LPPOINT lpPoint);	// returned pointer is merpanent

	/* I[o[Ch */
	bool		Attach(HWND hWnd, bool bSubClass = false);
protected:
	LRESULT					DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);
	static LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
};


// CListCtrl class implementation
/////////////////////////////////////////////////////////////////////////////

inline SIZE CListCtrl::ApproximateViewRect(SIZE size, int iCount /* = -1 */) const {
	AssertValidAsWindow();
	
	DWORD dwSize;
	dwSize = static_cast<DWORD>(::SendMessage(m_hWnd, LVM_APPROXIMATEVIEWRECT, iCount, MAKELONG(size.cx, size.cy)));
	size.cx = LOWORD(dwSize);
	size.cy = HIWORD(dwSize);

	return size;
}

inline bool CListCtrl::Arrange(UINT nCode) {
	return (SendMessage(LVM_ARRANGE, nCode) != 0);
}

inline bool CListCtrl::Attach(HWND hWnd, bool bSubClass /* = false */) {
	AssertValid();

	if(IsWindow() || !::IsWindow(hWnd) || m_OldProc != 0)
		return false;
	m_hWnd = hWnd;

	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CListCtrl::WndProc)));
		assert(m_OldProc != CListCtrl::WndProc);
		m_lOldUserData = GetWindowLong(GWL_USERDATA);
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CListCtrl::Create(bool bSubClass, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindow(WC_LISTVIEW, _T(""),
		dwStyle, lpRect->left, lpRect->top, lpRect->right - lpRect->left, lpRect->bottom - lpRect->top,
		hwndParent, reinterpret_cast<HMENU>(id), reinterpret_cast<HINSTANCE>(::GetModuleHandle(0)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CListCtrl::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline Manah::Windows::CImageList* CListCtrl::CreateDragImage(int iItem, LPPOINT lpPoint) {
	return CImageList::FromHandlePermanent(reinterpret_cast<HIMAGELIST>(
		SendMessage(LVM_CREATEDRAGIMAGE, iItem, reinterpret_cast<LPARAM>(lpPoint))));
}

inline bool CListCtrl::CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindowEx(dwExStyle, WC_LISTVIEW, _T(""),
		dwStyle, lpRect->left, lpRect->top, lpRect->right - lpRect->left, lpRect->bottom - lpRect->top,
		hwndParent, reinterpret_cast<HMENU>(id), reinterpret_cast<HINSTANCE>(::GetModuleHandle(0)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CListCtrl::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CListCtrl::DeleteAllItems() {
	return (SendMessage(LVM_DELETEALLITEMS) != 0);
}

inline bool CListCtrl::DeleteColumn(int iCol) {
	return (SendMessage(LVM_DELETECOLUMN, iCol) != 0);
}

inline bool CListCtrl::DeleteItem(int iItem) {
	return (SendMessage(LVM_DELETEITEM, iItem) != 0);
}

inline LRESULT CListCtrl::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	if(PreTranslateMessage(message, wParam, lParam))
		return false;
	return CWindow::DispatchEvent(message, wParam, lParam);
}

inline CEdit* CListCtrl::EditLabel(int iItem) {
	static CEdit	oEdit;
	oEdit.Attach(reinterpret_cast<HWND>(SendMessage(LVM_EDITLABEL, iItem)));
	return &oEdit;
}

inline bool CListCtrl::EnsureVisible(int iItem, bool bPartialOK) {
	return (SendMessage(LVM_ENSUREVISIBLE, iItem, bPartialOK) != 0);
}

inline int CListCtrl::FindItem(LVFINDINFO* pFindInfo, int iStart /* = -1 */) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_FINDITEM, iStart, reinterpret_cast<LPARAM>(pFindInfo)));
}

inline COLORREF CListCtrl::GetBkColor() const {
	AssertValidAsWindow();
	return static_cast<COLORREF>(::SendMessage(m_hWnd, LVM_GETBKCOLOR, 0, 0));
}

inline bool CListCtrl::GetBkImage(LVBKIMAGE* pImage) const {
	AssertValidAsWindow();
	if(pImage == 0)
		return false;
	return (::SendMessage(m_hWnd, LVM_GETBKIMAGE, 0, reinterpret_cast<LPARAM>(pImage)) != 0);
}

inline UINT CListCtrl::GetCallbackMask() const {
	AssertValidAsWindow();
	return static_cast<UINT>(::SendMessage(m_hWnd, LVM_GETCALLBACKMASK, 0, 0L));
}

inline bool CListCtrl::GetCheck(int iItem) const {
	AssertValidAsWindow();
	return (((static_cast<UINT>(::SendMessage(m_hWnd, LVM_GETITEMSTATE, iItem, LVIS_STATEIMAGEMASK)) >> 12) - 1) != 0);
}

inline bool CListCtrl::GetColumn(int iCol, LVCOLUMN* pColumn) const {
	AssertValidAsWindow();
	return (::SendMessage(m_hWnd, LVM_GETCOLUMN, iCol, reinterpret_cast<LPARAM>(pColumn)) != 0);
}

inline bool CListCtrl::GetColumnOrderArray(LPINT piArray, int iCount /* = -1 */) const {
	AssertValidAsWindow();
	if(iCount == -1)
		iCount = sizeof(*piArray) / sizeof(int);
	return (::SendMessage(m_hWnd, LVM_GETCOLUMNORDERARRAY, iCount, reinterpret_cast<LPARAM>(piArray)) != 0);
}

inline int CListCtrl::GetColumnWidth(int iCol) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETCOLUMNWIDTH, iCol, 0L));
}

inline int CListCtrl::GetCountPerPage() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETCOUNTPERPAGE, 0, 0L));
}

inline CEdit* CListCtrl::GetEditControl() const {
	AssertValidAsWindow();
	static CEdit	oEdit;
	HWND			hEdit;
	
	hEdit = reinterpret_cast<HWND>(::SendMessage(m_hWnd, LVM_GETEDITCONTROL, 0, 0L));
	if(hEdit == 0)
		return 0;
	oEdit.Attach(hEdit);
	return &oEdit;
}

inline DWORD CListCtrl::GetExtendedStyle() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::SendMessage(m_hWnd, LVM_GETEXTENDEDLISTVIEWSTYLE, 0, 0L));
}

inline HWND CListCtrl::GetHeaderControl() const {
	AssertValidAsWindow();
	return reinterpret_cast<HWND>(::SendMessage(m_hWnd, LVM_GETHEADER, 0, 0L));
}

inline HCURSOR CListCtrl::GetHotCursor() const {
	AssertValidAsWindow();
	return reinterpret_cast<HCURSOR>(::SendMessage(m_hWnd, LVM_GETHOTCURSOR, 0, 0L));
}

inline int CListCtrl::GetHotItem() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETHOTITEM, 0, 0L));
}

inline DWORD CListCtrl::GetHoverTime() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::SendMessage(m_hWnd, LVM_GETHOVERTIME, 0, 0L));
}

inline Manah::Windows::CImageList* CListCtrl::GetImageList(int nImageListType) const {
	AssertValidAsWindow();
	return CImageList::FromHandle(
		reinterpret_cast<HIMAGELIST>(::SendMessage(m_hWnd, LVM_GETIMAGELIST, nImageListType, 0L)));
}

inline bool CListCtrl::GetItem(LVITEM* pItem) const {
	AssertValidAsWindow();
	return (::SendMessage(m_hWnd, LVM_GETITEM, 0, reinterpret_cast<LPARAM>(pItem)) != 0);
}

inline int CListCtrl::GetItemCount() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETITEMCOUNT, 0, 0L));
}

inline DWORD CListCtrl::GetItemData(int iItem) const {
	AssertValidAsWindow();

	LVITEM	item;
	item.mask = LVIF_PARAM;
	item.iItem = iItem;
	item.iSubItem = 0;
	GetItem(&item);
	return item.lParam;
}

inline bool CListCtrl::GetItemPosition(int iItem, LPPOINT lpPoint) const {
	AssertValidAsWindow();
	if(lpPoint == 0)
		return false;
	return (::SendMessage(m_hWnd, LVM_GETITEMPOSITION, iItem, reinterpret_cast<LPARAM>(lpPoint)) != 0);
}

inline bool CListCtrl::GetItemRect(int iItem, LPRECT lpRect, UINT nCode) const {
	AssertValidAsWindow();
	if(lpRect == 0)
		return false;
	lpRect->left = nCode;
	return (::SendMessage(m_hWnd, LVM_GETITEMRECT, iItem, reinterpret_cast<LPARAM>(lpRect)) != 0);
}

inline UINT CListCtrl::GetItemState(int iItem, UINT nMask) const {
	AssertValidAsWindow();
	return static_cast<UINT>(::SendMessage(m_hWnd, LVM_GETITEMSTATE, iItem, nMask));
}

inline int CListCtrl::GetItemText(int iItem, int iSubItem, TCHAR* lpszText, int nLength) const {
	AssertValidAsWindow();

	LVITEM	item;
	item.iSubItem = iSubItem;
	item.pszText = lpszText;
	item.cchTextMax = nLength;
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETITEMTEXT, iItem, reinterpret_cast<LPARAM>(&item)));
}

inline int CListCtrl::GetNextItem(int iItem, int nFlag) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETNEXTITEM, iItem, nFlag));
}

inline bool CListCtrl::GetOrigin(LPPOINT lpPoint) const {
	AssertValidAsWindow();
	if(lpPoint == 0)
		return false;
	return (::SendMessage(m_hWnd, LVM_GETORIGIN, 0, reinterpret_cast<LPARAM>(lpPoint)) != 0);
}

inline UINT CListCtrl::GetSelectedCount() const {
	AssertValidAsWindow();
	return static_cast<UINT>(::SendMessage(m_hWnd, LVM_GETSELECTEDCOUNT, 0, 0L));
}

inline int CListCtrl::GetSelectionMark() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETSELECTIONMARK, 0, 0L));
}

inline int CListCtrl::GetStringWidth(const TCHAR* lpszText) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETSTRINGWIDTH, 0, reinterpret_cast<LPARAM>(lpszText)));
}

inline bool CListCtrl::GetSubItemRect(int iItem, int iSubItem, int nArea, LPRECT lpRect) const {
	AssertValidAsWindow();
	if(lpRect == 0)
		return false;
	lpRect->left = nArea;
	lpRect->top = iSubItem;
	return (::SendMessage(m_hWnd, LVM_GETSUBITEMRECT, iItem, reinterpret_cast<LPARAM>(lpRect)) != 0);
}

inline COLORREF CListCtrl::GetTextBkColor() const {
	AssertValidAsWindow();
	return static_cast<COLORREF>(::SendMessage(m_hWnd, LVM_GETTEXTBKCOLOR, 0, 0L));
}

inline COLORREF CListCtrl::GetTextColor() const {
	AssertValidAsWindow();
	return static_cast<COLORREF>(::SendMessage(m_hWnd, LVM_GETTEXTCOLOR, 0, 0L));
}

inline int CListCtrl::GetTopIndex() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETTOPINDEX, 0, 0L));
}

inline bool CListCtrl::GetViewRect(LPRECT lpRect) const {
	AssertValidAsWindow();
	if(lpRect == 0)
		return false;
	return (::SendMessage(m_hWnd, LVM_GETVIEWRECT, 0, reinterpret_cast<LPARAM>(lpRect)) != 0);
}

inline void CListCtrl::GetWorkAreas(int nWorkAreas, LPRECT lpRect) const {
	AssertValidAsWindow();
	if(lpRect == 0)
		return;
	::SendMessage(m_hWnd, LVM_GETWORKAREAS, nWorkAreas, reinterpret_cast<LPARAM>(lpRect));
}

inline int CListCtrl::HitTest(LVHITTESTINFO* pHitTestInfo) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_HITTEST, 0, reinterpret_cast<LPARAM>(pHitTestInfo)));
}

inline int CListCtrl::HitTest(POINT pt, UINT* pFlags /* = 0 */) const {
	AssertValidAsWindow();

	LVHITTESTINFO	lvti;
	lvti.pt = pt;
	lvti.flags = (pFlags != 0) ? *pFlags : 0;
	return HitTest(&lvti);
}

inline int CListCtrl::InsertColumn(int iCol, LVCOLUMN* pColumn) {
	return static_cast<int>(SendMessage(LVM_INSERTCOLUMN, iCol, reinterpret_cast<LPARAM>(pColumn)));
}

inline int CListCtrl::InsertColumn(int iCol, const TCHAR* lpszColumnHeading, int nFormat /* = LVCFMT_LEFT */, int nWidth /* = -1 */, int iSubItem /* = -1 */) {
	LVCOLUMN	lvcol = {
		LVCF_FMT | LVCF_TEXT
			| ((iSubItem != -1) ? LVCF_SUBITEM : 0)
			| ((nWidth != -1) ? LVCF_WIDTH : 0),
		nFormat, nWidth, const_cast<TCHAR*>(lpszColumnHeading), _tcslen(lpszColumnHeading), iSubItem
	};
	return InsertColumn(iCol, &lvcol);
}

inline int CListCtrl::InsertItem(const LVITEM* pItem) {
	if(pItem == 0)
		return -1;
	return static_cast<int>(SendMessage(LVM_INSERTITEM, 0, reinterpret_cast<LPARAM>(pItem)));
}

inline int CListCtrl::InsertItem(int iItem, const TCHAR* lpszText) {
	LVITEM	item;

	ZeroMemory(&item, sizeof(LVITEM));
	item.mask = LVIF_TEXT;
	item.iItem = iItem;
	item.pszText = const_cast<TCHAR*>(lpszText);
	item.cchTextMax = _tcslen(lpszText);

	return InsertItem(&item);
}

inline int CListCtrl::InsertItem(int iItem, const TCHAR* lpszText, int iImage) {
	LVITEM	item;

	ZeroMemory(&item, sizeof(LVITEM));
	item.mask = LVIF_TEXT | LVIF_IMAGE;
	item.iItem = iItem;
	item.pszText = const_cast<TCHAR*>(lpszText);
	item.cchTextMax = _tcslen(lpszText);
	item.iImage = iImage;

	return InsertItem(&item);
}

inline int CListCtrl::InsertItem(UINT nMask, int iItem, const TCHAR* lpszText, UINT nState, UINT nStateMask, int iImage, LPARAM lParam) {
	LVITEM	item = {
		nMask, iItem, 0, nState, nStateMask, const_cast<TCHAR*>(lpszText), _tcslen(lpszText), iImage, lParam
	};

	return InsertItem(&item);
}

inline bool CListCtrl::RedrawItems(int iFirst, int iLast) {
	return (SendMessage(LVM_REDRAWITEMS, iFirst, iLast) != 0);
}

inline bool CListCtrl::Scroll(SIZE size) {
	return (SendMessage(LVM_SCROLL, size.cx, size.cy) != 0);
}

inline bool CListCtrl::SetBkColor(COLORREF clr) {
	return (SendMessage(LVM_SETBKCOLOR, 0, clr) != 0);
}

inline bool CListCtrl::SetBkImage(LVBKIMAGE* pImage) {
	return (SendMessage(LVM_SETBKIMAGE, 0, reinterpret_cast<LPARAM>(pImage)) != 0);
}

inline bool CListCtrl::SetBkImage(HBITMAP hBMP, bool fTile /* = true */, int xOffsetPercent /* = 0 */, int yOffsetPercent /* = 0 */) {
	LVBKIMAGE	bkImage = {
		LVBKIF_SOURCE_HBITMAP | ((xOffsetPercent == 0 && yOffsetPercent == 0) ?
			LVBKIF_STYLE_TILE : LVBKIF_STYLE_NORMAL),
		hBMP, 0, 0, xOffsetPercent, yOffsetPercent
	};
	return SetBkImage(&bkImage);
}

inline bool CListCtrl::SetBkImage(const TCHAR* lpszURL, bool fTile /* = true */, int xOffsetPercent /* = 0 */, int yOffsetPercent /* = 0 */) {
	LVBKIMAGE	bkImage = {
		LVBKIF_SOURCE_URL | ((xOffsetPercent == 0 && yOffsetPercent == 0) ?
			LVBKIF_STYLE_TILE : LVBKIF_STYLE_NORMAL),
		0, const_cast<TCHAR*>(lpszURL), _tcslen(lpszURL), xOffsetPercent, yOffsetPercent
	};
	return SetBkImage(&bkImage);
}

inline bool CListCtrl::SetCallbackMask(UINT nMask) {
	return (SendMessage(LVM_SETCALLBACKMASK, nMask, 0L) != 0);
}

inline bool CListCtrl::SetCheck(int iItem, bool fCheck /* = true */) {
	LVITEM	item = {0, iItem, 0, INDEXTOSTATEIMAGEMASK(fCheck - 1), LVIS_STATEIMAGEMASK, 0, 0, 0, 0};
	return (SendMessage(LVM_SETITEMSTATE, iItem, reinterpret_cast<LPARAM>(&item)) != 0);
}

inline bool CListCtrl::SetColumn(int iCol, const LVCOLUMN* pColumn) {
	return (SendMessage(LVM_SETCOLUMN, iCol, reinterpret_cast<LPARAM>(pColumn)) != 0);
}

inline bool CListCtrl::SetColumnOrderArray(int iCount, LPINT piArray) {
	return (SendMessage(LVM_SETCOLUMNORDERARRAY, iCount, reinterpret_cast<LPARAM>(piArray)) != 0);
}

inline bool CListCtrl::SetColumnWidth(int iCol, int cx) {
	return (SendMessage(LVM_SETCOLUMNWIDTH, iCol, cx) != 0);
}

inline DWORD CListCtrl::SetExtendedStyle(DWORD dwNewStyle) {
	return static_cast<DWORD>(SendMessage(LVM_SETEXTENDEDLISTVIEWSTYLE, 0, dwNewStyle));
}

inline DWORD CListCtrl::SetExtendedStyleEx(DWORD dwExMask, DWORD dwExStyle) {
	return static_cast<DWORD>(SendMessage(LVM_SETEXTENDEDLISTVIEWSTYLE, dwExMask, dwExStyle));
}

inline HCURSOR CListCtrl::SetHotCursor(HCURSOR hc) {
	return reinterpret_cast<HCURSOR>(SendMessage(LVM_SETHOTCURSOR, 0, reinterpret_cast<LPARAM>(hc)));
}

inline int CListCtrl::SetHotItem(int iIndex) {
	return static_cast<int>(SendMessage(LVM_SETHOTITEM, iIndex));
}

inline DWORD CListCtrl::SetHoverTime(DWORD dwHoverTime /* = static_cast<DWORD>(-1) */) {
	return static_cast<DWORD>(SendMessage(LVM_SETHOVERTIME, 0, dwHoverTime));
}

inline SIZE CListCtrl::SetIconSpacing(int cx, int cy) {
	SIZE	size;
	DWORD	dwSize;

	dwSize = SendMessage(LVM_SETICONSPACING, 0, MAKELONG(cx, cy));
	size.cx = LOWORD(dwSize);
	size.cy = HIWORD(dwSize);
	return size;
}

inline SIZE CListCtrl::SetIconSpacing(SIZE size) {
	DWORD	dwSize;

	dwSize = SendMessage(LVM_SETICONSPACING, 0, MAKELONG(size.cx, size.cy));
	size.cx = LOWORD(dwSize);
	size.cy = HIWORD(dwSize);
	return size;
}

inline Manah::Windows::CImageList* CListCtrl::SetImageList(Manah::Windows::CImageList* pImageList, int nImageListType) {
	return CImageList::FromHandle(reinterpret_cast<HIMAGELIST>(
		SendMessage(LVM_SETIMAGELIST, nImageListType, reinterpret_cast<LPARAM>(pImageList->GetSafeHandle()))));
}

inline bool CListCtrl::SetItem(const LVITEM* pItem) {
	return (SendMessage(LVM_SETITEM, 0, reinterpret_cast<LPARAM>(pItem)) != 0);
}

inline void CListCtrl::SetItemCount(int nItems) {
	SendMessage(LVM_SETITEMCOUNT, nItems, 0L);
}

inline void CListCtrl::SetItemCountEx(int iCount, DWORD dwFlags /* = LVSICF_NOINVALIDATEALL */) {
	SendMessage(LVM_SETITEMCOUNT, iCount, dwFlags);
}

inline bool CListCtrl::SetItemData(int iItem, DWORD dwData) {
	LVITEM	item;
	item.mask = LVIF_PARAM;
	item.iItem = iItem;
	item.iSubItem = 0;
	item.lParam = dwData;
	return SetItem(&item);
}

inline bool CListCtrl::SetItem(int iItem, int iSubItem, UINT nMask, const TCHAR* lpszItem, int iImage, UINT nState, UINT nStateMask, LPARAM lParam) {
	LVITEM	item = {nMask, iItem, iSubItem, nState, nStateMask, const_cast<TCHAR*>(lpszItem), _tcslen(lpszItem), iImage};
	return (SendMessage(LVM_SETITEM, 0, reinterpret_cast<LPARAM>(&item)) != 0);
}

inline bool CListCtrl::SetItemPosition(int iItem, POINT pt) {
	return (SendMessage(LVM_SETITEMPOSITION32, iItem, MAKELPARAM(pt.x, pt.y)) != 0);
}

inline bool CListCtrl::SetItemState(int iItem, LVITEM* pItem) {
	return (SendMessage(LVM_SETITEMSTATE, iItem, reinterpret_cast<LPARAM>(pItem)) != 0);
}

inline bool CListCtrl::SetItemText(int iItem, int iSubItem, const TCHAR* lpszText) {
	LVITEM	item;
	item.iSubItem = iSubItem;
	item.pszText = const_cast<TCHAR*>(lpszText);
	return (SendMessage(LVM_SETITEMTEXT, iItem, reinterpret_cast<LPARAM>(&item)) != 0);
}

inline int CListCtrl::SetSelectionMark(int iIndex) {
	return static_cast<int>(SendMessage(LVM_SETSELECTIONMARK, 0, iIndex));
}

inline bool CListCtrl::SetItemState(int iItem, UINT nState, UINT nMask) {
	LVITEM	item;
	item.state = nState;
	item.stateMask = nMask;
	return (SendMessage(LVM_SETITEMSTATE, iItem, reinterpret_cast<LPARAM>(&item)) != 0);
}

inline bool CListCtrl::SetTextBkColor(COLORREF clr) {
	return (SendMessage(LVM_SETTEXTBKCOLOR, 0, static_cast<LPARAM>(clr)) != 0);
}

inline bool CListCtrl::SetTextColor(COLORREF clr) {
	return (SendMessage(LVM_SETTEXTCOLOR, 0, static_cast<LPARAM>(clr)) != 0);
}

inline void CListCtrl::SetWorkAreas(int nWorkAreas, const LPRECT lpRect) {
	SendMessage(LVM_SETWORKAREAS, nWorkAreas, reinterpret_cast<LPARAM>(lpRect));
}

inline bool CListCtrl::SortItems(PFNLVCOMPARE pfnCompare, DWORD dwData) {
	return (SendMessage(LVM_SORTITEMS, dwData, reinterpret_cast<LPARAM>(pfnCompare)) != 0);
}

inline int CListCtrl::SubItemHitTest(LPLVHITTESTINFO pInfo) {
	return static_cast<int>(SendMessage(LVM_SUBITEMHITTEST, 0, reinterpret_cast<LPARAM>(pInfo)));
}

inline bool CListCtrl::Update(int iItem) {
	return (SendMessage(LVM_UPDATE, iItem) != 0);
}

inline LRESULT CALLBACK CListCtrl::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	CListCtrl*	pListCtrl = 0;

	if(message == WM_MATTACH) {
		pListCtrl = reinterpret_cast<CListCtrl*>(lParam);
		pListCtrl->SetWindowLong(GWL_USERDATA, lParam);
		assert(hWnd == pListCtrl->m_hWnd);
		return pListCtrl->DispatchEvent(WM_MATTACH, 0, 0L);
	} else {
		pListCtrl = reinterpret_cast<CListCtrl*>(::GetWindowLong(hWnd, GWL_USERDATA));
		return (pListCtrl != 0 && pListCtrl->m_hWnd == hWnd) ?
			pListCtrl->DispatchEvent(message, wParam, lParam) : ::DefWindowProc(hWnd, message, wParam, lParam);
	}
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _LIST_CTRL_H_ */

/* [EOF] */