// Menu.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _MENU_H_
#define _MENU_H_

#pragma warning(disable : 4297)

#include "..\Manah\Object.h"
#include <commctrl.h>
#include <set>
#include <stdexcept>
using namespace std;


namespace Manah {
namespace Windows {
namespace Controls {


// CMenu class definition
/////////////////////////////////////////////////////////////////////////////

class CMenu : public CObject {
	// RXgN^
public:
	CMenu();
	CMenu(HMENU hMenu);
	virtual ~CMenu();

	// \bh
public:
	bool			AppendMenuItem(UINT nID, UINT nType, const TCHAR* lpszCaption = 0);
	bool			AppendMenuItem(UINT nID, UINT nType, UINT nState, const TCHAR* lpszCaption = 0);
	DWORD			CheckMenuItem(UINT nID, bool bCheck = true);
	bool			CheckMenuRadioItem(UINT nFirstID, UINT nLastID, UINT nItemID);
	bool			DeleteMenuItem(UINT iItem, bool bByCommand = true);
	bool			EnableMenuItem(UINT nID, bool bEnable = true);
	LRESULT			ExecuteMenuChar(TCHAR chCode, UINT nFlag);
	UINT			GetDefaultMenuItem(UINT gmdiFlags) const;
	UINT			GetItemCount() const;
	bool			GetMenuItemCaption(UINT iItem, TCHAR* lpsz, int cchMax, bool bByCommand = true) const;
	UINT			GetMenuItemID(int iItem) const;
	bool			GetMenuItemInfo(UINT iItem, MENUITEMINFO* pInfo, bool bByCommand = true) const;
	UINT			GetMenuItemState(UINT iItem, bool bByCommand = true) const;
	HMENU			GetSafeHmenu() const;
	CMenu*			GetSubMenu(UINT iItem) const;
	bool			InsertMenuItem(UINT nID, UINT nPrevID, UINT nType, UINT nState, const TCHAR* lpszCaption);
	static CMenu*	Load(HINSTANCE hInstance, const TCHAR* lpszName);
	static CMenu*	Load(const MENUTEMPLATE* pTemplate);
	bool			ModifyMenuItem(UINT nID, UINT nFlags, const TCHAR* lpszPrompt);
	bool			RemoveMenuItem(UINT iItem, bool bByCommand = true);
	bool			SetChildPopup(const CMenu* pPopup, UINT iItem, bool bDelegateOwnership = true, bool bByCommand = false);
	bool			SetDefaultMenuItem(UINT iItem, bool bByCommand = true);
	bool			TrackPopupMenu(UINT nFlags, int x, int y, HWND hWnd, const RECT* prcRect = 0);

	// f[^o
private:
	HMENU					m_hMenu;	// handle for popup menu
	std::set<const CMenu*>	m_children;	// for deletion
	bool					m_bManage;	// if contained popup menu was created/will be destroyed by this object
};


// CMenu class implementation
/////////////////////////////////////////////////////////////////////////////

inline CMenu::CMenu() : m_hMenu(::CreatePopupMenu()), m_bManage(true) {
}

inline CMenu::CMenu(HMENU hMenu) : m_hMenu(hMenu), m_bManage(false) {
	if(!IsMenu(m_hMenu))
		throw std::invalid_argument("specified menu handle is not valid.");
}

inline CMenu::~CMenu() {
	if(m_bManage) {
		while(true) {
			const UINT	c = GetItemCount();
			if(c == -1 || c == 0)
				break;
			RemoveMenuItem(0, false);
		}
		for(std::set<const CMenu*>::iterator it = m_children.begin(); it != m_children.end(); ++it)
			delete *it;
		::DestroyMenu(m_hMenu);
	}
}

inline bool CMenu::AppendMenuItem(UINT nID, UINT nType, const TCHAR* lpszCaption /* = 0 */) {
	AssertValid();
	return AppendMenuItem(nID, nType, MFS_ENABLED, lpszCaption);
}

inline bool CMenu::AppendMenuItem(UINT nID, UINT nType, UINT nState, const TCHAR* lpszCaption /* = 0 */) {
	AssertValid();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_FTYPE | MIIM_ID | MIIM_STATE;
	mii.fState = nState;
	mii.fType = nType;
	mii.wID = nID;
	if(toBoolean(nType & MFT_OWNERDRAW) && lpszCaption != 0) {
		mii.fMask |= MIIM_DATA;
		mii.dwItemData = reinterpret_cast<DWORD>(lpszCaption);
	}
	if(lpszCaption != 0) {
		mii.fMask |= MIIM_STRING;
		mii.dwTypeData = const_cast<TCHAR*>(lpszCaption);
	}
	return toBoolean(::InsertMenuItem(m_hMenu, ::GetMenuItemCount(m_hMenu), true, &mii));
}

inline DWORD CMenu::CheckMenuItem(UINT nID, bool bCheck /* = true */) {
	AssertValid();
	return ::CheckMenuItem(m_hMenu, nID, bCheck ? MFS_CHECKED : MFS_UNCHECKED);
}

inline bool CMenu::CheckMenuRadioItem(UINT nFirstID, UINT nLastID, UINT nItemID) {
	AssertValid();
	return toBoolean(::CheckMenuRadioItem(m_hMenu, nFirstID, nLastID, nItemID, MF_BYCOMMAND));
}

inline bool CMenu::DeleteMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValid();
	return toBoolean(::DeleteMenu(m_hMenu, iItem, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION));
}

inline bool CMenu::EnableMenuItem(UINT nID, bool bEnable /* = true */) {
	AssertValid();
	return toBoolean(::EnableMenuItem(m_hMenu, nID, MF_BYCOMMAND | (bEnable ? MFS_ENABLED : MFS_GRAYED)));
}

inline LRESULT CMenu::ExecuteMenuChar(TCHAR chCode, UINT nFlag) {
	AssertValid();

	const UINT	cMenuItems = GetItemCount();

	if(chCode >= _T('a') && chCode <= _T('z'))	// make upper
		chCode -= 0x20;
	for(UINT iItem = 0; iItem < cMenuItems; ++iItem) {
		TCHAR			szCaption[100];
		const TCHAR*	pszAccel = 0;

		GetMenuItemCaption(iItem, szCaption, 100, false);
		pszAccel = _tcschr(szCaption, chCode);
		if(pszAccel != 0) {
			if(pszAccel[1] == chCode)
				return (iItem | 0x00020000);
		}
	}
	return MNC_IGNORE;
}

inline UINT CMenu::GetDefaultMenuItem(UINT gmdiFlags) const {
	AssertValid();
	return ::GetMenuDefaultItem(m_hMenu, false, gmdiFlags);
}

inline bool CMenu::GetMenuItemCaption(UINT iItem, TCHAR* lpsz, int cchMax, bool bByCommand /* = true */) const {
	AssertValid();
	return toBoolean(::GetMenuString(m_hMenu, iItem, lpsz, cchMax, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION));
}

inline UINT CMenu::GetItemCount() const {
	AssertValid();
	return ::GetMenuItemCount(m_hMenu);
}

inline UINT CMenu::GetMenuItemID(int iItem) const {
	AssertValid();
	return ::GetMenuItemID(m_hMenu, iItem);
}

inline bool CMenu::GetMenuItemInfo(UINT iItem, MENUITEMINFO* pInfo, bool bByCommand /* = true */) const {
	AssertValid();
	if(pInfo == 0)
		return false;
	return toBoolean(::GetMenuItemInfo(m_hMenu, iItem, !bByCommand, pInfo));
}

inline UINT CMenu::GetMenuItemState(UINT iItem, bool bByCommand /* = true */) const {
	AssertValid();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_STATE;
	::GetMenuItemInfo(m_hMenu, iItem, !bByCommand, &mii);
	return mii.fState;
}

inline HMENU CMenu::GetSafeHmenu() const {
	return (this != 0) ? m_hMenu : 0;
}

inline CMenu* CMenu::GetSubMenu(UINT iItem) const {
	AssertValid();
	HMENU	hSubMenu = ::GetSubMenu(m_hMenu, iItem);
	return (hSubMenu != 0) ? new CMenu(hSubMenu) : 0;
}

inline bool CMenu::InsertMenuItem(UINT nID, UINT nPrevID,
		UINT nType, UINT nState, const TCHAR* lpszCaption) {
	AssertValid();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_FTYPE | MIIM_ID | MIIM_STATE;
	mii.fType = nType;
	mii.fState = nState;
	mii.wID = nID;
	if(toBoolean(nType & MFT_OWNERDRAW) && lpszCaption != 0) {
		mii.fMask |= MIIM_DATA;
		mii.dwItemData = reinterpret_cast<DWORD>(lpszCaption);
	}
	if(lpszCaption != 0) {
		mii.fMask |= MIIM_STRING;
		mii.dwTypeData = const_cast<TCHAR*>(lpszCaption);
	}
	return toBoolean(::InsertMenuItem(m_hMenu, nPrevID, false, &mii));
}

inline CMenu* CMenu::Load(HINSTANCE hInstance, const TCHAR* lpszName) {
	if(HMENU hMenu = ::LoadMenu(hInstance, lpszName)) {
		CMenu*	pMenu = new CMenu(hMenu);
		pMenu->m_bManage = true;
		return pMenu;
	} else
		return 0;
}

inline CMenu* CMenu::Load(const MENUTEMPLATE* pTemplate) {
	if(HMENU hMenu = ::LoadMenuIndirect(pTemplate)) {
		CMenu*	pMenu = new CMenu(hMenu);
		pMenu->m_bManage = true;
		return pMenu;
	} else
		return 0;
}

inline bool CMenu::ModifyMenuItem(UINT nID, UINT nFlags, const TCHAR* lpszPrompt) {
	AssertValid();
	assert(!(nFlags & MF_POPUP));
	return toBoolean(::ModifyMenu(m_hMenu, nID, MF_BYCOMMAND | nFlags, nID, lpszPrompt));
}

inline bool CMenu::RemoveMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValid();
	return toBoolean(::RemoveMenu(m_hMenu, iItem, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION));
}

inline bool CMenu::SetChildPopup(const CMenu* pPopup, UINT iItem, bool bDelegateOwnership /* = true */, bool bByCommand /* = false */) {
	AssertValid();
	if(pPopup == 0)
		return false;

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_SUBMENU;
	mii.hSubMenu = pPopup->m_hMenu;

	if(toBoolean(::SetMenuItemInfo(m_hMenu, iItem, !bByCommand, &mii))) {
		if(bDelegateOwnership)
			m_children.insert(pPopup);
		return true;
	} else
		return false;
}

inline bool CMenu::SetDefaultMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValid();
	return toBoolean(::SetMenuDefaultItem(m_hMenu, iItem, !bByCommand));
}

inline bool CMenu::TrackPopupMenu(UINT nFlags, int x, int y, HWND hWnd, const RECT* prcRect /* = 0 */) {
	AssertValid();
	return toBoolean(::TrackPopupMenu(m_hMenu, nFlags, x, y, 0, hWnd, prcRect));
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#pragma warning(default : 4297)

#endif /* _MENU_OPERATOR_H_ */

/* [EOF] */