// Object.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _OBJECT_H_
#define _OBJECT_H_

#include <cassert>
#include <string>
#include <sstream>
#define size_t	std::size_t
#include <windows.h>
#include <tchar.h>
#undef size_t

#ifdef _UNICODE
#define tstring	std::wstring
#else
#define tstring	std::string
#endif

namespace Manah {
	namespace Windows {
		class CDumpContext;
	}
}


// modern types
/////////////////////////////////////////////////////////////////////////////

typedef unsigned char		uchar;
typedef unsigned short		ushort;
typedef unsigned int		uint;
typedef unsigned long		ulong;


// CObject class definition and implementation
/////////////////////////////////////////////////////////////////////////////

namespace Manah {

class CObject {
	// methods
protected:
#ifdef _DEBUG
	virtual void AssertValid() const {
		assert(this != 0);
	}
#else
	virtual void AssertValid() const {}
#endif /* _DEBUG */
};


// CDumpContext class definition
/////////////////////////////////////////////////////////////////////////////

namespace Windows {

class CDumpContext : public CObject {
	// constructor
public:
	CDumpContext(const TCHAR* lpszFileName = 0) throw(std::exception);

	// operators
	template <typename T>
	CDumpContext&	operator <<(const T& rhs);

	// methods
public:
	void	Flush() throw();
	void	HexDump(const TCHAR* lpszLine, uchar* pb, int nBytes, int nWidth = 0x10) throw();

	// data member
private:
	TCHAR*	m_pszFileName;	// error log
};

namespace {
	CDumpContext	dout;
}

} /* namespace Windows */
} /* namespace Manah */


// Macros
/////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
#define ASSERT(exp)	assert(exp)
#define VERIFY(exp)	assert(exp)
#else
#define ASSERT(exp)
#define VERIFY(exp)	(exp)
#endif /* _DEBUG */

#define toBoolean(exp)				((exp) != 0)

#ifndef LVS_EX_LABELTIP	// VC6
	#define LVS_EX_LABELTIP	0x4000
#endif
#ifndef ListView_SetCheckState	// VC6
	#define ListView_SetCheckState(hwndLV, i, fCheck)	\
	ListView_SetItemState(hwndLV, i,					\
	INDEXTOSTATEIMAGEMASK((fCheck)? 2 : 1), LVIS_STATEIMAGEMASK)
#endif


// CDumpContext class implementation
/////////////////////////////////////////////////////////////////////////////

inline Manah::Windows::CDumpContext::CDumpContext(
		const TCHAR* lpszFileName /* = 0 */) throw(std::exception) {
#ifdef _DEBUG
	if(lpszFileName != 0)
		throw std::exception("File log is not supported!");
		// _tcscpy(m_szFileName, lpszFileName);
#endif /* _DEBUG */
}

inline void Manah::Windows::CDumpContext::Flush() {
	AssertValid();
	/* not implemented */
}

inline void Manah::Windows::CDumpContext::HexDump(
		const TCHAR* lpszLine, uchar* pb, int nBytes, int nWidth /* = 0x10 */) {
#ifdef _DEBUG
	AssertValid();

	TCHAR*	pszOutput;	// all line
	TCHAR	szByte[4];

	pszOutput = new TCHAR[static_cast<std::size_t>(
		(_tcslen(lpszLine) + 3 * nWidth + 2) * static_cast<float>(nBytes / nWidth))];
	_tcscpy(pszOutput, lpszLine);
	for(int i = 0; i < nBytes; i++){
		wsprintf(szByte, _T(" %d"), pb);
		_tcscat(pszOutput, szByte);
		if(i % nWidth == 0){
			_tcscat(pszOutput, _T("\n"));
			_tcscat(pszOutput, lpszLine);
		}
	}
	::OutputDebugString(_T("\n>----Dump is started"));
	::OutputDebugString(pszOutput);
	::OutputDebugString(_T("\n>----Dump is done"));
	delete[] pszOutput;
#endif /* _DEBUG */
}

template <typename T>
inline Manah::Windows::CDumpContext&
Manah::Windows::CDumpContext::operator <<(const T& rhs) throw() {
#ifdef _DEBUG
	AssertValid();
	std::basic_ostringstream<TCHAR>	ss;
	ss << rhs;
	::OutputDebugString(ss.str().c_str());
#endif /* _DEBUG */
	return *this;
}

#endif /* _MANAH_OBJECT_H_ */

/* [EOF] */