// Text.h
// (c) 2002-2004 exeal

#ifndef _TEXT_H_
#define _TEXT_H_

#include <stdexcept>
#include <clocale>
#include <cstdlib>
#define size_t	std::size_t
#include <mbstring.h>
#include <mbctype.h>
#undef size_t


namespace Manah {
namespace Text {

	///	R[h|Cg^
	typedef unsigned long	CodePoint;	// uint32_t

	/// Shift-JIS ɂ镶̃^Cv
	enum SJISCharType {
		/// ASCII 䕶
		CT_CONTROL				= 0x00,
		///  (0x09-0x0D 0x20)
		CT_SPACE				= 0x01,
		/// 0-9
		CT_DIGIT				= 0x10,
		/// A-Z
		CT_UPPERALPHA			= 0x11,
		/// a-z
		CT_LOWERALPHA			= 0x12,
		/// _
		CT_UNDERSCORE			= 0x13,
		/// !"#$%&'()*+,-./:;<=>?@[\]^`{|}~
		CT_SYMBOL				= 0x20,
		/// pJi (0xA1-0xDF)
		CT_KANA_HALFWIDTH		= 0x30,
		/// SpL (0x8140-0x81FC, 0xFA55-57)
		CT_KANA_SYMBOL			= 0x40,
		/// Sp (0x824F-0x8258)
		CT_FULLWIDTH_DIGIT		= 0x50,
		/// SpAt@xbg啶 (0x8260-0x8270)
		CT_FULLWIDTH_UPPERALPHA	= 0x51,
		/// SpAt@xbg (0x8281-0x829A)
		CT_FULLWIDTH_LOWERALPHA	= 0x52,
		///  (0x829F-0x82F1) ƐL΂ (0x8154, 0x8155)
		CT_KANA_HIRA			= 0x60,
		/// Љ (0x8340-0x8396) ƐL΂ (0x8152, 0x8153)
		CT_KANA_KATA			= 0x70,
		/// SpMV啶 (0x839F-0x83B6)
		CT_GREEK_UPPER			= 0x80,
		/// SpMV (0x83BF-0x83D6)
		CT_GREEK_LOWER			= 0x81,
		/// SpL啶 (0x8440-0x8460)
		CT_CYRILLIC_UPPER		= 0x90,
		/// SpL (0x8470-0x8491)
		CT_CYRILLIC_LOWER		= 0x91,
		///	r (0x849F-0x84BE)
		CT_SPECIAL_BOXDRAWING	= 0xA0,
		///	ې (0x8740-0x8753)
		CT_SPECIAL_CIRCLEDDIGIT	= 0xA1,
		///	[} (0x8754-0x875D, 0xFA40-0xFA49)
		CT_SPECIAL_ROMANNUMERAL	= 0xA2,
		///	P (0x875F-0x8775)
		CT_SPECIAL_UNIT			= 0xA3,
		///	wL (0x8793-94, 98-99)
		CT_SPECIAL_MATHEMATICAL	= 0xA4,
		///	̑̓ꕶ (0x877E-0x878F)
		CT_SPECIAL_OTHER		= 0xA5,
		/// JIS 1 (0x889F-0x9872)
		CT_KANJI1				= 0xB0,
		/// JIS 2 (0x989F-0xEA9E)
		CT_KANJI2				= 0xB1,
		///  JIS  (0xFA5C-0xFC4B)
		CT_KANJI3				= 0xB2,
		/// ̑Asȕ
		CT_OTHER				= 0xF0
	};

	/// 悭gR[h
	enum CharCode {
		/// (ʁAftHg)
		CC_AUTO,
		/// Shift-JIS
		CC_SHIFTJIS,
		/// JIS
		CC_JIS,
		/// EUC-JP
		CC_EUCJP,
		/// ISO-2022-KR
		CC_ISOKR,
		/// ISO-2022-CN
		CC_ISOCN,
		/// UTF-16 little endian
		CC_UTF16LE,
		/// UTF-16 big endian
		CC_UTF16BE,
		/// UTF-8
		CC_UTF8,
		/// UTF-7
		CC_UTF7,
		///	UTF-5
		CC_UTF5,
		/// UTF-32 little endian
		CC_UTF32LE,
		/// UTF-32 big endian
		CC_UTF32BE
	};

	///	Unicode 4.0 JeS (Win32 API ͓rbgtB[hpӂĂ邪Â)
	enum UnicodeSubrange {
		US_BASIC_LATIN, US_LATIN1_SUPPLEMENT, US_LATIN_EXTENDED_A,
		US_LATIN_EXTENDED_B, US_IPA_EXTENSIONS, US_SPACING_MODIFIER_LETTERS,
		US_COMBINING_DIACRITICAL_MARKS, US_GREEK, US_CYRILLIC,
		US_CYRILLIC_SUPPLEMENTARY, US_ARMENIAN, US_HEBREW, US_ARABIC, US_SYRIAC,
		US_THAANA, US_DEVANAGARI, US_BENGALI, US_GURMUKHI, US_GUJARATI, US_ORIYA,
		US_TAMIL, US_TELUGU, US_KANNADA, US_MALAYALAM, US_SINHALA, US_THAI, US_LAO,
		US_TIBETAN, US_MYANMAR, US_GEORGIAN, US_HANGUL_JAMO, US_ETHIOPIC,
		US_CHEROKEE, US_UNIFIED_CANADIAN_ABORIGINAL_SYLLABIC, US_OGHAM, US_RUNIC,
		US_TAGALOG, US_HANUNOO, US_BUHID, US_TAGBANWA, US_KHMER, US_MONGOLIAN,
		US_LIMBU, US_TAI_LE, US_KHMER_SYMBOLS, US_PHONETIC_EXTENSIONS,
		US_LATIN_EXTENDED_ADDITIONAL, US_GREEK_EXTENDED, US_GENERAL_PUNCTUATION,
		US_SUPERSCRIPTS_AND_SUBSCRIPTS, US_CURRENCY_SYMBOLS,
		US_COMBINING_MARKS_FOR_SYMBOLS, US_LETTERLIKE_SYMBOLS, US_NUMBER_FORMS,
		US_ARROWS, US_MATHEMATICAL_OPERATORS, US_MISCELLANEOUS_TECHNICAL,
		US_CONTROL_PICTURES, US_OPTICAL_CHARACTER_RECOGNITION,
		US_ENCLOSED_ALPHANUMERICS, US_BOX_DRAWING, US_BLOCK_ELEMENTS,
		US_GEOMETRIC_SHAPES, US_MISCELLANEOUS_SYMBOLS, US_DINGBATS,
		US_MISCELLANEOUS_MATHEMATICAL_SYMBOLS_A, US_SUPPLEMENTAL_ARROWS_A,
		US_BRAILLE_PATTERNS, US_SUPPLEMENTAL_ARROWS_B,
		US_MISCELLANEOUS_MATHEMATICAL_SYMBOLS_B,
		US_SUPPLEMENTAL_MATHEMATICAL_OPERATORS,
		US_MISCELLANEOUS_SYMBOLS_AND_ARROWS, US_CJK_RADICALS_SUPPLEMENT,
		US_KANGXI_RADICALS, US_IDEOGRAPHIC_DESCRIPTION_CHARACTERS,
		US_CJK_SYMBOLS_AND_PUNCTUATION, US_HIRAGANA, US_KATAKANA, US_BOPOMOFO,
		US_HANGUL_COMPATIBILITY_JAMO, US_KANBUN, US_BOPOMOFO_EXTENDED,
		US_KATAKANA_PHONETIC_EXTENSIONS, US_ENCLOSED_CJK_LETTERS_AND_MONTHS,
		US_CJK_COMPATIBILITY, US_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A,
		US_YIJING_HEXAGRAM_SYMBOLS, US_CJK_UNIFIED_IDEOGRAPHS, US_YI_SYLLABLES,
		US_YI_RADICALS, US_HANGUL_SYLLABLES, US_HIGH_SURROGATES,
		US_HIGH_PRIVATE_USE_SURROGATES, US_LOW_SURROGATES, US_PRIVATE_USE_AREA,
		US_CJK_COMPATIBILITY_IDEOGRAPHS, US_ALPHABETIC_PRESENTATION_FORMS,
		US_ARABIC_PRESENTATION_FORMS_A, US_VARIATION_SELECTORS,
		US_COMBINING_HALF_MARKS, US_CJK_COMPATIBILITY_FORMS,
		US_SMALL_FORM_VARIANTS, US_ARABIC_PRESENTATION_FORMS_B,
		US_HALFWIDTH_AND_FULLWIDTH_FORMS, US_SPECIALS, US_LINEAR_B_SYLLABARY,
		US_LINEAR_B_IDEOGRAMS, US_AEGEAN_NUMBERS, US_OLD_ITALIC, US_GOTHIC,
		US_UGARITIC, US_DESERET, US_SHAVIAN, US_OSMANYA, US_CYPRIOT_SYLLABARY,
		US_BYZANTINE_MUSICAL_SYMBOLS, US_MUSICAL_SYMBOLS,
		US_TAI_XUAN_JING_SYMBOLS, US_MATHEMATICAL_ALPHANUMERIC_SYMBOLS,
		US_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B,
		US_CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT, US_TAGS,
		US_VARIATION_SELECTORS_SUPPLEMENT, US_SUPPLEMENTARY_PRIVATE_USE_AREA_A,
		US_SUPPLEMENTARY_PRIVATE_USE_AREA_B, US_UNASSIGNED
	};

	///	Unicode 4.0 Script vpeB
	enum UnicodeScriptProperty {
		USP_LATIN, USP_GREEK, USP_CYRILLIC, USP_ARMENIAN, USP_HEBREW,
		USP_ARABIC, USP_SYRIAC, USP_THAANA, USP_DEVANAGARI, USP_BENGALI,
		USP_GURMUKHI, USP_GUJARATI, USP_ORIYA, USP_TAMIL, USP_TELUGU,
		USP_KANNADA, USP_MALAYALAM, USP_SINHARA, USP_THAI, USP_LAO,
		USP_TIBETAN, USP_MYANMAR, USP_GEORGIAN, USP_HANGUL, USP_ETHIOPIC,
		USP_CHEROKEE, USP_CANADIAN_ABORIGINAL, USP_OGHAM, USP_RUNIC,
		USP_KHMER, USP_MONGOLIAN, USP_HIRAGANA, USP_KATAKANA, USP_BOPOMOFO,
		USP_HAN, USP_YI, USP_OLD_ITALIC, USP_GOTHIC, USP_DESERET,
		USP_INHERITED, USP_TAGALOG, USP_HANUNOO, USP_BUHID, USP_TAGBANWA,
		USP_LIMBU, USP_TAI_LE, USP_LINEAR_B, USP_UGARITIC, USP_SHAVIAN,
		USP_OSMANYA, USP_CYPRIOT, USP_BRAILLE, USP_NOT_SCRIPT
	};


	/* R[hϊ */
	std::size_t	ConvertEUCToShiftJIS(char* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertJISToShiftJIS(char* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertShiftJISToEUC(char* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertShiftJISToJIS(char* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertShiftJISToUnicode(char* pszDest, const char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertShiftJISToUnicode(wchar_t* pszDest, const char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertShiftJISToUTF8(char* pszDest, const char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertUnicodeTo(CharCode& charCode, char* pszDest,
				const wchar_t* pszSrc, bool* pbUsedDefaultChar = 0, std::size_t cch = -1);
	std::size_t	ConvertUnicodeToShiftJIS(char* pszDest,
				const char* pszSrc, bool* pbUsedDefaultChar = 0, std::size_t cch = -1);
	std::size_t	ConvertUnicodeToShiftJIS(char* pszDest,
				const wchar_t* pszSrc, bool* pbUsedDefaultChar = 0, std::size_t cch = -1);
	std::size_t	ConvertUnicodeToUTF8(char* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertUTF8ToShiftJIS(char* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);
	std::size_t	ConvertToUnicode(CharCode& charCode, wchar_t* pszDest, const unsigned char* pszSrc, std::size_t cch = -1);

	/* 1̏ */
	UnicodeScriptProperty	GetScriptProperty(CodePoint cp);
	UnicodeSubrange			GetUnicodeCharCategory(CodePoint cp);
	UnicodeSubrange			GetUnicodeStringCategory(const wchar_t* pch, std::size_t cch);
	bool					IsEUCLeadByte(unsigned char ch);
	bool					IsEUCTrailByte(unsigned char ch);
	bool					IsEUCCharacters(const unsigned char* pch);
	bool					IsShiftJISCharacters(const unsigned char* pch);
	bool					IsUTF16HighSurrogate(wchar_t ch);
	bool					IsUTF16LowSurrogate(wchar_t ch);

	/* UTF-16 TQ[gyA */
	CodePoint	DecodeUTF16SurrogatePairToCodePoint(const wchar_t* pszSrc, std::size_t cch);
	bool		EncodeCodePointToUTF16SurrogatePair(CodePoint nCodePoint, wchar_t* pszDest);

	/* BOM */
	bool		HasBOMOfISO2022CN(const unsigned char* pszText, std::size_t cch);
	bool		HasBOMOfISO2022KR(const unsigned char* pszText, std::size_t cch);

	/* R[h */
	CharCode	DetectCharacterCode(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsShiftJISString(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsJISString(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsEUCString(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsUTF16LEString(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsUTF16BEString(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsUTF32LEString(const unsigned char* pszText, std::size_t cch);
	std::size_t	IsUTF32BEString(const unsigned char* pszText, std::size_t cch);

	/* Unicode ̕ϊ */
	CodePoint		FoldCaseSimple(CodePoint cp);
	CodePoint		ToAsciiDigit(CodePoint cp);
	wchar_t			ToUpper(wchar_t ch);
	void			ToUpper(wchar_t* psz, std::size_t cch = -1);
	wchar_t			ToLower(wchar_t ch);
	void			ToLower(wchar_t* psz, std::size_t cch = -1);
	wchar_t			ToHiragana(wchar_t ch);
	void			ToHiragana(wchar_t* psz, std::size_t cch = -1);
	wchar_t			ToKatakana(wchar_t ch);
	void			ToKatakana(wchar_t* psz, std::size_t cch = -1);
	const wchar_t	ToFullWidth(wchar_t ch);
	const wchar_t	ToHalfWidth(wchar_t ch);

	/* ̃^CvƒPꋫE */
	// ̃\bh͏I[EƂ݂ȂȂ
	SJISCharType			GetCharType(const unsigned char* pszText);
	const unsigned char*	FindWordBorder(const unsigned char* pszText,
								unsigned int iBegin = 0, bool bIgnoreSpaces = true,
								bool bReverse = false, std::size_t len = -1);
	const wchar_t*			FindWordBorder(const wchar_t* pszText,
								unsigned int iBegin = 0, bool bIgnoreSpaces = true,
								bool bReverse = false, std::size_t len = -1);
	bool					HasWordBorderAt(const unsigned char* pszText, unsigned int i);
	bool					HasWordBorderAt(const wchar_t* pszText, unsigned int i);

} // namespace Text
} // namespace Manah


/**
 *	EUC-JP  Shift-JIS ɕϊ
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertEUCToShiftJIS(
		char* pszDest, const unsigned char* pszSrc, std::size_t cchSrc /* = -1 */) {
	std::size_t		iSrc = 0;		// pszSrc ̈ʒu
	std::size_t		iDest = 0;		// pszDest ̈ʒu
	unsigned int	nKanjiChar;		// _mbcjistojms Ŏgp
	unsigned int	nResultChar;	// _mbcjistojms ̖߂l

	assert(pszDest != 0);

	if(cchSrc == -1)
		cchSrc = strlen(reinterpret_cast<const char*>(pszSrc));

	while(iSrc < cchSrc) {	// ϊ
		if(*(pszSrc + iSrc) == 0x8E) {	// pJi
			if(pszSrc[iSrc + 1] == 0) {	// '0x8E' ͏I[łĂ͂ȂȂ
				pszDest = 0;
				return -1;
			}
			*(pszDest + iDest) = *(pszSrc + iSrc + 1);
			++iDest;
			iSrc += 2;
		} else if(*(pszSrc + iSrc + 1) != '\0'
			&& (pszSrc[iSrc + 0] >= 0xA1 && pszSrc[iSrc + 1] <= 0xFE)
			&& (pszSrc[iSrc + 1] >= 0xA1) && pszSrc[iSrc + 1] <= 0xFE) {	// EUC 
			nKanjiChar = (pszSrc[iSrc + 0] & 0xFF) << 8;
			nKanjiChar |= pszSrc[iSrc + 1];
			nKanjiChar &= 0x7F7F;	// EUC -> JIS
			nResultChar = _mbcjistojms(nKanjiChar);
			if(nResultChar != 0){	// _mbcjistojms 
				pszDest[iDest + 0] = (nResultChar & 0xFF00) >> 8;
				pszDest[iDest + 1] = nResultChar & 0x00FF;
				iSrc += 2;
				iDest += 2;
			} else {
				pszDest[iDest] = pszSrc[iSrc];
				++iSrc;
				++iDest;
			}
		} else {	// ̑
			pszDest[iDest] = pszSrc[iSrc];
			++iSrc;
			++iDest;
		}
	}
	return iDest;
}

/**
 *	JIS  Shift-JIS ɕϊ
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertJISToShiftJIS(
		char* pszDest, const unsigned char* pszSrc, std::size_t cchSrc /* = -1 */) {
	enum JISCharType {
		JISCT_ASCII, JISCT_8BIT, JISCT_FULLWIDTH
	};
	std::size_t		iSrc = 0;
	std::size_t		iDest = 0;
	JISCharType		jctCurrent = JISCT_ASCII;
	unsigned int	nConverted;

	if(cchSrc == -1)
		cchSrc = strlen(reinterpret_cast<const char*>(pszSrc));

	while(iSrc < cchSrc) {
		if(iSrc < cchSrc - 2
				&& pszSrc[iSrc + 0] == 0x1B
				&& pszSrc[iSrc + 1] == '('
				&& (pszSrc[iSrc + 2] == 'B' || pszSrc[iSrc + 2] == 'J')) {	// ASCII
			jctCurrent = JISCT_ASCII;
			iSrc += 3;
		} else if(iSrc < cchSrc - 2
				&& pszSrc[iSrc + 0] == 0x1B
				&& pszSrc[iSrc + 1] == '('
				&& pszSrc[iSrc + 2] == 'I') {	// 8rbg
			jctCurrent = JISCT_8BIT;
			iSrc += 3;
		} else if(iSrc < cchSrc - 2
				&& pszSrc[iSrc + 0] == 0x1B
				&& pszSrc[iSrc + 1] == '$'
				&& (pszSrc[iSrc + 2] == 'B' || pszSrc[iSrc + 2] == '@')) {	// Sp
			jctCurrent = JISCT_FULLWIDTH;
			iSrc += 3;
		} else if(jctCurrent == JISCT_ASCII) {
			pszDest[iDest] = pszSrc[iSrc];
			++iSrc;
			++iDest;
		} else if(jctCurrent == JISCT_8BIT) {
			pszDest[iDest] = pszSrc[iSrc] | 0x80;
			++iSrc;
			++iDest;
		} else {
			nConverted = _mbcjistojms((pszSrc[iSrc] << 8) | (pszSrc[iSrc + 1]));
			if(nConverted != 0) {	// 
				jctCurrent = JISCT_FULLWIDTH;
				pszDest[iDest + 0] = (nConverted >> 8);
				pszDest[iDest + 1] = nConverted;
				iSrc += 2;
				iDest += 2;
			} else {
				pszDest[iDest] = pszSrc[iSrc];
				++iSrc;
				++iDest;
			}
		}
	}

	return iDest;
}

/**
 *	Shift-JIS  EUC ɕϊ
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertShiftJISToEUC(
		char* pszDest, const unsigned char* pszSrc, std::size_t cchSrc /* = -1 */) {
	std::size_t		iSrc = 0;		// pszSrc ̈ʒu
	std::size_t		iDest = 0;		// pszDest ̈ʒu
	unsigned int	nKanjiChar;		// _mbcjmstojis Ŏgp
	unsigned int	nResultChar;	// _mbcjistojms ̖߂l

	assert(pszDest != 0);

	if(cchSrc == -1)
		cchSrc = strlen(reinterpret_cast<const char*>(pszSrc));

	while(iSrc < cchSrc) {	// ϊ
		if(_ismbbkana(pszSrc[iSrc])) {	// pJi
			pszDest[iDest + 0] = static_cast<unsigned char>(0x8E);
			pszDest[iDest + 1] = pszSrc[iSrc];
			iDest += 2;
			++iSrc;
		} else if(_ismbblead(pszSrc[iSrc])
			&& (_ismbbtrail(pszSrc[iSrc]))) {	// 2oCg
			nKanjiChar = (pszSrc[iSrc + 0] << 8);
			nKanjiChar |= pszSrc[iSrc + 1];
			nResultChar = _mbcjmstojis(nKanjiChar);
			if(nResultChar != 0) {	// _mbcjmstojis 
				pszDest[iDest + 0] = (nResultChar | 0x8000) >> 8;
				pszDest[iDest + 1] = nResultChar | 0x0080;
				iSrc += 2;
				iDest += 2;
			} else {
				pszDest[iDest + 0] = pszSrc[iSrc + 0];
				pszDest[iDest + 1] = pszSrc[iSrc + 1];
				iSrc += 2;
				iDest += 2;
			}
		} else {	// ̑
			pszDest[iDest] = pszSrc[iSrc];
			++iSrc;
			++iDest;
		}
	}
	return iDest;
}

/**
 *	Shift-JIS  JIS ɕϊ
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertShiftJISToJIS(
		char* pszDest, const unsigned char* pszSrc, std::size_t cchSrc /* = -1 */) {
	static enum JISCharType {
		JISCT_ASCII, JISCT_8BIT, JISCT_FULLWIDTH
	};
	std::size_t		iSrc = 0;
	std::size_t		iDest = 0;
	JISCharType		jctPrevious = JISCT_ASCII;
	JISCharType		jctCurrent = JISCT_ASCII;	// ݂̎̕
	char			szConvertedChars[2];
	unsigned int	nConverted;

	assert(pszDest != 0);

	if(cchSrc == -1)
		cchSrc = strlen(reinterpret_cast<const char*>(pszSrc));

	while(iSrc < cchSrc) {
		if(iSrc < cchSrc - 1 && IsShiftJISCharacters(pszSrc + iSrc)) {	// Shift-JIS
			nConverted = _mbcjmstojis(
				static_cast<int>((pszSrc[iSrc] << 8) | pszSrc[iSrc + 1]));
			if(nConverted != 0) {	// 
				jctCurrent = JISCT_FULLWIDTH;
				szConvertedChars[0] = (nConverted >> 8);
				szConvertedChars[1] = nConverted;
			} else {
				jctCurrent = JISCT_ASCII;
				szConvertedChars[0] = pszSrc[iSrc];
				szConvertedChars[1] = 0;
			}
		} else if(*(pszSrc + iSrc) & 0x80) {	// 8rbg
			jctCurrent = JISCT_8BIT;
			szConvertedChars[0] = pszSrc[iSrc] - 0x80;
			szConvertedChars[1] = 0;
		} else {	// ASCII
			jctCurrent = JISCT_ASCII;
			szConvertedChars[0] = pszSrc[iSrc];
			szConvertedChars[1] = 0;
		}
		if(jctCurrent != jctPrevious) {	// ̎ނ̕ύX
			if(jctCurrent == JISCT_ASCII)
				strncpy(pszDest + iDest, "\x1B(B", 3);
			else if(jctCurrent == JISCT_8BIT)
				strncpy(pszDest + iDest, "\x1B(I", 3);
			else
				strncpy(pszDest + iDest, "\x1B$B", 3);
			iDest += 3;
			jctPrevious = jctCurrent;
		}
		strncpy(pszDest + iDest, szConvertedChars, 2);
		iSrc += (jctCurrent != JISCT_FULLWIDTH) ? 1 : 2;
		iDest += (jctCurrent != JISCT_FULLWIDTH) ? 1 : 2;
	}
	if(jctCurrent != JISCT_ASCII) {
		strncpy(pszDest + iDest, "\x1B(B", 3);
		iDest += 3;
	}
	return iDest;
}

/**
 *	Shift-JIS  UTF-16 ɕϊ
 *	@param pszDest	ϊ̕ (char)
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertShiftJISToUnicode(
		char* pszDest, const char* pszSrc, std::size_t cchSrc /* = -1 */) {
	wchar_t		wchChar;		// mbtowc ŕϊꂽCh
	int			nChangedLength;	// mbtowc ̖߂l
	std::size_t	iSrc = 0;		// pszSrc ̈ʒu
	std::size_t	iDest = 0;		// pszDest ̈ʒu

	assert(pszDest != 0);

	if(cchSrc == -1)
		cchSrc = strlen(pszSrc);

	while(iSrc < cchSrc) {
		nChangedLength = mbtowc(&wchChar, pszSrc + iSrc, 2);
		if(nChangedLength != 0 && nChangedLength != -1) {	// mbtowc 
			*(reinterpret_cast<wchar_t*>(pszDest + iDest)) = wchChar;
			iSrc += nChangedLength;
		} else {
			pszDest[iDest + 0] = 0;
			pszDest[iDest + 1] = pszSrc[iSrc];
			++iSrc;
		}
		iDest += 2;
	}
	return iDest;
}

/**
 *	Shift-JIS  UTF-16 ɕϊ (: MultiByteToWideChar)
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertShiftJISToUnicode(
		wchar_t* pszDest, const char* pszSrc, std::size_t cchSrc /* = -1 */) {
	int			cChanged;
	wchar_t		wch;
	std::size_t	iSrc = 0;
	std::size_t	iDest = 0;

	assert(pszDest != 0);

	if(cchSrc == -1)
		cchSrc = strlen(pszSrc);

	setlocale(LC_CTYPE, "japanese");
	while(iSrc < cchSrc) {
		cChanged = mbtowc(&wch, pszSrc + iSrc, 2);
		if(cChanged != 0 && cChanged != -1) {
			iSrc += cChanged;
			*(pszDest + iDest) = wch;
		} else {
			++iSrc;
			*(pszDest + iDest) = *(pszSrc + iSrc);
		}
		++iDest;
	}
	return iDest;
}

/**
 *	Shift-JIS  UTF-8 ɕϊB
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertShiftJISToUTF8(char* pszDest, const char* pszSrc, std::size_t cchSrc /* = -1 */) {
//	len = ConvertShiftJISToUnicode(pszDest, pszSrc, cchSrc);
//	return ConvertUnicodeToUTF8(pszSrc, cchSrc);
	return -1;
}

/**
 *	Cӂ̕R[h Unicode ɕϊ
 *	@param charCode	R[hBCC_AUTO ̏ꍇʂAgpR[hi[
 *	@param pszDest	ϊ̕
 *	@param pszSrc	ϊ镶
 *	@param cchSrc	ϊ镶̒Bȗ null IƂČvZ
 *	@return			ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertToUnicode(
		CharCode& charCode, wchar_t* pszDest, const unsigned char* pszSrc, std::size_t cchSrc /* = -1 */) {
	char*	pszBuffer = 0;

	if(cchSrc == -1)
		cchSrc = strlen(reinterpret_cast<const char*>(pszSrc));
	if(charCode == CC_AUTO)
		charCode = DetectCharacterCode(pszSrc, (cchSrc <= 200 * 1024) ? cchSrc : 200 * 1024);

	switch(charCode) {
	case CC_SHIFTJIS:
		return ConvertShiftJISToUnicode(pszDest, reinterpret_cast<const char*>(pszSrc), cchSrc);
	case CC_EUCJP:
		pszBuffer = new char[cchSrc * 2 + 1];
		cchSrc = ConvertEUCToShiftJIS(pszBuffer, pszSrc, cchSrc);
		cchSrc = ConvertShiftJISToUnicode(pszDest, pszBuffer, cchSrc);
		delete[] pszBuffer;
		return cchSrc;
	case CC_JIS:
		pszBuffer = new char[cchSrc * 3 + 1];
		cchSrc = ConvertJISToShiftJIS(pszBuffer, pszSrc, cchSrc);
		cchSrc = ConvertShiftJISToUnicode(pszDest, pszBuffer, cchSrc);
		delete[] pszBuffer;
		return cchSrc;
//	case CC_UTF16LE:
//		assert(pszDest != 0);
//		btBOMOffset = (HasBOMOfUTF16(pszSrc, cchSrc) != 0) ? 2 : 0;
//		for(i = btBOMOffset; i < cchSrc; i += 2)
//			*(pszDest + (i - btBOMOffset) / 2) = (pszSrc[i] & 0x00FF) | ((pszSrc[i + 1] << 8) & 0xFF00);
//		return cchSrc / 2 - 1;
//	case CC_UTF16BE:
//		assert(pszDest != 0);
//		btBOMOffset = (HasBOMOfUTF16(pszSrc, cchSrc) != 0) ? 2 : 0;
//		for(i = btBOMOffset; i < cchSrc; i += 2)
//			*(pszDest + (i - btBOMOffset) / 2) = ((pszSrc[i] << 8) & 0xFF00) | (pszSrc[i + 1] & 0x00FF);
//		return cchSrc / 2 - 1;
//	case CC_UTF32LE:
//		assert(pszDest != 0);
//		btBOMOffset = (HasBOMOfUTF32(pszSrc, cchSrc) != 0) ? 4 : 0;
//		for(i = btBOMOffset; i < cchSrc; i += 4)
//			*(pszDest + (i - btBOMOffset) / 4) = (pszSrc[i] & 0x00FF) | ((pszSrc[i + 1] << 8) & 0xFF00);
//		return cchSrc / 4 - 1;
//	case CC_UTF32BE:
//		assert(pszDest != 0);
//		btBOMOffset = (HasBOMOfUTF32(pszSrc, cchSrc) != 0) ? 4 : 0;
//		for(i = btBOMOffset; i < cchSrc; i += 4)
//			*(pszDest + (i - btBOMOffset) / 4) = ((pszSrc[i] << 8) & 0xFF00) | (pszSrc[i + 1] & 0x00FF);
//		return cchSrc / 4 - 1;
//	case CC_UTF8:
//		return ConvertUTF8ToUnicode(pszDest, pszSrc, cchSrc);
//	case CC_UTF7:
//		return ConvertUTF7ToUnicode(pszDest, pszSrc, cchSrc);
//	case CC_UTF5:
//		return ConvertUTF5ToUnicode(pszDest, pszSrc, cchSrc);
	}
	assert(false);
	return 0;
}

/**
 *	Unicode  Cӂ̕R[hɕϊ
 *	@param charCode				R[h
 *	@param pszDest				ϊ̕
 *	@param pszSrc				ϊ镶
 *	@param pbUsedDefaultChar	ϊɑΉ镶̂Ŋ̑֕g
 *	@param cchSrc				ϊ镶̒Bȗ null IƂČvZ
 *	@return						ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertUnicodeTo(
		CharCode& charCode, char* pszDest, const wchar_t* pszSrc,
		bool* pbUsedDefaultChar /* = 0*/, std::size_t cchSrc /* = -1 */) {
	std::size_t	i;
	char*		pszBuffer = 0;

	if(pbUsedDefaultChar != 0)
		*pbUsedDefaultChar = false;
	if(cchSrc == -1)
		cchSrc = wcslen(pszSrc);

	switch(charCode) {
	case CC_AUTO:
	case CC_UTF16LE:
		assert(pszDest != 0);
		for(i = 0; i < cchSrc; ++i) {
			pszDest[i * 2 + 0] = pszSrc[i] & 0x00FF;
			pszDest[i * 2 + 1] = (pszSrc[i] & 0xFF00) >> 8;
		}
		return cchSrc * 2;
	case CC_UTF16BE:
		assert(pszDest != 0);
		for(i = 0; i < cchSrc; ++i) {
			pszDest[i * 2 + 0] = (pszSrc[i] & 0xFF00) >> 8;
			pszDest[i * 2 + 1] = pszSrc[i] & 0x00FF;
		}
		return cchSrc * 2;
	case CC_UTF32LE:
		assert(pszDest != 0);
		for(i = 0; i < cchSrc; ++i) {
			pszDest[i * 4 + 0] = pszSrc[i] & 0x00FF;
			pszDest[i * 4 + 1] = (pszSrc[i] & 0xFF00) >> 8;
			pszDest[i * 4 + 2] = 0x0000;
			pszDest[i * 4 + 3] = 0x0000;
		}
		return cchSrc * 4;
	case CC_UTF32BE:
		assert(pszDest != 0);
		for(i = 0; i < cchSrc; ++i) {
			pszDest[i * 4 + 0] = 0x0000;
			pszDest[i * 4 + 1] = 0x0000;
			pszDest[i * 4 + 2] = (pszSrc[i] & 0xFF00) >> 8;
			pszDest[i * 4 + 3] = pszSrc[i] & 0x00FF;
		}
		return cchSrc * 4;
//	case CC_UTF8:
//		return ConvertUnicodeToUTF8(pszDest, pszSrc, cchSrc);
//	case CC_UTF5:
//		return ConvertUnicodeToUTF5(pszDest, pszSrc, cchSrc);
	case CC_SHIFTJIS:
		return ConvertUnicodeToShiftJIS(pszDest, pszSrc, pbUsedDefaultChar, cchSrc);
	case CC_EUCJP:
		pszBuffer = new char[cchSrc * 2];
		cchSrc = ConvertUnicodeToShiftJIS(pszBuffer, pszSrc, pbUsedDefaultChar, cchSrc);
		cchSrc = ConvertShiftJISToEUC(pszDest, reinterpret_cast<const unsigned char*>(pszBuffer), cchSrc);
		delete[] pszBuffer;
		return cchSrc;
	case CC_JIS:
		pszBuffer = new char[cchSrc * 3];
		cchSrc = ConvertUnicodeToShiftJIS(pszBuffer, pszSrc, pbUsedDefaultChar, cchSrc);
		cchSrc = ConvertShiftJISToJIS(pszDest, reinterpret_cast<const unsigned char*>(pszBuffer), cchSrc);
		delete[] pszBuffer;
		return cchSrc;
	}
	assert(false);
	return 0;	// for warning
}

/**
 *	Unicode  Shift-JIS ɕϊ
 *	@param pszDest				ϊ̕
 *	@param pszSrc				ϊ镶
 *	@param pbUsedDefaultChar	Shift-JIS ɖ̑֕ɕϊ
 *	@param cchSrc				ϊ镶̒Bȗ null IƂČvZ
 *	@return						ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertUnicodeToShiftJIS(
		char* pszDest, const char* pszSrc, bool* pbUsedDefaultChar /* = 0 */, std::size_t cchSrc /* = -1 */) {
	std::size_t		iSrc = 0;
	std::size_t		iDest = 0;
	unsigned int	nWBytes;	// wctomb ̖߂l

	assert(pszDest != 0);

	if(pbUsedDefaultChar != 0)
		*pbUsedDefaultChar = false;
	if(cchSrc == -1)
		cchSrc = strlen(pszSrc);

	setlocale(LC_CTYPE, "japanese");
	while(iSrc < cchSrc) {	// ϊ
		nWBytes = wctomb(pszDest + iDest, *(reinterpret_cast<const wchar_t*>(pszSrc + iSrc)));
		if(nWBytes != -1) {
			iSrc += MB_CUR_MAX;
			iDest += nWBytes;
		} else {
			if(pbUsedDefaultChar != 0)
				*pbUsedDefaultChar = true;
			pszDest[iDest] = '?';
			++iDest;
			++iSrc;
		}
	}
	return iDest;
}

/**
 *	Unicode  Shift-JIS ɕϊ (: WideCharToMultiByte)
 *	@param pszDest				ϊ̕
 *	@param pszSrc				ϊ镶
 *	@param pbUsedDefaultChar	Shift-JIS ɖ̑֕ɕϊ
 *	@param cchSrc				ϊ镶̒Bȗ null IƂČvZ
 *	@return						ϊ̒̕B-1Ǝs
 */
inline std::size_t Manah::Text::ConvertUnicodeToShiftJIS(
		char* pszDest, const wchar_t* pszSrc, bool* pbUsedDefaultChar /* = 0 */, std::size_t cchSrc /* = -1 */) {
	std::size_t		iSrc = 0;
	std::size_t		iDest = 0;
	unsigned int	nWBytes;	// wctomb ̖߂l

	assert(pszDest != 0);

	if(pbUsedDefaultChar != 0)
		*pbUsedDefaultChar = false;
	if(cchSrc == -1)
		cchSrc = wcslen(pszSrc);

	setlocale(LC_CTYPE, "japanese");
	while(iSrc < cchSrc) {	// ϊ
		nWBytes = wctomb(pszDest + iDest, pszSrc[iSrc]);
		if(nWBytes != -1) {
			++iSrc;
			iDest += nWBytes;
		} else {
			if(pbUsedDefaultChar != 0)
				*pbUsedDefaultChar = true;
			pszDest[iDest] = '?';
			++iDest;
			++iSrc;
		}
	}
	return iDest;
}

/**
 *	UTF-16 TQ[gyAR[h|Cg쐬
 *	@param pszSrc	TQ[gyA
 *	@param cch		<var>pszSrc</var> ̒
 *	@return			R[h|Cg
 */
inline Manah::Text::CodePoint Manah::Text::DecodeUTF16SurrogatePairToCodePoint(const wchar_t* pszSrc, std::size_t cch) {
	assert(pszSrc != 0 && cch != 0);

	if(cch == 1 || pszSrc[0] < 0xD800 || pszSrc[0] > 0xDBFF || pszSrc[1] < 0xDC00 || pszSrc[1] > 0xDFFF)
		return pszSrc[0];
	else
		return 0x10000 + (pszSrc[0] - 0xD800) * 0x400 + pszSrc[1] - 0xDC00;
}

/**
 *	R[h̔ʁB\̂镶R[hƔ͈ȉ̒ʂ:
 *	<dl>
 *		<dt>UTF-5</dt>
 *		<dd>̕R[hԂ邱Ƃ͂Ȃ</dd>
 *		<dt>UTF-7</dt>
 *		<dd>ĂȂ</dd>
 *		<dt>UTF-8</dt>
 *		<dd>BOM (0xEF 0xBB 0xBF) ϊł</dd>
 *		<dt>UTF-16</dt>
 *		<dd>BOM (0xFF 0xFE) ݂̂ɂ</dd>
 *		<dt>UTF-16 big endian</dt>
 *		<dd>BOM (0xFE 0xFF) ݂̂ɂ</dd>
 *		<dt>UTF-32</dt>
 *		<dd>BOM (0xFF 0xFE 0x00 0x00) ݂̂ɂ</dd>
 *		<dt>UTF-32 big endian</dt>
 *		<dd>BOM (0x00 0x00 0xFE 0xFF) ݂̂ɂ</dd>
 *		<dt>ISO-2022-CN</dt>
 *		<dd>BOM (0x1B 0x24 0x29 0x41) ݂̂ɂ</dd>
 *		<dt>ISO-2022=KR</dt>
 *		<dd>BOM (0x1B 0x24 0x29 0x43) ݂̂ɂ</dd>
 *		<dt>Shift-JIS</dt>
 *		<dd>ϊł</dd>
 *		<dt>JIS</dt>
 *		<dd>ϊł</dd>
 *		<dt>EUC-JP</dt>
 *		<dd>ϊł</dd>
 *	</dl>
 *
 *	@param pszText	ʂ镶
 *	@cch			̒BʂɎg
 *	@return			ʌ
 */
inline Manah::Text::CharCode Manah::Text::DetectCharacterCode(const unsigned char* pszText, std::size_t cch) {
	std::size_t	cchConvertable, cchMax;
	CharCode	ccMax;

/*	cchConvertable = HasBOMOfUTF32(pszText, cch);
	if(cchConvertable == 1)
		return CC_UTF32LE;
	else if(cchConvertable == -1)
		return CC_UTF32BE;

	cchConvertable = HasBOMOfUTF16(pszText, cch);
	if(cchConvertable == 1)
		return CC_UTF16LE;
	else if(cchConvertable == -1)
		return CC_UTF16BE;

	if(HasBOMOfUTF8(pszText, cch))
		return CC_UTF8;
*/
	if(::IsValidCodePage(936) && HasBOMOfISO2022CN(pszText, cch))
		return CC_ISOCN;

	if(::IsValidCodePage(949) && HasBOMOfISO2022KR(pszText, cch))
		return CC_ISOKR;

	if(::IsValidCodePage(932)) {
		cchConvertable = IsShiftJISString(pszText, cch);
		if(cchConvertable == cch)
			return CC_SHIFTJIS;
		ccMax = CC_SHIFTJIS;
		cchMax = cchConvertable;

		cchConvertable = IsEUCString(pszText, cch);
		if(cchConvertable == cch)
			return CC_EUCJP;
		if(cchConvertable > cchMax) {
			ccMax = CC_EUCJP;
			cchMax = cchConvertable;
		}

		cchConvertable = IsJISString(pszText, cch);
		if(cchConvertable == cch)
			return CC_JIS;
		if(cchConvertable > cchMax) {
			ccMax = CC_JIS;
			cchMax = cchConvertable;
		}
	}

//	cchConvertable = IsUTF8String(pszText, cch);
//	if(cchConvertable == cch)
//		return CC_UTF8;
//	if(cchConvertable > cch) {
//		ccMax = CC_UTF8;
//		cchMax = cchConvertable;
//	}

	return ccMax;
}

/**
 *	R[h|Cg UTF-16 TQ[gyA쐬
 *	@param nCodePoint	R[h|Cg
 *	@param pszDest		TQ[gyA
 *	@return				TQ[gyAƂ^
 */
inline bool Manah::Text::EncodeCodePointToUTF16SurrogatePair(Manah::Text::CodePoint cp, wchar_t* pszDest) {
	assert(pszDest != 0);

	if(cp < 0x00010000) {
		pszDest[0] = static_cast<wchar_t>(cp & 0xFFFF);
		return false;
	} else if(cp <= 0x0010FFFF) {
		cp -= 0x00010000;
		pszDest[0] = 0xD800 | static_cast<wchar_t>((cp & 0x001FFC00) >> 10);
		pszDest[1] = 0xDC00 | static_cast<wchar_t>((cp & 0x000003FF) >> 0);
		return true;
	}
	return false;
}

/**
 *	PꋫE (Shift-JIS )
 *	@param pszText		ׂ镶
 *	@param iBegin		Jnʒu
 *	@param bIgnoreSpace	󔒂𖳎
 *	@param bReverse		璲ׂ
 *	@param len			̒
 *	@return				PꋫËʒuBȂꍇ0
 */
inline const unsigned char* Manah::Text::FindWordBorder(const unsigned char* pszText,
		unsigned int iBegin /* = 0 */, bool bIgnoreSpaces /* = true */,
		bool bReverse /* = false */, std::size_t len /* = -1 */) {
	unsigned int	i;
	SJISCharType	ct, ctPrev;

	if(len == -1)
		len = strlen(reinterpret_cast<const char*>(pszText));

	if(!bReverse) {
		ctPrev = GetCharType(pszText + iBegin);
		i = iBegin + ((ctPrev <= CT_KANA_HALFWIDTH) ? 1 : 2);
		while(i < len) {
			ct = GetCharType(pszText + i);
			if(ct == CT_SPACE && bIgnoreSpaces)	// 
				++i;
			else {
				if(((ct & 0xF0) != (ctPrev & 0xF0))
						&& !(pszText[i - 0] == 0x81 && pszText[i + 1] == 0x5B && (ctPrev == CT_KANA_HIRA || ctPrev == CT_KANA_KATA))
						&& !(pszText[i - 2] == 0x81 && pszText[i - 1] == 0x5B && (ct == CT_KANA_HIRA || ct == CT_KANA_KATA))
						&& !(pszText[i - 0] == 0x81 && pszText[i + 1] == 0x58 && (ctPrev & 0xF0) == 0xA0)
						&& !(pszText[i - 2] == 0x81 && pszText[i - 1] == 0x58 && (ct & 0xF0) == 0xA0))
					return pszText + i;
				i += (ct <= CT_KANA_HALFWIDTH) ? 1 : 2;
			}
			ctPrev = ct;
		}
	} else {
		unsigned int	iLast = 0;
		if(iBegin == 0 || iBegin > len)
			return 0;
		ctPrev = GetCharType(pszText);
		i = (ctPrev <= CT_KANA_HALFWIDTH) ? 1 : 2;
		while(i < iBegin) {
			ct = GetCharType(pszText + i);
			if(ct == CT_SPACE && bIgnoreSpaces)	// 
				++i;
			else {
				if((ct & 0xF0) != (ctPrev & 0xF0)
						&& !(pszText[i - 0] == 0x81 && pszText[i + 1] == 0x5B && (ct == CT_KANA_HIRA || ct == CT_KANA_KATA))
						&& !(pszText[i - 2] == 0x81 && pszText[i - 1] == 0x5B && (ctPrev == CT_KANA_HIRA || ctPrev == CT_KANA_KATA))
						&& !(pszText[i - 0] == 0x81 && pszText[i + 1] == 0x58 && (ct & 0xF0) == 0xA0)
						&& !(pszText[i - 2] == 0x81 && pszText[i - 1] == 0x58 && (ctPrev & 0xF0) == 0xA0))
					iLast = i;
				i += (ct <= CT_KANA_HALFWIDTH) ? 1 : 2;
			}
			ctPrev = ct;
		}
		if(iLast != 0)
			return pszText + iLast;
	}
	return 0;
}

/**
 *	PꋫE (Unicode )
 *	@param pszText		ׂ镶
 *	@param iBegin		Jnʒu
 *	@param bIgnoreSpace	󔒂𖳎
 *	@param bReverse		璲ׂ
 *	@param len			̒
 *	@return				PꋫËʒuBȂꍇ0
 */
inline const wchar_t* Manah::Text::FindWordBorder(const wchar_t* pszText,
		unsigned int iBegin /* = 0 */, bool bIgnoreSpaces /* = true */,
		bool bReverse /* = false */, std::size_t len /* = -1 */) {
	unsigned int	i;
	WORD			wCharType, wPrevCharType;
	WORD			wSpace, wPrevSpace;

	if(len == -1)
		len = wcslen(pszText);

	if(!bReverse) {
		if(pszText[iBegin] == L'_')
			wPrevCharType = 0x8040;
		else if(pszText[iBegin] == 0x3005)
			wPrevCharType = 0x8100;
		else
			::GetStringTypeW(CT_CTYPE3, pszText + iBegin, 1, &wPrevCharType);
		::GetStringTypeW(CT_CTYPE1, pszText + iBegin, 1, &wPrevSpace);
		wPrevCharType &= ~(C3_DIACRITIC | C3_VOWELMARK | C3_FULLWIDTH | C3_LEXICAL | C3_ALPHA);
		i = iBegin + 1;
		while(i < len) {
			if(pszText[i] == L'_')
				wCharType = 0x8040;
			else if(pszText[i] == 0x3005)
				wCharType = 0x8100;
			else
				::GetStringTypeW(CT_CTYPE3, pszText + i, 1, &wCharType);
			::GetStringTypeW(CT_CTYPE1, pszText + i, 1, &wSpace);
			wCharType &= ~(C3_DIACRITIC | C3_VOWELMARK | C3_FULLWIDTH | C3_LEXICAL | C3_ALPHA);
			if(bIgnoreSpaces) {
				if(!(wSpace & C1_SPACE)) {
					if(((wCharType != wPrevCharType)
							|| ((wPrevSpace & C1_SPACE) && (wSpace != wPrevSpace)))
							&& !(pszText[i - 0] == 0x30FC && wPrevCharType & 0x0030)
							&& !(pszText[i - 1] == 0x30FC && wCharType & 0x0030))
						return pszText + i;
				}
			} else if(((wCharType != wPrevCharType)
					|| ((wSpace & C1_SPACE || wPrevSpace & C1_SPACE) && (wSpace != wPrevSpace)))
					&& !(pszText[i - 0] == 0x30FC && wPrevCharType & 0x0030)
					&& !(pszText[i - 1] == 0x30FC && wCharType & 0x0030)) {
				return pszText + i;
			}
			++i;
			wPrevCharType = wCharType;
			wPrevSpace = wSpace;
		}
	} else {
		if(iBegin == 0)
			return 0;
		if(pszText[iBegin - 1] == L'_')
			wPrevCharType = 0x8040;
		else if(pszText[iBegin - 1] == 0x3005)
			wPrevCharType = 0x8100;
		else
			::GetStringTypeW(CT_CTYPE3, pszText + iBegin - 1, 1, &wPrevCharType);
		::GetStringTypeW(CT_CTYPE1, pszText + iBegin - 1, 1, &wPrevSpace);
		wPrevCharType &= ~(C3_DIACRITIC | C3_VOWELMARK | C3_FULLWIDTH | C3_LEXICAL | C3_ALPHA);
		i = iBegin - 1;
		while(i != 0) {
			if(pszText[i - 1] == L'_')
				wCharType = 0x8040;
			else if(pszText[iBegin - 1] == 0x3005)
				wCharType = 0x8100;
			else
				::GetStringTypeW(CT_CTYPE3, pszText + i - 1, 1, &wCharType);
			::GetStringTypeW(CT_CTYPE1, pszText + i - 1, 1, &wSpace);
			wCharType &= ~(C3_DIACRITIC | C3_VOWELMARK | C3_FULLWIDTH | C3_LEXICAL | C3_ALPHA);
			if(bIgnoreSpaces) {
				if(!(wSpace & C1_SPACE)) {
					if(((wCharType != wPrevCharType)
							|| ((wPrevSpace & C1_SPACE) && (wSpace != wPrevSpace)))
							&& !(pszText[i - 0] == 0x30FC && wCharType & 0x0030)
							&& !(pszText[i - 1] == 0x30FC && wPrevCharType & 0x0030))
						return pszText + i;
				}
			} else if(((wCharType != wPrevCharType)
					|| ((wSpace & C1_SPACE || wPrevSpace & C1_SPACE) && (wSpace != wPrevSpace)))
					&& !(pszText[i - 0] == 0x30FC && wCharType & 0x0030)
					&& !(pszText[i - 1] == 0x30FC && wPrevCharType & 0x0030)) {
				return pszText + i;
			}
			--i;
			wPrevCharType = wCharType;
			wPrevSpace = wSpace;
		}
	}
	return 0;
}

///	啶ʂɔrł悤1ϊ
/// (CaseFolding.txt ɂ莩)
inline Manah::Text::CodePoint Manah::Text::FoldCaseSimple(Manah::Text::CodePoint cp) {
	switch(cp) {
	case 0x0041: return 0x0061;	case 0x0042: return 0x0062;
	case 0x0043: return 0x0063;	case 0x0044: return 0x0064;
	case 0x0045: return 0x0065;	case 0x0046: return 0x0066;
	case 0x0047: return 0x0067;	case 0x0048: return 0x0068;
	case 0x0049: return 0x0069;	case 0x004A: return 0x006A;
	case 0x004B: return 0x006B;	case 0x004C: return 0x006C;
	case 0x004D: return 0x006D;	case 0x004E: return 0x006E;
	case 0x004F: return 0x006F;	case 0x0050: return 0x0070;
	case 0x0051: return 0x0071;	case 0x0052: return 0x0072;
	case 0x0053: return 0x0073;	case 0x0054: return 0x0074;
	case 0x0055: return 0x0075;	case 0x0056: return 0x0076;
	case 0x0057: return 0x0077;	case 0x0058: return 0x0078;
	case 0x0059: return 0x0079;	case 0x005A: return 0x007A;
	case 0x00B5: return 0x03BC;	case 0x00C0: return 0x00E0;
	case 0x00C1: return 0x00E1;	case 0x00C2: return 0x00E2;
	case 0x00C3: return 0x00E3;	case 0x00C4: return 0x00E4;
	case 0x00C5: return 0x00E5;	case 0x00C6: return 0x00E6;
	case 0x00C7: return 0x00E7;	case 0x00C8: return 0x00E8;
	case 0x00C9: return 0x00E9;	case 0x00CA: return 0x00EA;
	case 0x00CB: return 0x00EB;	case 0x00CC: return 0x00EC;
	case 0x00CD: return 0x00ED;	case 0x00CE: return 0x00EE;
	case 0x00CF: return 0x00EF;	case 0x00D0: return 0x00F0;
	case 0x00D1: return 0x00F1;	case 0x00D2: return 0x00F2;
	case 0x00D3: return 0x00F3;	case 0x00D4: return 0x00F4;
	case 0x00D5: return 0x00F5;	case 0x00D6: return 0x00F6;
	case 0x00D8: return 0x00F8;	case 0x00D9: return 0x00F9;
	case 0x00DA: return 0x00FA;	case 0x00DB: return 0x00FB;
	case 0x00DC: return 0x00FC;	case 0x00DD: return 0x00FD;
	case 0x00DE: return 0x00FE;	case 0x0100: return 0x0101;
	case 0x0102: return 0x0103;	case 0x0104: return 0x0105;
	case 0x0106: return 0x0107;	case 0x0108: return 0x0109;
	case 0x010A: return 0x010B;	case 0x010C: return 0x010D;
	case 0x010E: return 0x010F;	case 0x0110: return 0x0111;
	case 0x0112: return 0x0113;	case 0x0114: return 0x0115;
	case 0x0116: return 0x0117;	case 0x0118: return 0x0119;
	case 0x011A: return 0x011B;	case 0x011C: return 0x011D;
	case 0x011E: return 0x011F;	case 0x0120: return 0x0121;
	case 0x0122: return 0x0123;	case 0x0124: return 0x0125;
	case 0x0126: return 0x0127;	case 0x0128: return 0x0129;
	case 0x012A: return 0x012B;	case 0x012C: return 0x012D;
	case 0x012E: return 0x012F;	case 0x0132: return 0x0133;
	case 0x0134: return 0x0135;	case 0x0136: return 0x0137;
	case 0x0139: return 0x013A;	case 0x013B: return 0x013C;
	case 0x013D: return 0x013E;	case 0x013F: return 0x0140;
	case 0x0141: return 0x0142;	case 0x0143: return 0x0144;
	case 0x0145: return 0x0146;	case 0x0147: return 0x0148;
	case 0x014A: return 0x014B;	case 0x014C: return 0x014D;
	case 0x014E: return 0x014F;	case 0x0150: return 0x0151;
	case 0x0152: return 0x0153;	case 0x0154: return 0x0155;
	case 0x0156: return 0x0157;	case 0x0158: return 0x0159;
	case 0x015A: return 0x015B;	case 0x015C: return 0x015D;
	case 0x015E: return 0x015F;	case 0x0160: return 0x0161;
	case 0x0162: return 0x0163;	case 0x0164: return 0x0165;
	case 0x0166: return 0x0167;	case 0x0168: return 0x0169;
	case 0x016A: return 0x016B;	case 0x016C: return 0x016D;
	case 0x016E: return 0x016F;	case 0x0170: return 0x0171;
	case 0x0172: return 0x0173;	case 0x0174: return 0x0175;
	case 0x0176: return 0x0177;	case 0x0178: return 0x00FF;
	case 0x0179: return 0x017A;	case 0x017B: return 0x017C;
	case 0x017D: return 0x017E;	case 0x017F: return 0x0073;
	case 0x0181: return 0x0253;	case 0x0182: return 0x0183;
	case 0x0184: return 0x0185;	case 0x0186: return 0x0254;
	case 0x0187: return 0x0188;	case 0x0189: return 0x0256;
	case 0x018A: return 0x0257;	case 0x018B: return 0x018C;
	case 0x018E: return 0x01DD;	case 0x018F: return 0x0259;
	case 0x0190: return 0x025B;	case 0x0191: return 0x0192;
	case 0x0193: return 0x0260;	case 0x0194: return 0x0263;
	case 0x0196: return 0x0269;	case 0x0197: return 0x0268;
	case 0x0198: return 0x0199;	case 0x019C: return 0x026F;
	case 0x019D: return 0x0272;	case 0x019F: return 0x0275;
	case 0x01A0: return 0x01A1;	case 0x01A2: return 0x01A3;
	case 0x01A4: return 0x01A5;	case 0x01A6: return 0x0280;
	case 0x01A7: return 0x01A8;	case 0x01A9: return 0x0283;
	case 0x01AC: return 0x01AD;	case 0x01AE: return 0x0288;
	case 0x01AF: return 0x01B0;	case 0x01B1: return 0x028A;
	case 0x01B2: return 0x028B;	case 0x01B3: return 0x01B4;
	case 0x01B5: return 0x01B6;	case 0x01B7: return 0x0292;
	case 0x01B8: return 0x01B9;	case 0x01BC: return 0x01BD;
	case 0x01C4: return 0x01C6;	case 0x01C5: return 0x01C6;
	case 0x01C7: return 0x01C9;	case 0x01C8: return 0x01C9;
	case 0x01CA: return 0x01CC;	case 0x01CB: return 0x01CC;
	case 0x01CD: return 0x01CE;	case 0x01CF: return 0x01D0;
	case 0x01D1: return 0x01D2;	case 0x01D3: return 0x01D4;
	case 0x01D5: return 0x01D6;	case 0x01D7: return 0x01D8;
	case 0x01D9: return 0x01DA;	case 0x01DB: return 0x01DC;
	case 0x01DE: return 0x01DF;	case 0x01E0: return 0x01E1;
	case 0x01E2: return 0x01E3;	case 0x01E4: return 0x01E5;
	case 0x01E6: return 0x01E7;	case 0x01E8: return 0x01E9;
	case 0x01EA: return 0x01EB;	case 0x01EC: return 0x01ED;
	case 0x01EE: return 0x01EF;	case 0x01F1: return 0x01F3;
	case 0x01F2: return 0x01F3;	case 0x01F4: return 0x01F5;
	case 0x01F6: return 0x0195;	case 0x01F7: return 0x01BF;
	case 0x01F8: return 0x01F9;	case 0x01FA: return 0x01FB;
	case 0x01FC: return 0x01FD;	case 0x01FE: return 0x01FF;
	case 0x0200: return 0x0201;	case 0x0202: return 0x0203;
	case 0x0204: return 0x0205;	case 0x0206: return 0x0207;
	case 0x0208: return 0x0209;	case 0x020A: return 0x020B;
	case 0x020C: return 0x020D;	case 0x020E: return 0x020F;
	case 0x0210: return 0x0211;	case 0x0212: return 0x0213;
	case 0x0214: return 0x0215;	case 0x0216: return 0x0217;
	case 0x0218: return 0x0219;	case 0x021A: return 0x021B;
	case 0x021C: return 0x021D;	case 0x021E: return 0x021F;
	case 0x0220: return 0x019E;	case 0x0222: return 0x0223;
	case 0x0224: return 0x0225;	case 0x0226: return 0x0227;
	case 0x0228: return 0x0229;	case 0x022A: return 0x022B;
	case 0x022C: return 0x022D;	case 0x022E: return 0x022F;
	case 0x0230: return 0x0231;	case 0x0232: return 0x0233;
	case 0x0345: return 0x03B9;	case 0x0386: return 0x03AC;
	case 0x0388: return 0x03AD;	case 0x0389: return 0x03AE;
	case 0x038A: return 0x03AF;	case 0x038C: return 0x03CC;
	case 0x038E: return 0x03CD;	case 0x038F: return 0x03CE;
	case 0x0391: return 0x03B1;	case 0x0392: return 0x03B2;
	case 0x0393: return 0x03B3;	case 0x0394: return 0x03B4;
	case 0x0395: return 0x03B5;	case 0x0396: return 0x03B6;
	case 0x0397: return 0x03B7;	case 0x0398: return 0x03B8;
	case 0x0399: return 0x03B9;	case 0x039A: return 0x03BA;
	case 0x039B: return 0x03BB;	case 0x039C: return 0x03BC;
	case 0x039D: return 0x03BD;	case 0x039E: return 0x03BE;
	case 0x039F: return 0x03BF;	case 0x03A0: return 0x03C0;
	case 0x03A1: return 0x03C1;	case 0x03A3: return 0x03C3;
	case 0x03A4: return 0x03C4;	case 0x03A5: return 0x03C5;
	case 0x03A6: return 0x03C6;	case 0x03A7: return 0x03C7;
	case 0x03A8: return 0x03C8;	case 0x03A9: return 0x03C9;
	case 0x03AA: return 0x03CA;	case 0x03AB: return 0x03CB;
	case 0x03C2: return 0x03C3;	case 0x03D0: return 0x03B2;
	case 0x03D1: return 0x03B8;	case 0x03D5: return 0x03C6;
	case 0x03D6: return 0x03C0;	case 0x03D8: return 0x03D9;
	case 0x03DA: return 0x03DB;	case 0x03DC: return 0x03DD;
	case 0x03DE: return 0x03DF;	case 0x03E0: return 0x03E1;
	case 0x03E2: return 0x03E3;	case 0x03E4: return 0x03E5;
	case 0x03E6: return 0x03E7;	case 0x03E8: return 0x03E9;
	case 0x03EA: return 0x03EB;	case 0x03EC: return 0x03ED;
	case 0x03EE: return 0x03EF;	case 0x03F0: return 0x03BA;
	case 0x03F1: return 0x03C1;	case 0x03F4: return 0x03B8;
	case 0x03F5: return 0x03B5;	case 0x03F7: return 0x03F8;
	case 0x03F9: return 0x03F2;	case 0x03FA: return 0x03FB;
	case 0x0400: return 0x0450;	case 0x0401: return 0x0451;
	case 0x0402: return 0x0452;	case 0x0403: return 0x0453;
	case 0x0404: return 0x0454;	case 0x0405: return 0x0455;
	case 0x0406: return 0x0456;	case 0x0407: return 0x0457;
	case 0x0408: return 0x0458;	case 0x0409: return 0x0459;
	case 0x040A: return 0x045A;	case 0x040B: return 0x045B;
	case 0x040C: return 0x045C;	case 0x040D: return 0x045D;
	case 0x040E: return 0x045E;	case 0x040F: return 0x045F;
	case 0x0410: return 0x0430;	case 0x0411: return 0x0431;
	case 0x0412: return 0x0432;	case 0x0413: return 0x0433;
	case 0x0414: return 0x0434;	case 0x0415: return 0x0435;
	case 0x0416: return 0x0436;	case 0x0417: return 0x0437;
	case 0x0418: return 0x0438;	case 0x0419: return 0x0439;
	case 0x041A: return 0x043A;	case 0x041B: return 0x043B;
	case 0x041C: return 0x043C;	case 0x041D: return 0x043D;
	case 0x041E: return 0x043E;	case 0x041F: return 0x043F;
	case 0x0420: return 0x0440;	case 0x0421: return 0x0441;
	case 0x0422: return 0x0442;	case 0x0423: return 0x0443;
	case 0x0424: return 0x0444;	case 0x0425: return 0x0445;
	case 0x0426: return 0x0446;	case 0x0427: return 0x0447;
	case 0x0428: return 0x0448;	case 0x0429: return 0x0449;
	case 0x042A: return 0x044A;	case 0x042B: return 0x044B;
	case 0x042C: return 0x044C;	case 0x042D: return 0x044D;
	case 0x042E: return 0x044E;	case 0x042F: return 0x044F;
	case 0x0460: return 0x0461;	case 0x0462: return 0x0463;
	case 0x0464: return 0x0465;	case 0x0466: return 0x0467;
	case 0x0468: return 0x0469;	case 0x046A: return 0x046B;
	case 0x046C: return 0x046D;	case 0x046E: return 0x046F;
	case 0x0470: return 0x0471;	case 0x0472: return 0x0473;
	case 0x0474: return 0x0475;	case 0x0476: return 0x0477;
	case 0x0478: return 0x0479;	case 0x047A: return 0x047B;
	case 0x047C: return 0x047D;	case 0x047E: return 0x047F;
	case 0x0480: return 0x0481;	case 0x048A: return 0x048B;
	case 0x048C: return 0x048D;	case 0x048E: return 0x048F;
	case 0x0490: return 0x0491;	case 0x0492: return 0x0493;
	case 0x0494: return 0x0495;	case 0x0496: return 0x0497;
	case 0x0498: return 0x0499;	case 0x049A: return 0x049B;
	case 0x049C: return 0x049D;	case 0x049E: return 0x049F;
	case 0x04A0: return 0x04A1;	case 0x04A2: return 0x04A3;
	case 0x04A4: return 0x04A5;	case 0x04A6: return 0x04A7;
	case 0x04A8: return 0x04A9;	case 0x04AA: return 0x04AB;
	case 0x04AC: return 0x04AD;	case 0x04AE: return 0x04AF;
	case 0x04B0: return 0x04B1;	case 0x04B2: return 0x04B3;
	case 0x04B4: return 0x04B5;	case 0x04B6: return 0x04B7;
	case 0x04B8: return 0x04B9;	case 0x04BA: return 0x04BB;
	case 0x04BC: return 0x04BD;	case 0x04BE: return 0x04BF;
	case 0x04C1: return 0x04C2;	case 0x04C3: return 0x04C4;
	case 0x04C5: return 0x04C6;	case 0x04C7: return 0x04C8;
	case 0x04C9: return 0x04CA;	case 0x04CB: return 0x04CC;
	case 0x04CD: return 0x04CE;	case 0x04D0: return 0x04D1;
	case 0x04D2: return 0x04D3;	case 0x04D4: return 0x04D5;
	case 0x04D6: return 0x04D7;	case 0x04D8: return 0x04D9;
	case 0x04DA: return 0x04DB;	case 0x04DC: return 0x04DD;
	case 0x04DE: return 0x04DF;	case 0x04E0: return 0x04E1;
	case 0x04E2: return 0x04E3;	case 0x04E4: return 0x04E5;
	case 0x04E6: return 0x04E7;	case 0x04E8: return 0x04E9;
	case 0x04EA: return 0x04EB;	case 0x04EC: return 0x04ED;
	case 0x04EE: return 0x04EF;	case 0x04F0: return 0x04F1;
	case 0x04F2: return 0x04F3;	case 0x04F4: return 0x04F5;
	case 0x04F8: return 0x04F9;	case 0x0500: return 0x0501;
	case 0x0502: return 0x0503;	case 0x0504: return 0x0505;
	case 0x0506: return 0x0507;	case 0x0508: return 0x0509;
	case 0x050A: return 0x050B;	case 0x050C: return 0x050D;
	case 0x050E: return 0x050F;	case 0x0531: return 0x0561;
	case 0x0532: return 0x0562;	case 0x0533: return 0x0563;
	case 0x0534: return 0x0564;	case 0x0535: return 0x0565;
	case 0x0536: return 0x0566;	case 0x0537: return 0x0567;
	case 0x0538: return 0x0568;	case 0x0539: return 0x0569;
	case 0x053A: return 0x056A;	case 0x053B: return 0x056B;
	case 0x053C: return 0x056C;	case 0x053D: return 0x056D;
	case 0x053E: return 0x056E;	case 0x053F: return 0x056F;
	case 0x0540: return 0x0570;	case 0x0541: return 0x0571;
	case 0x0542: return 0x0572;	case 0x0543: return 0x0573;
	case 0x0544: return 0x0574;	case 0x0545: return 0x0575;
	case 0x0546: return 0x0576;	case 0x0547: return 0x0577;
	case 0x0548: return 0x0578;	case 0x0549: return 0x0579;
	case 0x054A: return 0x057A;	case 0x054B: return 0x057B;
	case 0x054C: return 0x057C;	case 0x054D: return 0x057D;
	case 0x054E: return 0x057E;	case 0x054F: return 0x057F;
	case 0x0550: return 0x0580;	case 0x0551: return 0x0581;
	case 0x0552: return 0x0582;	case 0x0553: return 0x0583;
	case 0x0554: return 0x0584;	case 0x0555: return 0x0585;
	case 0x0556: return 0x0586;	case 0x1E00: return 0x1E01;
	case 0x1E02: return 0x1E03;	case 0x1E04: return 0x1E05;
	case 0x1E06: return 0x1E07;	case 0x1E08: return 0x1E09;
	case 0x1E0A: return 0x1E0B;	case 0x1E0C: return 0x1E0D;
	case 0x1E0E: return 0x1E0F;	case 0x1E10: return 0x1E11;
	case 0x1E12: return 0x1E13;	case 0x1E14: return 0x1E15;
	case 0x1E16: return 0x1E17;	case 0x1E18: return 0x1E19;
	case 0x1E1A: return 0x1E1B;	case 0x1E1C: return 0x1E1D;
	case 0x1E1E: return 0x1E1F;	case 0x1E20: return 0x1E21;
	case 0x1E22: return 0x1E23;	case 0x1E24: return 0x1E25;
	case 0x1E26: return 0x1E27;	case 0x1E28: return 0x1E29;
	case 0x1E2A: return 0x1E2B;	case 0x1E2C: return 0x1E2D;
	case 0x1E2E: return 0x1E2F;	case 0x1E30: return 0x1E31;
	case 0x1E32: return 0x1E33;	case 0x1E34: return 0x1E35;
	case 0x1E36: return 0x1E37;	case 0x1E38: return 0x1E39;
	case 0x1E3A: return 0x1E3B;	case 0x1E3C: return 0x1E3D;
	case 0x1E3E: return 0x1E3F;	case 0x1E40: return 0x1E41;
	case 0x1E42: return 0x1E43;	case 0x1E44: return 0x1E45;
	case 0x1E46: return 0x1E47;	case 0x1E48: return 0x1E49;
	case 0x1E4A: return 0x1E4B;	case 0x1E4C: return 0x1E4D;
	case 0x1E4E: return 0x1E4F;	case 0x1E50: return 0x1E51;
	case 0x1E52: return 0x1E53;	case 0x1E54: return 0x1E55;
	case 0x1E56: return 0x1E57;	case 0x1E58: return 0x1E59;
	case 0x1E5A: return 0x1E5B;	case 0x1E5C: return 0x1E5D;
	case 0x1E5E: return 0x1E5F;	case 0x1E60: return 0x1E61;
	case 0x1E62: return 0x1E63;	case 0x1E64: return 0x1E65;
	case 0x1E66: return 0x1E67;	case 0x1E68: return 0x1E69;
	case 0x1E6A: return 0x1E6B;	case 0x1E6C: return 0x1E6D;
	case 0x1E6E: return 0x1E6F;	case 0x1E70: return 0x1E71;
	case 0x1E72: return 0x1E73;	case 0x1E74: return 0x1E75;
	case 0x1E76: return 0x1E77;	case 0x1E78: return 0x1E79;
	case 0x1E7A: return 0x1E7B;	case 0x1E7C: return 0x1E7D;
	case 0x1E7E: return 0x1E7F;	case 0x1E80: return 0x1E81;
	case 0x1E82: return 0x1E83;	case 0x1E84: return 0x1E85;
	case 0x1E86: return 0x1E87;	case 0x1E88: return 0x1E89;
	case 0x1E8A: return 0x1E8B;	case 0x1E8C: return 0x1E8D;
	case 0x1E8E: return 0x1E8F;	case 0x1E90: return 0x1E91;
	case 0x1E92: return 0x1E93;	case 0x1E94: return 0x1E95;
	case 0x1E9B: return 0x1E61;	case 0x1EA0: return 0x1EA1;
	case 0x1EA2: return 0x1EA3;	case 0x1EA4: return 0x1EA5;
	case 0x1EA6: return 0x1EA7;	case 0x1EA8: return 0x1EA9;
	case 0x1EAA: return 0x1EAB;	case 0x1EAC: return 0x1EAD;
	case 0x1EAE: return 0x1EAF;	case 0x1EB0: return 0x1EB1;
	case 0x1EB2: return 0x1EB3;	case 0x1EB4: return 0x1EB5;
	case 0x1EB6: return 0x1EB7;	case 0x1EB8: return 0x1EB9;
	case 0x1EBA: return 0x1EBB;	case 0x1EBC: return 0x1EBD;
	case 0x1EBE: return 0x1EBF;	case 0x1EC0: return 0x1EC1;
	case 0x1EC2: return 0x1EC3;	case 0x1EC4: return 0x1EC5;
	case 0x1EC6: return 0x1EC7;	case 0x1EC8: return 0x1EC9;
	case 0x1ECA: return 0x1ECB;	case 0x1ECC: return 0x1ECD;
	case 0x1ECE: return 0x1ECF;	case 0x1ED0: return 0x1ED1;
	case 0x1ED2: return 0x1ED3;	case 0x1ED4: return 0x1ED5;
	case 0x1ED6: return 0x1ED7;	case 0x1ED8: return 0x1ED9;
	case 0x1EDA: return 0x1EDB;	case 0x1EDC: return 0x1EDD;
	case 0x1EDE: return 0x1EDF;	case 0x1EE0: return 0x1EE1;
	case 0x1EE2: return 0x1EE3;	case 0x1EE4: return 0x1EE5;
	case 0x1EE6: return 0x1EE7;	case 0x1EE8: return 0x1EE9;
	case 0x1EEA: return 0x1EEB;	case 0x1EEC: return 0x1EED;
	case 0x1EEE: return 0x1EEF;	case 0x1EF0: return 0x1EF1;
	case 0x1EF2: return 0x1EF3;	case 0x1EF4: return 0x1EF5;
	case 0x1EF6: return 0x1EF7;	case 0x1EF8: return 0x1EF9;
	case 0x1F08: return 0x1F00;	case 0x1F09: return 0x1F01;
	case 0x1F0A: return 0x1F02;	case 0x1F0B: return 0x1F03;
	case 0x1F0C: return 0x1F04;	case 0x1F0D: return 0x1F05;
	case 0x1F0E: return 0x1F06;	case 0x1F0F: return 0x1F07;
	case 0x1F18: return 0x1F10;	case 0x1F19: return 0x1F11;
	case 0x1F1A: return 0x1F12;	case 0x1F1B: return 0x1F13;
	case 0x1F1C: return 0x1F14;	case 0x1F1D: return 0x1F15;
	case 0x1F28: return 0x1F20;	case 0x1F29: return 0x1F21;
	case 0x1F2A: return 0x1F22;	case 0x1F2B: return 0x1F23;
	case 0x1F2C: return 0x1F24;	case 0x1F2D: return 0x1F25;
	case 0x1F2E: return 0x1F26;	case 0x1F2F: return 0x1F27;
	case 0x1F38: return 0x1F30;	case 0x1F39: return 0x1F31;
	case 0x1F3A: return 0x1F32;	case 0x1F3B: return 0x1F33;
	case 0x1F3C: return 0x1F34;	case 0x1F3D: return 0x1F35;
	case 0x1F3E: return 0x1F36;	case 0x1F3F: return 0x1F37;
	case 0x1F48: return 0x1F40;	case 0x1F49: return 0x1F41;
	case 0x1F4A: return 0x1F42;	case 0x1F4B: return 0x1F43;
	case 0x1F4C: return 0x1F44;	case 0x1F4D: return 0x1F45;
	case 0x1F59: return 0x1F51;	case 0x1F5B: return 0x1F53;
	case 0x1F5D: return 0x1F55;	case 0x1F5F: return 0x1F57;
	case 0x1F68: return 0x1F60;	case 0x1F69: return 0x1F61;
	case 0x1F6A: return 0x1F62;	case 0x1F6B: return 0x1F63;
	case 0x1F6C: return 0x1F64;	case 0x1F6D: return 0x1F65;
	case 0x1F6E: return 0x1F66;	case 0x1F6F: return 0x1F67;
	case 0x1F88: return 0x1F80;	case 0x1F89: return 0x1F81;
	case 0x1F8A: return 0x1F82;	case 0x1F8B: return 0x1F83;
	case 0x1F8C: return 0x1F84;	case 0x1F8D: return 0x1F85;
	case 0x1F8E: return 0x1F86;	case 0x1F8F: return 0x1F87;
	case 0x1F98: return 0x1F90;	case 0x1F99: return 0x1F91;
	case 0x1F9A: return 0x1F92;	case 0x1F9B: return 0x1F93;
	case 0x1F9C: return 0x1F94;	case 0x1F9D: return 0x1F95;
	case 0x1F9E: return 0x1F96;	case 0x1F9F: return 0x1F97;
	case 0x1FA8: return 0x1FA0;	case 0x1FA9: return 0x1FA1;
	case 0x1FAA: return 0x1FA2;	case 0x1FAB: return 0x1FA3;
	case 0x1FAC: return 0x1FA4;	case 0x1FAD: return 0x1FA5;
	case 0x1FAE: return 0x1FA6;	case 0x1FAF: return 0x1FA7;
	case 0x1FB8: return 0x1FB0;	case 0x1FB9: return 0x1FB1;
	case 0x1FBA: return 0x1F70;	case 0x1FBB: return 0x1F71;
	case 0x1FBC: return 0x1FB3;	case 0x1FBE: return 0x03B9;
	case 0x1FC8: return 0x1F72;	case 0x1FC9: return 0x1F73;
	case 0x1FCA: return 0x1F74;	case 0x1FCB: return 0x1F75;
	case 0x1FCC: return 0x1FC3;	case 0x1FD8: return 0x1FD0;
	case 0x1FD9: return 0x1FD1;	case 0x1FDA: return 0x1F76;
	case 0x1FDB: return 0x1F77;	case 0x1FE8: return 0x1FE0;
	case 0x1FE9: return 0x1FE1;	case 0x1FEA: return 0x1F7A;
	case 0x1FEB: return 0x1F7B;	case 0x1FEC: return 0x1FE5;
	case 0x1FF8: return 0x1F78;	case 0x1FF9: return 0x1F79;
	case 0x1FFA: return 0x1F7C;	case 0x1FFB: return 0x1F7D;
	case 0x1FFC: return 0x1FF3;	case 0x2126: return 0x03C9;
	case 0x212A: return 0x006B;	case 0x212B: return 0x00E5;
	case 0x2160: return 0x2170;	case 0x2161: return 0x2171;
	case 0x2162: return 0x2172;	case 0x2163: return 0x2173;
	case 0x2164: return 0x2174;	case 0x2165: return 0x2175;
	case 0x2166: return 0x2176;	case 0x2167: return 0x2177;
	case 0x2168: return 0x2178;	case 0x2169: return 0x2179;
	case 0x216A: return 0x217A;	case 0x216B: return 0x217B;
	case 0x216C: return 0x217C;	case 0x216D: return 0x217D;
	case 0x216E: return 0x217E;	case 0x216F: return 0x217F;
	case 0x24B6: return 0x24D0;	case 0x24B7: return 0x24D1;
	case 0x24B8: return 0x24D2;	case 0x24B9: return 0x24D3;
	case 0x24BA: return 0x24D4;	case 0x24BB: return 0x24D5;
	case 0x24BC: return 0x24D6;	case 0x24BD: return 0x24D7;
	case 0x24BE: return 0x24D8;	case 0x24BF: return 0x24D9;
	case 0x24C0: return 0x24DA;	case 0x24C1: return 0x24DB;
	case 0x24C2: return 0x24DC;	case 0x24C3: return 0x24DD;
	case 0x24C4: return 0x24DE;	case 0x24C5: return 0x24DF;
	case 0x24C6: return 0x24E0;	case 0x24C7: return 0x24E1;
	case 0x24C8: return 0x24E2;	case 0x24C9: return 0x24E3;
	case 0x24CA: return 0x24E4;	case 0x24CB: return 0x24E5;
	case 0x24CC: return 0x24E6;	case 0x24CD: return 0x24E7;
	case 0x24CE: return 0x24E8;	case 0x24CF: return 0x24E9;
	case 0xFF21: return 0xFF41;	case 0xFF22: return 0xFF42;
	case 0xFF23: return 0xFF43;	case 0xFF24: return 0xFF44;
	case 0xFF25: return 0xFF45;	case 0xFF26: return 0xFF46;
	case 0xFF27: return 0xFF47;	case 0xFF28: return 0xFF48;
	case 0xFF29: return 0xFF49;	case 0xFF2A: return 0xFF4A;
	case 0xFF2B: return 0xFF4B;	case 0xFF2C: return 0xFF4C;
	case 0xFF2D: return 0xFF4D;	case 0xFF2E: return 0xFF4E;
	case 0xFF2F: return 0xFF4F;	case 0xFF30: return 0xFF50;
	case 0xFF31: return 0xFF51;	case 0xFF32: return 0xFF52;
	case 0xFF33: return 0xFF53;	case 0xFF34: return 0xFF54;
	case 0xFF35: return 0xFF55;	case 0xFF36: return 0xFF56;
	case 0xFF37: return 0xFF57;	case 0xFF38: return 0xFF58;
	case 0xFF39: return 0xFF59;	case 0xFF3A: return 0xFF5A;
	case 0x10400: return 0x10428;	case 0x10401: return 0x10429;
	case 0x10402: return 0x1042A;	case 0x10403: return 0x1042B;
	case 0x10404: return 0x1042C;	case 0x10405: return 0x1042D;
	case 0x10406: return 0x1042E;	case 0x10407: return 0x1042F;
	case 0x10408: return 0x10430;	case 0x10409: return 0x10431;
	case 0x1040A: return 0x10432;	case 0x1040B: return 0x10433;
	case 0x1040C: return 0x10434;	case 0x1040D: return 0x10435;
	case 0x1040E: return 0x10436;	case 0x1040F: return 0x10437;
	case 0x10410: return 0x10438;	case 0x10411: return 0x10439;
	case 0x10412: return 0x1043A;	case 0x10413: return 0x1043B;
	case 0x10414: return 0x1043C;	case 0x10415: return 0x1043D;
	case 0x10416: return 0x1043E;	case 0x10417: return 0x1043F;
	case 0x10418: return 0x10440;	case 0x10419: return 0x10441;
	case 0x1041A: return 0x10442;	case 0x1041B: return 0x10443;
	case 0x1041C: return 0x10444;	case 0x1041D: return 0x10445;
	case 0x1041E: return 0x10446;	case 0x1041F: return 0x10447;
	case 0x10420: return 0x10448;	case 0x10421: return 0x10449;
	case 0x10422: return 0x1044A;	case 0x10423: return 0x1044B;
	case 0x10424: return 0x1044C;	case 0x10425: return 0x1044D;
	case 0x10426: return 0x1044E;	case 0x10427: return 0x1044F;
	default: return cp;
	}
}

///	Shift-JIS ̃^CvԂ
inline Manah::Text::SJISCharType Manah::Text::GetCharType(const unsigned char* pszText) {
	unsigned char	ch = pszText[0];
	unsigned char	chT = ((ch != 0) ? pszText[1] : 0);
	unsigned short	chKanji = (ch << 8) | chT;

	if((ch >= 0x09 && ch <= 0x0D) || ch == 0x20)
		return CT_SPACE;
	else if(ch < 0x20 || ch == 0x7F)
		return CT_CONTROL;
	else if(ch == 0x5F)
		return CT_UNDERSCORE;
	else if(ch < 0x30 || (ch >= 0x3A && ch <= 0x40) || (ch >= 0x5B && ch <= 0x60) || (ch >= 0x7B && ch <= 0x7E))
		return CT_SYMBOL;
	else if(ch < 0x40)
		return CT_DIGIT;
	else if(ch <= 0x5A)
		return CT_UPPERALPHA;
	else if(ch <= 0x7A)
		return CT_LOWERALPHA;
	else if(ch >= 0xA1 && ch <= 0xDF)
		return CT_KANA_HALFWIDTH;
	else if(ch == 0x81 && (chT >= 0x40 && chT <= 0xFC)) {
		if(chT == 0x52 || chT == 0x53)
			return CT_KANA_KATA;
		else if(chT == 0x54 || chT == 0x55)
			return CT_KANA_HIRA;
		else
			return CT_KANA_SYMBOL;
	}
	if(chT < 0x40 || chT > 0xFC || chT == 0x7F)
		return CT_OTHER;
	else if(ch == 0x82) {
		if(chT >= 0x4F && chT <= 0x58)	return CT_FULLWIDTH_DIGIT;
		if(chT >= 0x60 && chT <= 0x79)	return CT_FULLWIDTH_UPPERALPHA;
		if(chT >= 0x81 && chT <= 0x9A)	return CT_FULLWIDTH_LOWERALPHA;
		if(chT >= 0x9F && chT <= 0xF1)	return CT_KANA_HIRA;
	} else if(ch == 0x83) {
		if(chT >= 0x40 && chT <= 0x96)	return CT_KANA_KATA;
		if(chT >= 0x9F && chT <= 0xB6)	return CT_GREEK_UPPER;
		if(chT >= 0xBF && chT <= 0xD6)	return CT_GREEK_LOWER;
	} else if(ch == 0x84) {
		if(chT >= 0x40 && chT <= 0x60)	return CT_CYRILLIC_UPPER;
		if(chT >= 0x70 && chT <= 0x91)	return CT_CYRILLIC_LOWER;
	} else if(ch == 0x87) {
		if(chT >= 0x40 && chT <= 0x53)	return CT_SPECIAL_BOXDRAWING;
		if(chT >= 0x54 && chT <= 0x5D)	return CT_SPECIAL_ROMANNUMERAL;
		if(chT >= 0x5F && chT <= 0x75)	return CT_SPECIAL_UNIT;
		if(chT == 0x93 || chT == 0x94 || chT == 0x98 || chT == 0x99)	return CT_SPECIAL_MATHEMATICAL;
		if(chT >= 0x7E && chT <= 0x8F)	return CT_SPECIAL_OTHER;
	} else if(ch == 0xFA) {
		if(chT >= 0x55 && chT <= 0x57)	return CT_KANA_SYMBOL;
		if(chT >= 0x40 && chT <= 0x49)	return CT_SPECIAL_ROMANNUMERAL;
	}
	if(chKanji >= 0x889F && chKanji <= 0x9872)	return CT_KANJI1;
	if(chKanji >= 0x989F && chKanji <= 0xEA9E)	return CT_KANJI2;
	if(chKanji >= 0xFA5C && chKanji <= 0xFC4B)	return CT_KANJI3;
	return CT_OTHER;
}

///	w肵R[h|Cg Script vpeBԂ
inline Manah::Text::UnicodeScriptProperty Manah::Text::GetScriptProperty(Manah::Text::CodePoint cp) {
#define IF_CODEPOINT_IS_IN_RANGE(min_, max_)	\
	if(cp >= min_ && cp <= max_)

	IF_CODEPOINT_IS_IN_RANGE(0x0041, 0x005A)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x0061, 0x007A)	return USP_LATIN;
	if(cp == 0x00AA || cp == 0x00BA)			return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x00C0, 0x00D6)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x00D8, 0x00F6)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x00F8, 0x0236)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x0250, 0x02B8)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x02E0, 0x02E4)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x1D00, 0x1D25)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x1D2C, 0x1D5C)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x1D62, 0x1D65)	return USP_LATIN;
	if(cp == 0x1D6B)							return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x1E00, 0x1E9B)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0x1EA0, 0x1EF9)	return USP_LATIN;
	if(cp == 0x2071 || cp == 0x207F)			return USP_LATIN;
	if(cp == 0x212A || cp == 0x212B)			return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0xFB00, 0xFB06)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0xFF21, 0xFF3A)	return USP_LATIN;
	IF_CODEPOINT_IS_IN_RANGE(0xFF41, 0xFF5A)	return USP_LATIN;
	
	if(cp == 0x00B5 || cp == 0x037A)			return USP_GREEK;
	if(cp == 0x0386 || cp == 0x038C)			return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x0388, 0x038A)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x038E, 0x03A1)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x03A3, 0x03CE)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x03D0, 0x03F5)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x03F7, 0x03FB)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1D26, 0x1D2A)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1D5D, 0x1D61)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1D66, 0x1D6A)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F00, 0x1F15)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F18, 0x1F1D)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F20, 0x1F45)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F48, 0x1F4D)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F50, 0x1F57)	return USP_GREEK;
	if(cp == 0x1F59 || cp == 0x1F5B)			return USP_GREEK;
	if(cp == 0x1F5D)							return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F48, 0x1F4D)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F5F, 0x1F7D)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1F80, 0x1FB4)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FB6, 0x1FBC)	return USP_GREEK;
	if(cp == 0x1FBE || cp == 0x2126)			return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FC2, 0x1FC4)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FC6, 0x1FCC)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FD0, 0x1FD3)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FD6, 0x1FDB)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FE0, 0x1FEC)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FF2, 0x1FF4)	return USP_GREEK;
	IF_CODEPOINT_IS_IN_RANGE(0x1FF6, 0x1FFC)	return USP_GREEK;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0400, 0x0481)	return USP_CYRILLIC;
	IF_CODEPOINT_IS_IN_RANGE(0x0483, 0x0486)	return USP_CYRILLIC;
	IF_CODEPOINT_IS_IN_RANGE(0x048A, 0x04CE)	return USP_CYRILLIC;
	IF_CODEPOINT_IS_IN_RANGE(0x04D0, 0x04F5)	return USP_CYRILLIC;
	IF_CODEPOINT_IS_IN_RANGE(0x04F8, 0x04F9)	return USP_CYRILLIC;
	IF_CODEPOINT_IS_IN_RANGE(0x0500, 0x050F)	return USP_CYRILLIC;
	if(cp == 0x1D2B)							return USP_CYRILLIC;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0531, 0x0556)	return USP_ARMENIAN;
	if(cp == 0x0559)							return USP_ARMENIAN;
	IF_CODEPOINT_IS_IN_RANGE(0x0561, 0x0587)	return USP_ARMENIAN;
	IF_CODEPOINT_IS_IN_RANGE(0xFB13, 0xFB17)	return USP_ARMENIAN;
	
	IF_CODEPOINT_IS_IN_RANGE(0x05D0, 0x05EA)	return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0x05F0, 0x05F2)	return USP_HEBREW;
	if(cp == 0xFB1D || cp == 0xFB3E)			return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0xFB1F, 0xFB28)	return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0xFB2A, 0xFB36)	return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0xFB38, 0xFB3C)	return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0xFB40, 0xFB41)	return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0xFB43, 0xFB44)	return USP_HEBREW;
	IF_CODEPOINT_IS_IN_RANGE(0xFB46, 0xFB4F)	return USP_HEBREW;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0621, 0x063A)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0x0641, 0x064A)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0x066E, 0x066F)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0x0671, 0x06D3)	return USP_ARABIC;
	if(cp == 0x06D5 || cp == 0x06FF)			return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0x06E5, 0x06E6)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0x06EE, 0x06EF)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0x06FA, 0x06FC)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFB50, 0xFBB1)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFBD3, 0xFD3D)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFD50, 0xFD8F)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFD92, 0xFDC7)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFDF0, 0xFDFB)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFE70, 0xFE74)	return USP_ARABIC;
	IF_CODEPOINT_IS_IN_RANGE(0xFE76, 0xFEFC)	return USP_ARABIC;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0710, 0x074A)	return USP_SYRIAC;
	IF_CODEPOINT_IS_IN_RANGE(0x074D, 0x074F)	return USP_SYRIAC;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0780, 0x07B1)	return USP_THAANA;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0901, 0x0939)	return USP_DEVANAGARI;
	IF_CODEPOINT_IS_IN_RANGE(0x093C, 0x094D)	return USP_DEVANAGARI;
	IF_CODEPOINT_IS_IN_RANGE(0x0950, 0x0954)	return USP_DEVANAGARI;
	IF_CODEPOINT_IS_IN_RANGE(0x0958, 0x0963)	return USP_DEVANAGARI;
	IF_CODEPOINT_IS_IN_RANGE(0x0966, 0x096F)	return USP_DEVANAGARI;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0981, 0x0983)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x0985, 0x098C)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x098F, 0x0990)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x0993, 0x09A8)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09AA, 0x09B0)	return USP_BENGALI;
	if(cp == 0x09B2 || cp == 0x09D7)			return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09B6, 0x09B9)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09BC, 0x09C4)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09C7, 0x09C8)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09CB, 0x09CD)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09DC, 0x09DD)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09DF, 0x09E3)	return USP_BENGALI;
	IF_CODEPOINT_IS_IN_RANGE(0x09E6, 0x09F1)	return USP_BENGALI;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0A02, 0x0A03)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A05, 0x0A0A)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A0F, 0x0A10)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A13, 0x0A28)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A2A, 0x0A30)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A32, 0x0A33)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A35, 0x0A36)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A38, 0x0A39)	return USP_GURMUKHI;
	if(cp == 0x0A3C || cp == 0x0A5E)			return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A3E, 0x0A42)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A47, 0x0A48)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A4B, 0x0A4D)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A59, 0x0A5C)	return USP_GURMUKHI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A66, 0x0A74)	return USP_GURMUKHI;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0A81, 0x0A83)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A85, 0x0A8D)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A8F, 0x0A91)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0A93, 0x0AA8)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0AAA, 0x0AB0)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0AB2, 0x0AB3)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0AB5, 0x0AB9)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0ABC, 0x0AC5)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0AC7, 0x0AC9)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0ACB, 0x0ACD)	return USP_GUJARATI;
	if(cp == 0x0AD0)							return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0AE0, 0x0AE3)	return USP_GUJARATI;
	IF_CODEPOINT_IS_IN_RANGE(0x0AE6, 0x0AEF)	return USP_GUJARATI;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0B01, 0x0B03)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B05, 0x0B0C)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B0F, 0x0B10)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B13, 0x0B28)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B2A, 0x0B30)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B32, 0x0B33)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B35, 0x0B39)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B3C, 0x0B43)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B47, 0x0B48)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B4B, 0x0B4D)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B56, 0x0B57)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B5C, 0x0B5D)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B5F, 0x0B61)	return USP_ORIYA;
	IF_CODEPOINT_IS_IN_RANGE(0x0B66, 0x0B6F)	return USP_ORIYA;
	if(cp == 0x0B71)							return USP_ORIYA;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0B82, 0x0B83)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0B85, 0x0B8A)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0B8E, 0x0B90)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0B92, 0x0B95)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0B99, 0x0B9A)	return USP_TAMIL;
	if(cp == 0x0B9C || cp == 0x0BD7)			return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0B9E, 0x0B9F)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BA3, 0x0BA4)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BA8, 0x0BAA)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BAE, 0x0BB5)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BB7, 0x0BB9)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BBE, 0x0BC2)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BC6, 0x0BC8)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BCA, 0x0BCD)	return USP_TAMIL;
	IF_CODEPOINT_IS_IN_RANGE(0x0BE7, 0x0BF2)	return USP_TAMIL;
	
	IF_CODEPOINT_IS_IN_RANGE(0x0C01, 0x0C03)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C05, 0x0C0C)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C0E, 0x0C10)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C12, 0x0C28)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C2A, 0x0C33)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C35, 0x0C39)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C3E, 0x0C44)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C46, 0x0C48)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C4A, 0x0C4D)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C55, 0x0C56)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C60, 0x0C61)	return USP_TELUGU;
	IF_CODEPOINT_IS_IN_RANGE(0x0C66, 0x0C6F)	return USP_TELUGU;

	// ܂x...

	return USP_NOT_SCRIPT;
}

///	w肵R[h|Cgǂ Unicode JeSɑ邩Ԃ
inline Manah::Text::UnicodeSubrange Manah::Text::GetUnicodeCharCategory(Manah::Text::CodePoint cp) {
	if(cp < 0x000080)	return US_BASIC_LATIN;
	if(cp < 0x000100)	return US_LATIN1_SUPPLEMENT;
	if(cp < 0x000180)	return US_LATIN_EXTENDED_A;
	if(cp < 0x000250)	return US_LATIN_EXTENDED_B;
	if(cp < 0x0002B0)	return US_IPA_EXTENSIONS;
	if(cp < 0x000300)	return US_SPACING_MODIFIER_LETTERS;
	if(cp < 0x000370)	return US_COMBINING_DIACRITICAL_MARKS;
	if(cp < 0x000400)	return US_GREEK;
	if(cp < 0x000500)	return US_CYRILLIC;
	if(cp < 0x000530)	return US_CYRILLIC_SUPPLEMENTARY;
	if(cp < 0x000590)	return US_ARMENIAN;
	if(cp < 0x000600)	return US_HEBREW;
	if(cp < 0x000700)	return US_ARABIC;
	if(cp < 0x000750)	return US_SYRIAC;
	if(cp < 0x0007C0 && cp >= 0x000780)	return US_THAANA;
	if(cp < 0x000980 && cp >= 0x000900)	return US_DEVANAGARI;
	if(cp < 0x000A00)	return US_BENGALI;
	if(cp < 0x000A80)	return US_GURMUKHI;
	if(cp < 0x000B00)	return US_GUJARATI;
	if(cp < 0x000B80)	return US_ORIYA;
	if(cp < 0x000C00)	return US_TAMIL;
	if(cp < 0x000C80)	return US_TELUGU;
	if(cp < 0x000D00)	return US_KANNADA;
	if(cp < 0x000D80)	return US_MALAYALAM;
	if(cp < 0x000E00)	return US_SINHALA;
	if(cp < 0x000E80)	return US_THAI;
	if(cp < 0x000F00)	return US_LAO;
	if(cp < 0x001000)	return US_TIBETAN;
	if(cp < 0x0010A0)	return US_MYANMAR;
	if(cp < 0x001100)	return US_GEORGIAN;
	if(cp < 0x001200)	return US_HANGUL_JAMO;
	if(cp < 0x0013A0)	return US_ETHIOPIC;
	if(cp < 0x001400)	return US_CHEROKEE;
	if(cp < 0x001680)	return US_UNIFIED_CANADIAN_ABORIGINAL_SYLLABIC;
	if(cp < 0x0016A0)	return US_OGHAM;
	if(cp < 0x001700)	return US_RUNIC;
	if(cp < 0x001720)	return US_TAGALOG;
	if(cp < 0x001740)	return US_HANUNOO;
	if(cp < 0x001760)	return US_BUHID;
	if(cp < 0x001780)	return US_TAGBANWA;
	if(cp < 0x001800)	return US_KHMER;
	if(cp < 0x0018B0)	return US_MONGOLIAN;
	if(cp < 0x001950 && cp >= 0x001900)	return US_LIMBU;
	if(cp < 0x001980)	return US_TAI_LE;
	if(cp < 0x001A00 && cp >= 0x0019E0)	return US_KHMER_SYMBOLS;
	if(cp < 0x001D80 && cp >= 0x001D00)	return US_PHONETIC_EXTENSIONS;
	if(cp < 0x001F00 && cp >= 0x001E00)	return US_LATIN_EXTENDED_ADDITIONAL;
	if(cp < 0x002000)	return US_GREEK_EXTENDED;
	if(cp < 0x002070)	return US_GENERAL_PUNCTUATION;
	if(cp < 0x0020A0)	return US_SUPERSCRIPTS_AND_SUBSCRIPTS;
	if(cp < 0x0020D0)	return US_CURRENCY_SYMBOLS;
	if(cp < 0x002100)	return US_COMBINING_MARKS_FOR_SYMBOLS;
	if(cp < 0x002150)	return US_LETTERLIKE_SYMBOLS;
	if(cp < 0x002190)	return US_NUMBER_FORMS;
	if(cp < 0x002200)	return US_ARROWS;
	if(cp < 0x002300)	return US_MATHEMATICAL_OPERATORS;
	if(cp < 0x002400)	return US_MISCELLANEOUS_TECHNICAL;
	if(cp < 0x002440)	return US_CONTROL_PICTURES;
	if(cp < 0x002460)	return US_OPTICAL_CHARACTER_RECOGNITION;
	if(cp < 0x002500)	return US_ENCLOSED_ALPHANUMERICS;
	if(cp < 0x002580)	return US_BOX_DRAWING;
	if(cp < 0x0025A0)	return US_BLOCK_ELEMENTS;
	if(cp < 0x002600)	return US_GEOMETRIC_SHAPES;
	if(cp < 0x002700)	return US_MISCELLANEOUS_SYMBOLS;
	if(cp < 0x0027D0)	return US_DINGBATS;
	if(cp < 0x0027F0)	return US_MISCELLANEOUS_MATHEMATICAL_SYMBOLS_A;
	if(cp < 0x002800)	return US_SUPPLEMENTAL_ARROWS_A;
	if(cp < 0x002900)	return US_BRAILLE_PATTERNS;
	if(cp < 0x002980)	return US_SUPPLEMENTAL_ARROWS_B;
	if(cp < 0x002A00)	return US_MISCELLANEOUS_MATHEMATICAL_SYMBOLS_B;
	if(cp < 0x002B00)	return US_SUPPLEMENTAL_MATHEMATICAL_OPERATORS;
	if(cp < 0x002C00)	return US_MISCELLANEOUS_SYMBOLS_AND_ARROWS;
	if(cp < 0x002F00 && cp >= 0x002E80)	return US_CJK_RADICALS_SUPPLEMENT;
	if(cp < 0x002FE0)	return US_KANGXI_RADICALS;
	if(cp < 0x003000 && cp >= 0x002FF0)	return US_IDEOGRAPHIC_DESCRIPTION_CHARACTERS;
	if(cp < 0x003040)	return US_CJK_SYMBOLS_AND_PUNCTUATION;
	if(cp < 0x0030A0)	return US_HIRAGANA;
	if(cp < 0x003100)	return US_KATAKANA;
	if(cp < 0x003130)	return US_BOPOMOFO;
	if(cp < 0x003190)	return US_HANGUL_COMPATIBILITY_JAMO;
	if(cp < 0x0031A0)	return US_KANBUN;
	if(cp < 0x0031C0)	return US_BOPOMOFO_EXTENDED;
	if(cp < 0x003200 && cp >= 0x0031F0)	return US_KATAKANA_PHONETIC_EXTENSIONS;
	if(cp < 0x003300)	return US_ENCLOSED_CJK_LETTERS_AND_MONTHS;
	if(cp < 0x003400)	return US_CJK_COMPATIBILITY;
	if(cp < 0x004DB6)	return US_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A;
	if(cp < 0x004E00 && cp >= 0x004DC0)	return US_YIJING_HEXAGRAM_SYMBOLS;
	if(cp < 0x009FA6)	return US_CJK_UNIFIED_IDEOGRAPHS;
	if(cp < 0x00A490 && cp >= 0x00A000)	return US_YI_SYLLABLES;
	if(cp < 0x00A4CF)	return US_YI_RADICALS;
	if(cp < 0x00D7A4 && cp >= 0x00AC00)	return US_HANGUL_SYLLABLES;
	if(cp < 0x00DB80 && cp >= 0x00D800)	return US_HIGH_SURROGATES;
	if(cp < 0x00DC00)	return US_HIGH_PRIVATE_USE_SURROGATES;
	if(cp < 0x00E000)	return US_LOW_SURROGATES;
	if(cp < 0x00F900)	return US_PRIVATE_USE_AREA;
	if(cp < 0x00FB00)	return US_CJK_COMPATIBILITY_IDEOGRAPHS;
	if(cp < 0x00FB50)	return US_ALPHABETIC_PRESENTATION_FORMS;
	if(cp < 0x00FE00)	return US_ARABIC_PRESENTATION_FORMS_A;
	if(cp < 0x00FE10)	return US_VARIATION_SELECTORS;
	if(cp < 0x00FE30 && cp >= 0x00FE20)	return US_COMBINING_HALF_MARKS;
	if(cp < 0x00FE50)	return US_CJK_COMPATIBILITY_FORMS;
	if(cp < 0x00FE70)	return US_SMALL_FORM_VARIANTS;
	if(cp < 0x00FF00)	return US_ARABIC_PRESENTATION_FORMS_B;
	if(cp < 0x00FFF0)	return US_HALFWIDTH_AND_FULLWIDTH_FORMS;
	if(cp < 0x010000)	return US_SPECIALS;
	if(cp >= 0x010000 && cp < 0x010080)	return US_LINEAR_B_SYLLABARY;
	if(cp >= 0x010080 && cp < 0x010100)	return US_LINEAR_B_IDEOGRAMS;
	if(cp >= 0x010100 && cp < 0x010140)	return US_AEGEAN_NUMBERS;
	if(cp >= 0x010300 && cp < 0x010330)	return US_OLD_ITALIC;
	if(cp >= 0x010330 && cp < 0x010350)	return US_GOTHIC;
	if(cp >= 0x010380 && cp < 0x0103A0)	return US_UGARITIC;
	if(cp >= 0x010400 && cp < 0x010450)	return US_DESERET;
	if(cp >= 0x010450 && cp < 0x010480)	return US_SHAVIAN;
	if(cp >= 0x010480 && cp < 0x0104B0)	return US_OSMANYA;
	if(cp >= 0x010800 && cp < 0x010840)	return US_CYPRIOT_SYLLABARY;
	if(cp >= 0x01D000 && cp < 0x01D100)	return US_BYZANTINE_MUSICAL_SYMBOLS;
	if(cp >= 0x01D100 && cp < 0x01D200)	return US_MUSICAL_SYMBOLS;
	if(cp >= 0x01D300 && cp < 0x01D360)	return US_TAI_XUAN_JING_SYMBOLS;
	if(cp >= 0x01D400 && cp < 0x01D800)	return US_MATHEMATICAL_ALPHANUMERIC_SYMBOLS;
	if(cp >= 0x020000 && cp < 0x02A6D7)	return US_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B;
	if(cp >= 0x02F800 && cp < 0x02FA20)	return US_CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT;
	if(cp >= 0x0E0000 && cp < 0x0E0080)	return US_TAGS;
	if(cp >= 0x0E0100 && cp < 0x0E01F0)	return US_VARIATION_SELECTORS_SUPPLEMENT;
	if(cp >= 0x0FFF80 && cp < 0x100000)	return US_SUPPLEMENTARY_PRIVATE_USE_AREA_A;
	if(cp >= 0x10FF80 && cp < 0x110000)	return US_SUPPLEMENTARY_PRIVATE_USE_AREA_B;
	return US_UNASSIGNED;
}

///	̐擪̕ǂ Unicode JeSɑ邩Ԃ
inline Manah::Text::UnicodeSubrange Manah::Text::GetUnicodeStringCategory(const wchar_t* pch, std::size_t cch) {
	assert(pch != 0 && cch != 0);
	return GetUnicodeCharCategory(DecodeUTF16SurrogatePairToCodePoint(pch, cch));
}

///	̐擪 ISO-2022-CN  BOM 邩ǂԂ
inline bool Manah::Text::HasBOMOfISO2022CN(const unsigned char* pszText, std::size_t cch) {
	return cch >= 4
			&& pszText[0] == 0x1B
			&& pszText[1] == 0x24
			&& pszText[2] == 0x29
			&& pszText[3] == 0x41;
}

///	̐擪 ISO-2022-KR  BOM 邩ǂԂ
inline bool Manah::Text::HasBOMOfISO2022KR(const unsigned char* pszText, std::size_t cch) {
	return cch >= 4
			&& pszText[0] == 0x1B
			&& pszText[1] == 0x24
			&& pszText[2] == 0x29
			&& pszText[3] == 0x43;
}

///	w肵ʒuPꋫEǂԂ (Shift-JIS )
inline bool Manah::Text::HasWordBorderAt(const unsigned char* pszText, unsigned int i) {
	SJISCharType	ct = GetCharType(pszText + i);

	if(i == 0)
		return true;
	if(ct == CT_OTHER)
		return true;
	if(ct <= CT_KANA_HALFWIDTH)
		return (ct & 0xF0) == (GetCharType(pszText + i - 1) & 0xF0);
	else if(i != 1) {
		SJISCharType	ctPrev = GetCharType(pszText + i - 2);
		return ((ct & 0xF0) == (GetCharType(pszText + i - 2) & 0xF0))
			&& !(pszText[i - 0] == 0x81 && pszText[i + 1] == 0x5B && (ctPrev == CT_KANA_HIRA || ctPrev == CT_KANA_KATA))
			&& !(pszText[i - 2] == 0x81 && pszText[i - 1] == 0x5B && (ct == CT_KANA_HIRA || ct == CT_KANA_KATA))
			&& !(pszText[i - 0] == 0x81 && pszText[i + 1] == 0x58 && (ctPrev & 0xF0) == 0xA0)
			&& !(pszText[i - 2] == 0x81 && pszText[i - 1] == 0x58 && (ct & 0xF0) == 0xA0);
	} else
		return true;
}

///	w肵ʒuPꋫEǂԂ (Unicode )
inline bool Manah::Text::HasWordBorderAt(const wchar_t* pszText, unsigned int i) {
	WORD	wCharType, wPrevCharType;

	if(i == 0)
		return true;
	if(pszText[i] == L'_')
		wCharType = 0x0040;
	else if(pszText[i] == 0x3005)
		wCharType = 0x8100;
	else {
		::GetStringTypeW(CT_CTYPE3, pszText + i - 0, 1, &wCharType);
		wCharType &= ~(C3_DIACRITIC | C3_VOWELMARK | C3_FULLWIDTH | C3_LEXICAL | C3_ALPHA);
	}
	if(pszText[i - 1] == L'_')
		wPrevCharType = 0x0040;
	else if(pszText[i - 1] == 0x3005)
		wPrevCharType = 0x8100;
	else {
		::GetStringTypeW(CT_CTYPE3, pszText + i - 1, 1, &wPrevCharType);
		wPrevCharType &= ~(C3_DIACRITIC | C3_VOWELMARK | C3_FULLWIDTH | C3_LEXICAL | C3_ALPHA);
	}

	return (wCharType != wPrevCharType)
		&& !(pszText[i - 0] == 0x30FC && wPrevCharType & 0x0030)
		&& !(pszText[i - 1] == 0x30FC && wCharType & 0x0030);
}

///	EUC 2oCgǂԂ
inline bool Manah::Text::IsEUCCharacters(const unsigned char* pch) {
	unsigned char	chLead	= pch[0];
	unsigned char	chTrail	= pch[1];

	if((chLead >= 0x81 && chLead <= 0x9F) || (chLead >= 0xE0 && chLead <= 0xFC))
		return (chTrail >= 0x40 && chTrail <= 0xFC && chTrail != 0x7F && chTrail != 0x8E);
	else if(chLead >= 0xFD && chLead <= 0xFE)
		return IsEUCTrailByte(chTrail);
	else
		return false;
}

///	EUC ̐soCgǂԂ
inline bool Manah::Text::IsEUCLeadByte(unsigned char ch) {
	return (ch >= 0x81 && ch <= 0x9F)
		|| (ch >= 0xE0 && ch <= 0xFE);
}

///	 EUC ƂĉLԂ
inline std::size_t Manah::Text::IsEUCString(const unsigned char* pszText, std::size_t cch) {
	bool		bTrail = false;
	std::size_t	i;

	for(i = 0; i < cch; ++i) {
		unsigned char	ch = pszText[i];
		if(!bTrail) {
			if(ch >= 0xA1 && ch <= 0xDF)	// pJ^Ji
				;
			else if(ch >= 0x09 && ch <= 0x0D)	// 󔒗ސ䕶
				;
			else if(ch < 0x20)	// 䕶
				return i;
			else if(ch <= 0x7F)	// ASCII 
				;
			else if(IsEUCLeadByte(ch))	// soCg
				bTrail = true;
			else	// s
				break;
		} else {
			if(IsEUCTrailByte(ch))	// 㑱oCg
				bTrail = false;
			else	// s
				break;
		}
	}
	return i;
}

///	EUC ̌㑱oCgǂԂ
inline bool Manah::Text::IsEUCTrailByte(unsigned char ch) {
	return (ch >= 0xA1 && ch <= 0xFC && ch != 0x7F && ch != 0x8E);
}

///	 JIS ƂĉLԂB
///	 ESC 1xoĂȂ0Ԃ
inline std::size_t Manah::Text::IsJISString(const unsigned char* pszText, std::size_t cch) {
	bool	bFound8bit = false;
	for(std::size_t i = 0; i < cch; ++i) {
		if(pszText[i] >= 0x80)
			return i;
		else if(pszText[i] == 0x1B)
			bFound8bit = true;
	}
	return bFound8bit ? cch : 0;
}

///	Shift-JIS 2oCgǂԂ
inline bool Manah::Text::IsShiftJISCharacters(const unsigned char* pch) {
	return ((*(pch + 0) >= 0x81 && *(pch + 0) <= 0x9F)
		|| (*(pch + 0) >= 0xE0 && *(pch + 0) <= 0xFC))
		&& ((*(pch + 1) >= 0x40 && *(pch + 1) <= 0x7E)
		|| (*(pch + 1) >= 0x80 && *(pch + 1) <= 0xFC));
}

///	 Shift-JIS ƂĉLԂ
inline std::size_t Manah::Text::IsShiftJISString(const unsigned char* pszText, std::size_t cch) {
	SJISCharType	ct;
	std::size_t		i;

	for(i = 0; i < cch; ++i) {
		ct = GetCharType(pszText + i);
		if(ct == CT_CONTROL)	// 䕶
			return i;
		else if(ct <= CT_KANA_HALFWIDTH)	// 1oCg
			;
		else if(ct != CT_OTHER)	// 2oCg
			++i;
		else	// s
			break;
	}
	return i;
}

///	 UTF-16 ʃTQ[głΐ^Ԃ
inline bool Manah::Text::IsUTF16HighSurrogate(wchar_t ch) {
	return ch >= 0xD800 && ch < 0xDC00;
}

///	 UTF-16 ʃTQ[głΐ^Ԃ
inline bool Manah::Text::IsUTF16LowSurrogate(wchar_t ch) {
	return ch >= 0xDC00 && ch < 0xE000;
}

///	ArA 0-9 ɕϊ (: FoldStringW(MAP_FOLDDIGITS))
inline Manah::Text::CodePoint Manah::Text::ToAsciiDigit(Manah::Text::CodePoint cp) {
#define FOLD_DIGIT(lo, up)	if(cp >= lo && cp <= up) return cp - up + L'9'

	// Unicode 4.0  'Nd' ɊÂ
	FOLD_DIGIT(0x0030, 0x0039);		// C0
	FOLD_DIGIT(0x0660, 0x0669);		// Arabic-Indic
	FOLD_DIGIT(0x06F0, 0x06F9);		// Extended Arabic-Indic
	FOLD_DIGIT(0x0966, 0x096F);		// Devanagari
	FOLD_DIGIT(0x09E6, 0x09EF);		// Bengali
	FOLD_DIGIT(0x0A66, 0x0A6F);		// Gurmukhi
	FOLD_DIGIT(0x0AE6, 0x0AEF);		// Gujarati
	FOLD_DIGIT(0x0B66, 0x0B6F);		// Oriya
	FOLD_DIGIT(0x0BE7, 0x0BEF);		// Tamil (0 not found)
	FOLD_DIGIT(0x0C66, 0x0C6F);		// Telugu
	FOLD_DIGIT(0x0CE6, 0x0CEF);		// Kannada
	FOLD_DIGIT(0x0D66, 0x0D6F);		// Malayalam
	FOLD_DIGIT(0x0E50, 0x0E59);		// Thai
	FOLD_DIGIT(0x0ED0, 0x0ED9);		// Lao
	FOLD_DIGIT(0x0F20, 0x0F29);		// Tibetan
	FOLD_DIGIT(0x1040, 0x1049);		// Myanmar
	FOLD_DIGIT(0x1369, 0x1371);		// Ethiopic (0 not found)
	FOLD_DIGIT(0x17E0, 0x17E9);		// Khmer
	FOLD_DIGIT(0x1810, 0x1819);		// Mongolian
	FOLD_DIGIT(0x1946, 0x194F);		// Limbu
	FOLD_DIGIT(0xFF10, 0xFF19);		// Fullwidth
	FOLD_DIGIT(0x0104A0, 0x0104A9);	// Osmanya
	FOLD_DIGIT(0x01D7CE, 0x01D7D7);	// Mathematical bold
	FOLD_DIGIT(0x01D7D8, 0x01D7E1);	// Mathematical double-struck
	FOLD_DIGIT(0x01D7E2, 0x01D7EB);	// Mathematical sans-serif
	FOLD_DIGIT(0x01D7EC, 0x01D7F5);	// Mathematical sans-serif bold
	FOLD_DIGIT(0x01D7F6, 0x01D7FF);	// Mathematical monospace
	return cp;

#undef FOLD_DIGIT
}

///	Spɕϊ (: LCMapStringW(LCMAP_FULLWIDTH))
inline const wchar_t Manah::Text::ToFullWidth(wchar_t ch) {
	if(ch == 0xFF61)		return 0x3002;
	else if(ch == 0xFF62)	return 0x300C;
	else if(ch == 0xFF63)	return 0x300D;
	else if(ch == 0xFF64)	return 0x3001;
	else if(ch == 0xFFA0)	return 0x3164;
	else if(ch >= 0xFFA1 && ch <= 0xFFBE)
		return ch - 0xCE70;
	else if(ch >= 0xFFC2 && ch <= 0xFFC7)
		return ch - 0xCE73;
	else if(ch >= 0xFFCA && ch <= 0xFFCF)
		return ch - 0xCE75;
	else if(ch >= 0xFFD2 && ch <= 0xFFD7)
		return ch - 0xCE77;
	else if(ch >= 0xFFDA && ch <= 0xFFDC)
		return ch - 0xCE79;
}

///	pɕϊ (: LCMapStringW(LCMAP_HALFWIDTH))
inline const wchar_t Manah::Text::ToHalfWidth(wchar_t ch) {
	if(ch == 0x3000)	return 0x0020;
	else if(ch == 0x3001)	return 0xFF64;
	else if(ch == 0x3002)	return 0xFF61;
	else if(ch == 0x300C)	return 0xFF62;
	else if(ch == 0x300D)	return 0xFF63;
	else if(ch == 0x3164)	return 0xFFA0;
	else if(ch >= 0xFF01 && ch <= 0xFF5E)
		return ch - 0xFED0;
	else if(ch == 0xFF5F)	return 0x2985;
	else if(ch == 0xFF60)	return 0x2986;
	else if(ch == 0xFFE0)	return 0x00A2;
	else if(ch == 0xFFE1)	return 0x00A3;
	else if(ch == 0xFFE2)	return 0x00AC;
	else if(ch == 0xFFE3)	return 0x00AF;
	else if(ch == 0xFFE4)	return 0x00A6;
	else if(ch == 0xFFE5)	return 0x00A5;
	else if(ch == 0xFFE6)	return 0x20A9;
}

///	ɕϊ (: LCMapStringW(LCMAP_HIRAGANA))
inline wchar_t Manah::Text::ToHiragana(wchar_t ch) {
	if((ch >= 0x3041 && ch <= 0x3096) || ch == 0x309D || ch == 0x309E)
		return ch + 0x0060;
	return ch;
}

/**
 *	ɕϊ (: LCMapStringW(LCMAP_HIRAGANA))
 *	@param psz	ϊ镶
 *	@param cch	̒
 */
inline void Manah::Text::ToHiragana(wchar_t* psz, std::size_t cch /* = -1 */) {
	if(cch == -1)
		cch = wcslen(psz);
	if(cch == 0)
		return;
	do {
		psz[cch - 1] = ToHiragana(psz[cch - 1]);
	} while(--cch != 0);
}

///	Љɕϊ (: LCMapStringW(LCMAP_KATAKANA))
inline wchar_t Manah::Text::ToKatakana(wchar_t ch) {
	if((ch >= 0x30A1 && ch <= 0x30F6) || ch == 0x30FD || ch == 0x30FE)
		return ch - 0x0060;
	return ch;
}

/**
 *	Љɕϊ (: LCMapStringW(LCMAP_KATAKANA))
 *	@param psz	ϊ镶
 *	@param cch	̒
 */
inline void Manah::Text::ToKatakana(wchar_t* psz, std::size_t cch /* = -1 */) {
	if(cch == -1)
		cch = wcslen(psz);
	if(cch == 0)
		return;
	do {
		psz[cch - 1] = ToKatakana(psz[cch - 1]);
	} while(--cch != 0);
}

///	ɕϊ (: CharLowerW)
inline wchar_t Manah::Text::ToLower(wchar_t ch) {
	return ch;
}

///	ɕϊ (: CharLowerW)
inline void Manah::Text::ToLower(wchar_t* psz, std::size_t cch /* = -1 */) {
	assert(psz != 0);

	if(cch == -1) {
		wchar_t*	p = psz;
		while(*p != 0)
			*(p++) = ToLower(*p);
	} else {
		for(std::size_t i = 0; i < cch; ++i)
			*(psz + i) = ToLower(*(psz + i));
	}
}

///	啶ɕϊ (: CharUpperW)
inline wchar_t Manah::Text::ToUpper(wchar_t ch) {
	return ch;
}

///	啶ɕϊ (: CharUpperW)
inline void Manah::Text::ToUpper(wchar_t* psz, std::size_t cch /* = -1 */) {
	assert(psz != 0);

	if(cch == -1) {
		wchar_t*	p = psz;
		while(*p != 0)
			*(p++) = ToUpper(*p);
	} else {
		for(std::size_t i = 0; i < cch; ++i)
			*(psz + i) = ToUpper(*(psz + i));
	}
}


#endif /* _TEXT_CONVERTER_H_ */

/* [EOF] */