// Window.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _WINDOW_H_
#define _WINDOW_H_

#include "Object.h"

#ifndef WS_EX_LAYERED
#define WS_EX_LAYERED	0x00080000
#define LWA_COLORKEY	0x00000001
#define LWA_ALPHA		0x00000002
/*typedef struct _BLENDFUNCTION {
	BYTE	BlendOp;
	BYTE	BlendFlags;
	BYTE	SourceConstantAlpha;
	BYTE	AlphaFormat;
} BLENDFUNCTION, *PBLENDFUNCTION;*/
#endif /* !WS_EX_LAYERED */

#ifndef AW_SLIDE
#define AW_HOR_POSITIVE	0x00000001
#define AW_HOR_NEGATIVE	0x00000002
#define AW_VER_POSITIVE	0x00000004
#define AW_VER_NEGATIVE	0x00000008
#define AW_CENTER		0x00000010
#define AW_HIDE			0x00010000
#define AW_ACTIVATE		0x00020000
#define AW_SLIDE		0x00040000
#define AW_BLEND		0x00080000
#endif /* !AW_SLIDE */


// CWindow class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CWindow : public CObject {
	// RXgN^
public:
	CWindow();
	CWindow(const CWindow& rhs);
	virtual ~CWindow();

	// Zq
public:
	operator HWND()	{
		return GetSafeHwnd();
	}
private:
	operator =(const CWindow& rhs);	// never been invoked

	// \bh
protected:
	virtual LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);
	virtual bool	PreTranslateMessage(UINT message, WPARAM wParam, LPARAM lParam);	/* called by only DE */
	static LRESULT CALLBACK	WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
public:
	/* j */
	virtual bool	DestroyWindow();

	/*  */
	bool	Create(bool bSubClass,
				const TCHAR* lpszClassName, const TCHAR* lpszWindowName, DWORD dwStyle,
				int x, int y, int nWidth, int nHeight,
				HWND hwndParentOrHInstance, HMENU hMenu = 0, LPVOID lpParam = 0L);
	bool	CreateEx(bool bSubClass,
				DWORD dwExStyle, const TCHAR* lpszClassName, const TCHAR* lpszWindowName, DWORD dwStyle,
				int x, int y, int nWidth, int nHeight,
				HWND hwndParentOrHInstance, HMENU hMenu = 0, LPVOID lpParam = 0L);
//	virtual bool	Create(bool bSubClass, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id) = 0;
//	virtual bool	CreateEx(
//		bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id) = 0;
	DWORD			GetStyle() const;
	DWORD			GetExStyle() const;
	virtual bool	Attach(HWND hWnd, bool bSubClass = false);
	HWND			Detach();
	HWND			GetSafeHwnd() const;
	LONG			GetWindowLong(int nIndex) const;
	LONG			SetWindowLong(int nIndex, LONG dwNewLong);

	/*  */
	bool			IsWindow() const;
	bool			IsWindowEnabled() const;
	bool			EnableWindow(bool bEnable = true);
	static CWindow*	GetActiveWindow();
	CWindow*		SetActiveWindow();
	static CWindow*	GetCapture();
	CWindow*		SetCapture();
	static bool		ReleaseCapture();
	static CWindow*	GetFocus();
	CWindow*		SetFocus();
	static CWindow*	GetDesktopWindow();
	static CWindow*	GetForegroundWindow();
	bool			SetForegroundWindow();
	HICON			GetIcon(bool bBigIcon = true) const;
	HICON			SetIcon(HICON hIcon, bool bBigIcon  = true);
	DWORD			GetWindowContextHelpId() const;
	bool			SetWindowContextHelpId(DWORD dwContextHelpId);
	bool			ModifyStyle(DWORD dwRemove, DWORD dwAdd);
	bool			ModifyStyleEx(DWORD dwRemove, DWORD dwAdd);

	/* TCYƈʒu */
	bool	GetWindowPlacement(WINDOWPLACEMENT* lpwndpl) const;
	bool	SetWindowPlacement(const WINDOWPLACEMENT* lpwndpl);
	int		GetWindowRgn(HRGN hRgn) const;
	int		SetWindowRgn(const HRGN hRgn, bool bRedraw = true);
	bool	IsIconic() const;
	bool	IsZoomed() const;
	void	MoveWindow(int x, int y, int nWidth, int nHeight, bool bRepaint = true);
	void	MoveWindow(const RECT* lpRect, bool bRepaint = true);
	bool	SetWindowPos(HWND hwndInsertAfter, int x, int y, int cx, int cy, UINT nFlags);
	bool	SetWindowPos(HWND hwndInsertAfter, LPRECT lpRect, UINT nFlags);
	UINT	ArrangeIconicWindows();
	void	BringWindowToTop();
	void	GetWindowRect(LPRECT lpRect) const;
	void	GetClientRect(LPRECT lpRect) const;

	/* EBhEANZX */
	CWindow*		ChildWindowFromPoint(POINT pt) const;
	CWindow*		ChildWindowFromPointEx(POINT pt, UINT nFlags) const;
	static CWindow*	FindWindow(const TCHAR* lpszClassName, const TCHAR* lpszWindowName);
	CWindow*		GetNext(UINT nFlag = GW_HWNDNEXT) const;
	CWindow*		GetOwner() const;
	CWindow*		GetTopWindow() const;
	CWindow*		GetWindow(UINT nCmd) const;
	CWindow*		GetLastActivePopup() const;
	bool			IsChild(HWND hWnd) const;
	CWindow*		GetParent() const;
	CWindow*		SetParent(HWND hwndNewParent);
	static CWindow*	WindowFromPoint(POINT pt);
	void			CenterWindow(HWND hwndAlternate = 0);
	int				GetDlgCtrlID() const;
	int				SetDlgCtrlID(int nID);

	/* XVƕ` */
	HDC		BeginPaint(LPPAINTSTRUCT lpPaint);
	void	EndPaint(const LPPAINTSTRUCT lpPaint);
	void	Print(HDC hDC, DWORD dwFlags) const;
	void	PrintClient(HDC hDC, DWORD dwFlags) const;
	bool	LockWindowUpdate();
	void	UnlockWindowUpdate();
	HDC		GetDC();
	HDC		GetDCEx(HRGN hrgnClip, DWORD dwFlags);
	bool	RedrawWindow(LPRECT lpRectUpdate = 0, HRGN hrgnClip = 0,
				UINT nFlags = RDW_INVALIDATE | RDW_UPDATENOW | RDW_ERASE);
	HDC		GetWindowDC();
	int		ReleaseDC(HDC hDC);
	void	UpdateWindow();
	void	SetRedraw(bool bRedraw = true);
	bool	GetUpdateRect(LPRECT lpRect, bool bErase = false);
	int		GetUpdateRgn(HRGN hRgn, bool bErase = false);
	void	Invalidate(bool bErase = true);
	void	InvalidateRect(LPRECT lpRect, bool bErase = true);
	void	InvalidateRgn(HRGN hRgn, bool bErase = true);
	void	ValidateRect(LPRECT lpRect);
	void	ValidateRgn(HRGN hRgn);
	bool	ShowWindow(UINT nCmdShow);
	bool	IsWindowVisible() const;
	void	ShowOwnedPopups(bool bShow = true);
	bool	EnableScrollBar(int nSBFlags, UINT nArrowFlags = ESB_ENABLE_BOTH);

	/* W}bsO */
	void	MapWindowPoints(HWND hWndTo, LPRECT lpRect) const;
	void	MapWindowPoints(HWND hWndTo, LPPOINT lpPoint, UINT nCount) const;
	void	ClientToScreen(LPPOINT lpPoint) const;
	void	ClientToScreen(LPRECT lpRect) const;
	void	ScreenToClient(LPPOINT lpPoint) const;
	void	ScreenToClient(LPRECT lpRect) const;

	/* EBhEeLXg */
	void	SetWindowText(const TCHAR* lpszText);
	int		GetWindowText(TCHAR* lpszText, int nMaxCount) const;
	int		GetWindowTextLength() const;
	void	SetFont(HFONT hFont, bool bRedraw = true);
	HFONT	GetFont() const;

	/* XN[ */
	bool	GetScrollInfo(int nBar, LPSCROLLINFO lpScrollInfo, UINT nMask = SIF_ALL) const;
	int		GetScrollPos(int nBar) const;
	void	GetScrollRange(int nBar, int* nMinPos, int* nMaxPos) const;
	int		GetScrollTrackPos(int nBar) const;
	void	ScrollWindow(int xAmount, int yAmount, LPRECT lpRect = 0, LPRECT lpClipRect = 0);
	int		ScrollWindowEx(int dx, int dy, LPRECT lpScrollRect, LPRECT lpClipRect,
				HRGN hrgnUpdate, LPRECT lpUpdateRect, UINT nFlags);
	int		GetScrollLimit(int nBar) const;
	bool	SetScrollInfo(int nBar, LPSCROLLINFO lpScrollInfo, bool bRedraw = true);
	int		SetScrollPos(int nBar, int nPos, bool bRedraw = true);
	void	SetScrollRange(int nBar, int nMinPos, int nMaxPos, bool bRedraw = true);
	void	ShowScrollBar(int nBar, bool bShow = true);

	/* hbOAhhbv */
	void	DragAcceptFiles(bool bAccept = true);
	HRESULT	RegisterDragDrop(IDropTarget* pDropTarget);
	HRESULT	RevokeDragDrop();

	/* Lbg */
	bool			CreateCaret(HBITMAP hBitmap, int nWidth, int nHeight);
	bool			CreateSolidCaret(int nWidth, int nHeight);
	bool			CreateGrayCaret(int nWidth, int nHeight);
	static LPPOINT	GetCaretPos();
	static void		SetCaretPos(POINT pt);
	void			HideCaret();
	void			ShowCaret();

	/* j[ */
	HMENU	GetMenu() const;
	bool	SetMenu(HMENU hMenu);
	void	DrawMenuBar();
	HMENU	GetSystemMenu(bool bRevert) const;
	bool	HiliteMenuItem(HMENU hMenu, UINT nItem, UINT nFlags);

	/* ^C} */
	UINT	SetTimer(UINT nIDEvent, UINT nElapse,
				void (CALLBACK* lpfnTimer)(HWND, UINT, UINT, DWORD));
	bool	KillTimer(UINT nIDEvent);

	/* x */
	bool	FlashWindow(bool bInvert);
	int		MessageBox(const TCHAR* lpszText, const TCHAR* lpszCaption = 0, UINT nType = MB_OK);

	/* EBhEbZ[W */
	LRESULT			SendMessage(UINT message, WPARAM wParam = 0, LPARAM lParam = 0L);
	LRESULT			PostMessage(UINT message, WPARAM wParam = 0, LPARAM lParam = 0L);
	bool			SendNotifyMessage(UINT message, WPARAM wParam, LPARAM lParam);
	virtual LRESULT	DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam);

	/* C */
	bool	AnimateWindow(DWORD dwTime, DWORD dwFlags, bool bCatchError = true);
	bool	SetLayeredWindowAttributes(COLORREF clrKey, BYTE bAlpha, DWORD dwFlags);
	bool	UpdateLayeredWindow(HDC hDest, POINT* pptDest,
				SIZE* pSize, HDC hSrc, POINT* pptSrc, COLORREF clrKey,
				BLENDFUNCTION* pBlend, DWORD dwFlags);

	/* I[o[Ch\ȃbZ[Wnh */
protected:
	virtual void	OnAttached();	// WM_MATTACH
	/* need to any implement */
	virtual bool	OnCommand(WORD wID, WORD wNotifyCode, HWND hwndCtrl);		// WM_COMMAND
	virtual void	OnClose();													// WM_CLOSE
	virtual void	OnDestroy();												// WM_DESTROY
	virtual bool	OnNotify(int idCtrl, LPNMHDR lpNMHDR);						// WM_NOTIFY
	virtual bool	OnSetCursor(HWND hWnd, UINT nHitTest, UINT message);		// WM_SETCURSOR
	/* noneed to implement */
	virtual void	OnActivate(UINT nState, HWND hwndPrevious, bool bMinimized);// WM_ACTIVATE
	virtual bool	OnContextMenu(HWND hWnd, POINT pt);							// WM_CONTEXTMENU
	virtual void	OnDrawItem(UINT idCtrl, LPDRAWITEMSTRUCT lpDrawItem);		// WM_DRAWITEM
	virtual void	OnKillFocus(HWND hwndNew);									// WM_KILLFOCUS
	virtual void	OnLButtonDown(UINT nFlags, POINT pt);						// WM_LBUTTONDOWN
	virtual void	OnLButtonDblClk(UINT nFlags, POINT pt);						// WM_LBUTTONDBLCLK
	virtual void	OnLButtonUp(UINT nFlags, POINT pt);							// WM_LBUTTONUP
	virtual void	OnMeasureItem(UINT idCtrl, LPMEASUREITEMSTRUCT lpMI);		// WM_MEASUREITEM
	virtual void	OnMouseMove(UINT nFlags, POINT pt);							// WM_MOUSEMOVE
	virtual void	OnPaint();													// WM_PAINT
	virtual void	OnSetFocus(HWND hwndOld);									// WM_SETFOCUS
	virtual void	OnShowWindow(bool bShow, UINT nStatus);						// WM_SHOWWINDOW
	virtual void	OnSize(UINT nType, int cx, int cy);							// WM_SIZE
	virtual void	OnSysColorChange();											// WM_SYSCOLORCHANGE
	virtual void	OnTimer(UINT nIDEvent);										// WM_TIMER

	/* fobO */
protected:
#ifdef _DEBUG
	virtual void AssertValid() const {
		CObject::AssertValid();
	}
	virtual void AssertValidAsWindow() const {
		CObject::AssertValid();
		assert(IsWindow());
	}
#else
	virtual void AssertValid() const {}
	virtual void AssertValidAsWindow() const {}
#endif /* _DEBUG */

	// typedef
private:
	typedef BOOL(WINAPI *AW)(HWND, DWORD, DWORD);
	typedef BOOL(WINAPI *SLWA)(HWND, COLORREF, BYTE, DWORD);
	typedef BOOL(WINAPI *ULW)(HWND, HDC, POINT*, SIZE*, HDC, POINT*, COLORREF, BLENDFUNCTION*, DWORD);

	// f[^o
public:
	HWND	m_hWnd;
protected:
	WNDPROC	m_OldProc;		// original proc before attached
	LPARAM	m_lOldUserData;	// original data which can get by ::GetWindowLong(GWL_USERDATA)
	HMODULE	m_hUser32Dll;	// loaded dynamically to use layered window APIs
	AW		m_pfnAnimateWindow;
	SLWA	m_pfnSetLayeredWindowAttributes;
	ULW		m_pfnUpdateLayeredWindow;
};

#define WM_MATTACH (WM_APP + 100)	/* only used when attach window proc to global special callback */
typedef struct _tagMPARAM {
	WPARAM	wParam;
	LPARAM	lParam;
} MPARAM;


// CWindow class implementation
/////////////////////////////////////////////////////////////////////////////

inline CWindow::CWindow() : m_hWnd(0), m_OldProc(0),
		m_lOldUserData(0L), m_hUser32Dll(0),
		m_pfnAnimateWindow(0), m_pfnSetLayeredWindowAttributes(0), m_pfnUpdateLayeredWindow(0) {
}

inline CWindow::CWindow(const CWindow& rhs) : m_hWnd(0), m_OldProc(0),
		m_lOldUserData(0L), m_hUser32Dll(0),
		m_pfnAnimateWindow(0), m_pfnSetLayeredWindowAttributes(0), m_pfnUpdateLayeredWindow(0) {
}

inline CWindow::~CWindow() {
	if(IsWindow())
		DestroyWindow();
	if(m_hUser32Dll != 0)
		::FreeLibrary(m_hUser32Dll);
}

inline bool CWindow::AnimateWindow(DWORD dwTime, DWORD dwFlags, bool bCatchError /* = true */) {
	AssertValidAsWindow();
	if(m_hUser32Dll == 0)
		m_hUser32Dll = ::LoadLibrary(_T("User32.dll"));
	if(m_hUser32Dll != 0 && m_pfnAnimateWindow == 0)
		m_pfnAnimateWindow =
			reinterpret_cast<AW>(::GetProcAddress(m_hUser32Dll, "AnimateWindow"));
	if((m_pfnAnimateWindow != 0) ?
			toBoolean((*m_pfnAnimateWindow)(m_hWnd, dwTime, dwFlags)) : false)
		return true;
	else if(bCatchError) {
		if(toBoolean(dwFlags & AW_HIDE))
			return ShowWindow(SW_HIDE);
		else
			return ShowWindow(toBoolean(dwFlags & AW_ACTIVATE) ? SW_SHOW : SW_SHOWNA);
	}
	return false;
}

inline UINT CWindow::ArrangeIconicWindows() {
	AssertValid();
	return ::ArrangeIconicWindows(m_hWnd);
}

inline bool CWindow::Attach(HWND hWnd, bool bSubClass /* = false */) {
	AssertValid();

	if(IsWindow() || !::IsWindow(hWnd) || m_OldProc != 0)
		return false;
	m_hWnd = hWnd;

	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CWindow::WndProc)));
		assert(m_OldProc != CWindow::WndProc);
		m_lOldUserData = GetWindowLong(GWL_USERDATA);
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline HDC CWindow::BeginPaint(LPPAINTSTRUCT lpPaint) {
	AssertValidAsWindow();
	return ::BeginPaint(m_hWnd, lpPaint);
}

inline void CWindow::BringWindowToTop() {
	AssertValidAsWindow();
	::BringWindowToTop(m_hWnd);
}

inline void CWindow::CenterWindow(HWND hwndAlternate /* = 0 */) {
	AssertValidAsWindow();

	RECT rectWin, rectAlt;

	if(hwndAlternate == 0){
		hwndAlternate = ::GetParent(m_hWnd);
		if(hwndAlternate == 0)
			hwndAlternate = ::GetDesktopWindow();
	}
	assert(::IsWindow(hwndAlternate));

	GetWindowRect(&rectWin);
	::GetWindowRect(hwndAlternate, &rectAlt);
	SetWindowPos(0, (rectAlt.right - rectAlt.left) / 2 - (rectWin.right - rectWin.left) / 2,
		(rectAlt.bottom - rectAlt.top) / 2 - (rectWin.bottom - rectWin.top) / 2,
		0, 0, SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOZORDER);
}

inline CWindow* CWindow::ChildWindowFromPoint(POINT pt) const {
	AssertValidAsWindow();

	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::ChildWindowFromPoint(m_hWnd, pt)))
		return &objWindow;
	else
		return 0;
}

inline CWindow* CWindow::ChildWindowFromPointEx(POINT pt, UINT nFlags) const {
	AssertValidAsWindow();

	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::ChildWindowFromPointEx(m_hWnd, pt, nFlags)))
		return &objWindow;
	else
		return 0;
}

inline void CWindow::ClientToScreen(LPRECT lpRect) const {
	AssertValidAsWindow();

	POINT	point[2];

	point[0].x = lpRect->left;
	point[0].y = lpRect->top;
	point[1].x = lpRect->right;
	point[1].y = lpRect->bottom;
	::ClientToScreen(m_hWnd, &point[0]);
	::ClientToScreen(m_hWnd, &point[1]);
	::SetRect(lpRect, point[0].x, point[0].y, point[1].x, point[1].y);
}

inline void CWindow::ClientToScreen(LPPOINT lpPoint) const {
	AssertValidAsWindow();
	::ClientToScreen(m_hWnd, lpPoint);
}

inline bool CWindow::Create(bool bSubClass, const TCHAR* lpszClassName, const TCHAR* lpszWindowName, DWORD dwStyle,
	int x, int y, int nWidth, int nHeight, HWND hwndParent, HMENU hMenu /* = 0 */, LPVOID lpParam /* = 0L */) {
	AssertValid();

	HINSTANCE	hInstance;
	HWND		_hwndParent;

	if(IsWindow())
		return false;

	if(::IsWindow(hwndParent)) {
		_hwndParent = hwndParent;
		hInstance = reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE));
	} else {
		_hwndParent = 0;
		hInstance = reinterpret_cast<HINSTANCE>(hwndParent);
	}

	m_hWnd = ::CreateWindow(
		lpszClassName, lpszWindowName, dwStyle,
		x, y, nWidth, nHeight, _hwndParent, hMenu, hInstance, lpParam);
	if(m_hWnd == 0)
		return false;
	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CWindow::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CWindow::CreateCaret(HBITMAP hBitmap, int nWidth, int nHeight) {
	AssertValidAsWindow();
	return toBoolean(::CreateCaret(m_hWnd, hBitmap, nWidth, nHeight));
}

inline bool CWindow::CreateEx(bool bSubClass, DWORD dwExStyle, const TCHAR* lpszClassName, const TCHAR* lpszWindowName, DWORD dwStyle,
	int x, int y, int nWidth, int nHeight, HWND hwndParent, HMENU hMenu /* = 0 */, LPVOID lpParam /* = 0L */) {
	AssertValid();

	HINSTANCE	hInstance;
	HWND		_hwndParent;

	if(IsWindow())
		return false;

	if(::IsWindow(hwndParent)) {
		_hwndParent = hwndParent;
		hInstance = reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE));
	} else {
		_hwndParent = 0;
		hInstance = reinterpret_cast<HINSTANCE>(hwndParent);
	}

	m_hWnd = ::CreateWindowEx(
		dwExStyle, lpszClassName, lpszWindowName, dwStyle,
		x, y, nWidth, nHeight, _hwndParent, hMenu, hInstance, lpParam);
	if(m_hWnd == 0)
		return false;
	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CWindow::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CWindow::CreateGrayCaret(int nWidth, int nHeight) {
	AssertValidAsWindow();
	return toBoolean(::CreateCaret(m_hWnd, reinterpret_cast<HBITMAP>(1), nWidth, nHeight));
}

inline bool CWindow::CreateSolidCaret(int nWidth, int nHeight) {
	AssertValidAsWindow();
	return toBoolean(::CreateCaret(m_hWnd, 0, nWidth, nHeight));
}

inline LRESULT CWindow::DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValidAsWindow();
	return (m_OldProc != 0) ?
		::CallWindowProc(m_OldProc, m_hWnd, message, wParam, lParam)
		: ::DefWindowProc(m_hWnd, message, wParam, lParam);
}

inline bool CWindow::DestroyWindow() {
	AssertValid();
	return IsWindow() ? toBoolean(::DestroyWindow(m_hWnd)) : false;
}

inline HWND CWindow::Detach() {
	AssertValid();

	HWND	hWnd;
	WNDPROC	wp;	// for diagnotic

	if(m_OldProc != 0) {
		wp = reinterpret_cast<WNDPROC>(SetWindowLong(GWL_WNDPROC, reinterpret_cast<LPARAM>(m_OldProc)));
		assert(wp != m_OldProc);
		SetWindowLong(GWL_USERDATA, m_lOldUserData);
	}
	hWnd = m_hWnd;
	m_hWnd = 0;
	m_OldProc = 0;
	m_lOldUserData = 0L;

	return hWnd;
}

inline LRESULT CWindow::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	POINT pt;

	if(PreTranslateMessage(message, wParam, lParam))
		return false;

	switch(message) {
	case WM_MATTACH:
		OnAttached();
		return false;
	case WM_ACTIVATE:
		OnActivate(LOWORD(wParam), reinterpret_cast<HWND>(lParam), toBoolean(HIWORD(wParam)));
		break;
	case WM_CLOSE:
		OnClose();
		return 0L;
	case WM_COMMAND:
		return OnCommand(LOWORD(wParam), HIWORD(wParam), reinterpret_cast<HWND>(lParam));
	case WM_CONTEXTMENU:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		if(OnContextMenu(reinterpret_cast<HWND>(wParam), pt))
			return 0L;
		break;
	case WM_DESTROY:
		OnDestroy();
		return 0L;
	case WM_DRAWITEM:
		OnDrawItem(static_cast<UINT>(wParam), reinterpret_cast<LPDRAWITEMSTRUCT>(lParam));
		break;
	case WM_KILLFOCUS:
		OnKillFocus(reinterpret_cast<HWND>(wParam));
		return 0L;
	case WM_LBUTTONDOWN:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonDown(wParam, pt);
		break;
	case WM_LBUTTONDBLCLK:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonDblClk(wParam, pt);
		break;
	case WM_LBUTTONUP:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonUp(wParam, pt);
		break;
	case WM_MEASUREITEM:
		OnMeasureItem(static_cast<UINT>(wParam), reinterpret_cast<LPMEASUREITEMSTRUCT>(lParam));
		break;
	case WM_MOUSEMOVE:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnMouseMove(wParam, pt);
		break;
	case WM_NCDESTROY:
		if(m_OldProc != 0)
			SetWindowLong(GWL_WNDPROC, reinterpret_cast<LPARAM>(m_OldProc));
		m_hWnd = 0;
		return 0L;
	case WM_NOTIFY:
		return OnNotify(wParam, reinterpret_cast<LPNMHDR>(lParam));
	case WM_PAINT:
		OnPaint();
		break;
	case WM_SETCURSOR:
		if(OnSetCursor(reinterpret_cast<HWND>(wParam),
			static_cast<UINT>(LOWORD(lParam)), static_cast<UINT>(HIWORD(lParam))))
			return false;
		break;
	case WM_SETFOCUS:
		OnSetFocus(reinterpret_cast<HWND>(wParam));
		return 0L;
	case WM_SHOWWINDOW:
		OnShowWindow(toBoolean(wParam), lParam);
		return 0L;
	case WM_SIZE:
		OnSize(wParam, LOWORD(lParam), HIWORD(lParam));
		break;
	case WM_SYSCOLORCHANGE:
		OnSysColorChange();
		break;
	case WM_TIMER:
		OnTimer(wParam);
		break;
	default:
		break;
	}

	return DefWindowProc(message, wParam, lParam);
}

inline void CWindow::DragAcceptFiles(bool bAccept /* = true */) {
	AssertValidAsWindow();
	::DragAcceptFiles(m_hWnd, bAccept);
}

inline void CWindow::DrawMenuBar() {
	AssertValidAsWindow();
	::DrawMenuBar(m_hWnd);
}

inline bool CWindow::EnableScrollBar(int nSBFlags, UINT nArrowFlags /* = ESB_ENABLE_BOTH */) {
	AssertValidAsWindow();
	return toBoolean(::EnableScrollBar(m_hWnd, nSBFlags, nArrowFlags));
}

inline bool CWindow::EnableWindow(bool bEnable /* = true */) {
	AssertValidAsWindow();
	return toBoolean(::EnableWindow(m_hWnd, bEnable));
}

inline void CWindow::EndPaint(const LPPAINTSTRUCT lpPaint) {
	AssertValidAsWindow();
	::EndPaint(m_hWnd, lpPaint);
}

inline CWindow* CWindow::FindWindow(const TCHAR* lpszClassName, const TCHAR* lpszWindowName) {
	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::FindWindow(lpszClassName, lpszWindowName)))
		return &objWindow;
	else
		return 0;
}

inline bool CWindow::FlashWindow(bool bInvert) {
	AssertValidAsWindow();
	return toBoolean(::FlashWindow(m_hWnd, bInvert));
}

inline CWindow* CWindow::GetActiveWindow() {
	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetActiveWindow()))
		return &objWindow;
	else
		return 0;
}

inline CWindow* CWindow::GetCapture() {
	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetCapture()))
		return &objWindow;
	else
		return 0;
}

inline LPPOINT CWindow::GetCaretPos() {
	static POINT pt;

	::GetCaretPos(&pt);
	return &pt;
}

inline void CWindow::GetClientRect(LPRECT lpRect) const {
	AssertValidAsWindow();
	::GetClientRect(m_hWnd, lpRect);
}

inline HDC CWindow::GetDC() {
	AssertValidAsWindow();
	return ::GetDC(m_hWnd);
}

inline HDC CWindow::GetDCEx(HRGN hrgnClip, DWORD dwFlags) {
	AssertValidAsWindow();
	return ::GetDCEx(m_hWnd, hrgnClip, dwFlags);
}

inline CWindow* CWindow::GetDesktopWindow() {
	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetDesktopWindow()))
		return &objWindow;
	else
		return 0;
}

inline int CWindow::GetDlgCtrlID() const {
	AssertValidAsWindow();
	return ::GetDlgCtrlID(m_hWnd);
}

inline DWORD CWindow::GetExStyle() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::GetWindowLong(m_hWnd, GWL_EXSTYLE));
}

inline CWindow* CWindow::GetFocus() {
	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetFocus()))
		return &objWindow;
	else
		return 0;
}

inline HFONT CWindow::GetFont() const {
	AssertValidAsWindow();
	return reinterpret_cast<HFONT>(::SendMessage(m_hWnd, WM_GETFONT, 0, 0L));
}

inline CWindow* CWindow::GetForegroundWindow() {
	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetForegroundWindow()))
		return &objWindow;
	else
		return 0;
}

inline HICON CWindow::GetIcon(bool bBigIcon /* = true */) const {
	AssertValidAsWindow();
	return reinterpret_cast<HICON>(::SendMessage(m_hWnd, WM_GETICON, bBigIcon ? ICON_BIG : ICON_SMALL, 0L));
}

inline CWindow* CWindow::GetLastActivePopup() const {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetLastActivePopup(m_hWnd)))
		return &objWindow;
	else
		return 0;
}

inline HMENU CWindow::GetMenu() const {
	AssertValidAsWindow();
	return ::GetMenu(m_hWnd);
}

inline CWindow* CWindow::GetNext(UINT nFlag /* = GW_HWNDNEXT */) const {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetNextWindow(m_hWnd, nFlag)))
		return &objWindow;
	else
		return 0;
}

inline CWindow* CWindow::GetOwner() const {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetWindow(m_hWnd, GW_OWNER)))
		return &objWindow;
	else
		return 0;
}

inline CWindow* CWindow::GetParent() const {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetParent(m_hWnd)))
		return &objWindow;
	else
		return 0;
}

inline HWND CWindow::GetSafeHwnd() const {
	return (this != 0) ? m_hWnd : 0;
}

inline bool CWindow::GetScrollInfo(int nBar, LPSCROLLINFO lpScrollInfo, UINT nMask /* = SIF_ALL */) const {
	AssertValidAsWindow();

	lpScrollInfo->cbSize = sizeof(SCROLLINFO);
	lpScrollInfo->fMask = nMask;

	return toBoolean(::GetScrollInfo(m_hWnd, nBar, lpScrollInfo));
}

inline int CWindow::GetScrollLimit(int nBar) const {
	AssertValidAsWindow();

	int nLimit, nDummy;

	::GetScrollRange(m_hWnd, nBar, &nDummy, &nLimit);

	return nLimit;
}

inline int CWindow::GetScrollPos(int nBar) const {
	AssertValidAsWindow();
	return ::GetScrollPos(m_hWnd, nBar);
}

inline void CWindow::GetScrollRange(int nBar, int* pMinPos, int* pMaxPos) const {
	AssertValidAsWindow();
	::GetScrollRange(m_hWnd, nBar, pMinPos, pMaxPos);
}

inline int CWindow::GetScrollTrackPos(int nBar) const {
	AssertValidAsWindow();
	SCROLLINFO	si;
	return (GetScrollInfo(nBar, &si, SIF_TRACKPOS)) ? si.nTrackPos : -1;
}

inline DWORD CWindow::GetStyle() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::GetWindowLong(m_hWnd, GWL_STYLE));
}

inline HMENU CWindow::GetSystemMenu(bool bRevert) const {
	AssertValidAsWindow();
	return ::GetSystemMenu(m_hWnd, bRevert);
}

inline CWindow* CWindow::GetTopWindow() const {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetTopWindow(m_hWnd)))
		return &objWindow;
	else
		return 0;
}

inline bool CWindow::GetUpdateRect(LPRECT lpRect, bool bErase /* = false */) {
	AssertValidAsWindow();
	return toBoolean(::GetUpdateRect(m_hWnd, lpRect, bErase));
}

inline int CWindow::GetUpdateRgn(HRGN hRgn, bool bErase /* = false */) {
	AssertValidAsWindow();
	return ::GetUpdateRgn(m_hWnd, hRgn, bErase);
}

inline CWindow* CWindow::GetWindow(UINT nCmd) const {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::GetWindow(m_hWnd, nCmd)))
		return &objWindow;
	else
		return 0;
}

inline DWORD CWindow::GetWindowContextHelpId() const {
	AssertValidAsWindow();
	return ::GetWindowContextHelpId(m_hWnd);
}

inline HDC CWindow::GetWindowDC() {
	AssertValidAsWindow();
	return ::GetWindowDC(m_hWnd);
}

inline LONG CWindow::GetWindowLong(int nIndex) const {
	AssertValidAsWindow();
	return ::GetWindowLong(m_hWnd, nIndex);
}

inline bool CWindow::GetWindowPlacement(WINDOWPLACEMENT* lpwndpl) const {
	AssertValidAsWindow();
	return toBoolean(::GetWindowPlacement(m_hWnd, lpwndpl));
}

inline void CWindow::GetWindowRect(LPRECT lpRect) const {
	AssertValidAsWindow();
	::GetWindowRect(m_hWnd, lpRect);
}

inline int CWindow::GetWindowRgn(HRGN hRgn) const {
	AssertValidAsWindow();
	return ::GetWindowRgn(m_hWnd, hRgn);
}

inline int CWindow::GetWindowText(TCHAR* lpszText, int nMaxCount) const {
	AssertValidAsWindow();
	return ::GetWindowText(m_hWnd, lpszText, nMaxCount);
}

inline int CWindow::GetWindowTextLength() const {
	AssertValidAsWindow();
	return ::SendMessage(m_hWnd, WM_GETTEXTLENGTH, 0, 0L);
}

inline void CWindow::HideCaret() {
	AssertValidAsWindow();
	::HideCaret(m_hWnd);
}

inline bool CWindow::HiliteMenuItem(HMENU hMenu, UINT nItem, UINT nFlags) {
	AssertValidAsWindow();
	return toBoolean(::HiliteMenuItem(m_hWnd, hMenu, nItem, nFlags));
}

inline void CWindow::Invalidate(bool bErase /* = true */) {
	AssertValidAsWindow();
	::InvalidateRect(m_hWnd, 0, bErase);
}

inline void CWindow::InvalidateRect(LPRECT lpRect, bool bErase /* = true */) {
	AssertValidAsWindow();
	::InvalidateRect(m_hWnd, lpRect, bErase);
}

inline void CWindow::InvalidateRgn(HRGN hRgn, bool bErase /* = true */) {
	AssertValidAsWindow();
	::InvalidateRgn(m_hWnd, hRgn, bErase);
}

inline bool CWindow::IsChild(HWND hWnd) const {
	AssertValidAsWindow();
	return toBoolean(::IsChild(m_hWnd, hWnd));
}

inline bool CWindow::IsIconic() const {
	AssertValidAsWindow();
	return toBoolean(::IsIconic(m_hWnd));
}

inline bool CWindow::IsWindow() const {
	AssertValid();
	return toBoolean(::IsWindow(m_hWnd));
}

inline bool CWindow::IsWindowEnabled() const {
	AssertValidAsWindow();
	return toBoolean(::IsWindowEnabled(m_hWnd));
}

inline bool CWindow::IsWindowVisible() const {
	AssertValid();
	return toBoolean(::IsWindowVisible(m_hWnd));
}

inline bool CWindow::IsZoomed() const {
	AssertValidAsWindow();
	return toBoolean(::IsZoomed(m_hWnd));
}

inline bool CWindow::KillTimer(UINT nIDEvent) {
	AssertValidAsWindow();
	return toBoolean(::KillTimer(m_hWnd, nIDEvent));
}

inline bool CWindow::LockWindowUpdate() {
	AssertValidAsWindow();
	return toBoolean(::LockWindowUpdate(m_hWnd));
}

inline void CWindow::MapWindowPoints(HWND hWndTo, LPRECT lpRect) const {
	AssertValidAsWindow();

	POINT point[2];

	point[0].x = lpRect->left;
	point[0].y = lpRect->top;
	point[1].x = lpRect->right;
	point[1].y = lpRect->bottom;
	::MapWindowPoints(m_hWnd, hWndTo, point, 2);
	::SetRect(lpRect, point[0].x, point[0].y, point[1].x, point[1].y);
}

inline void CWindow::MapWindowPoints(HWND hWndTo, LPPOINT lpPoint, UINT nCount) const {
	AssertValidAsWindow();
	::MapWindowPoints(m_hWnd, hWndTo, lpPoint, nCount);
}

inline int CWindow::MessageBox(const TCHAR* lpszText, const TCHAR* lpszCaption /* = 0 */, UINT nType /* = MB_OK */) {
	AssertValidAsWindow();
	return ::MessageBox(m_hWnd, lpszText, lpszCaption, nType);
}

inline bool CWindow::ModifyStyle(DWORD dwRemove, DWORD dwAdd) {
	AssertValidAsWindow();

	DWORD dwStyle;

	dwStyle = GetStyle();
	dwStyle &= ~dwRemove;
	dwStyle |= dwAdd;
	if(::SetWindowLong(m_hWnd, GWL_STYLE, dwStyle) == 0)
		return false;
	return true;
}

inline bool CWindow::ModifyStyleEx(DWORD dwRemove, DWORD dwAdd) {
	AssertValidAsWindow();

	DWORD dwExStyle;

	dwExStyle = GetExStyle();
	dwExStyle &= ~dwRemove;
	dwExStyle |= dwAdd;
	if(::SetWindowLong(m_hWnd, GWL_EXSTYLE, dwExStyle) == 0)
		return false;
	return true;
}

inline void CWindow::MoveWindow(int x, int y, int nWidth, int nHeight, bool bRepaint /* = true */) {
	AssertValidAsWindow();
	::MoveWindow(m_hWnd, x, y, nWidth, nHeight, bRepaint);
}

inline void CWindow::Print(HDC hDC, DWORD dwFlags) const {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, WM_PRINT, reinterpret_cast<WPARAM>(hDC), dwFlags);
}

inline void CWindow::PrintClient(HDC hDC, DWORD dwFlags) const {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, WM_PRINTCLIENT, reinterpret_cast<WPARAM>(hDC), dwFlags);
}

inline void CWindow::MoveWindow(const RECT* lpRect, bool bRepaint /* = true */) {
	AssertValidAsWindow();
	::MoveWindow(m_hWnd,
		lpRect->left, lpRect->top, lpRect->right - lpRect->left,
		lpRect->bottom - lpRect->top, bRepaint);
}

inline LRESULT CWindow::PostMessage(UINT message, WPARAM wParam /* = 0 */, LPARAM lParam /* = 0L */) {
	AssertValidAsWindow();
	return ::PostMessage(m_hWnd, message, wParam, lParam);
}

inline bool CWindow::PreTranslateMessage(UINT message, WPARAM wParam, LPARAM lParam) {
	return false;
}

inline bool CWindow::RedrawWindow(
		LPRECT lpRectUpdate /* = 0 */, HRGN hrgnClip /* = 0 */,
		UINT nFlags /* = RDW_INVALIDATE | RDW_UPDATENOW | RDW_ERASE */) {
	AssertValidAsWindow();
	return toBoolean(::RedrawWindow(m_hWnd, lpRectUpdate, hrgnClip, nFlags));
}

inline HRESULT CWindow::RegisterDragDrop(IDropTarget* pDropTarget) {
	AssertValidAsWindow();
	return ::RegisterDragDrop(m_hWnd, pDropTarget);
}

inline bool CWindow::ReleaseCapture() {
	return toBoolean(::ReleaseCapture());
}

inline int CWindow::ReleaseDC(HDC hDC) {
	AssertValidAsWindow();
	return ::ReleaseDC(m_hWnd, hDC);
}

inline HRESULT CWindow::RevokeDragDrop() {
	AssertValidAsWindow();
	return ::RevokeDragDrop(m_hWnd);
}

inline void CWindow::ScreenToClient(LPRECT lpRect) const {
	AssertValidAsWindow();

	POINT	point[2];

	point[0].x = lpRect->left;
	point[0].y = lpRect->top;
	point[1].x = lpRect->right;
	point[1].y = lpRect->bottom;
	::ScreenToClient(m_hWnd, &point[0]);
	::ScreenToClient(m_hWnd, &point[1]);
	::SetRect(lpRect, point[0].x, point[0].y, point[1].x, point[1].y);
}

inline void CWindow::ScrollWindow(int xAmount, int yAmount, LPRECT lpRect /* = 0 */, LPRECT lpClipRect /* = 0 */) {
	AssertValidAsWindow();
	::ScrollWindow(m_hWnd, xAmount, yAmount, lpRect, lpClipRect);
}

inline int CWindow::ScrollWindowEx(
		int dx, int dy, LPRECT lpScrollRect, LPRECT lpClipRect,
		HRGN hrgnUpdate, LPRECT lpUpdateRect, UINT nFlags) {
	AssertValidAsWindow();
	return ::ScrollWindowEx(m_hWnd, dx, dy, lpScrollRect, lpClipRect, hrgnUpdate, lpUpdateRect, nFlags);
}

inline LRESULT CWindow::SendMessage(UINT message, WPARAM wParam /* = 0 */, LPARAM lParam /* = 0L */) {
	AssertValidAsWindow();
	return ::SendMessage(m_hWnd, message, wParam, lParam);
}

inline bool CWindow::SendNotifyMessage(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValidAsWindow();
	return toBoolean(::SendNotifyMessage(m_hWnd, message, wParam, lParam));
}

inline void CWindow::ScreenToClient(LPPOINT lpPoint) const {
	AssertValidAsWindow();
	::ScreenToClient(m_hWnd, lpPoint);
}

inline CWindow* CWindow::SetActiveWindow() {
	AssertValidAsWindow();

	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::SetActiveWindow(m_hWnd)))
		return &objWindow;
	else
		return 0;
}

inline CWindow* CWindow::SetCapture() {
	AssertValidAsWindow();

	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::SetCapture(m_hWnd)))
		return &objWindow;
	else
		return 0;
}

inline void CWindow::SetCaretPos(POINT pt) {
	::SetCaretPos(pt.x, pt.y);
}

inline int CWindow::SetDlgCtrlID(int nID) {
	AssertValidAsWindow();
	return static_cast<int>(::SetWindowLong(m_hWnd, GWL_ID, nID));
}

inline CWindow* CWindow::SetFocus() {
	AssertValidAsWindow();

	static CWindow	objWindow;
	
	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::SetFocus(m_hWnd)))
		return &objWindow;
	else
		return 0;
}

inline void CWindow::SetFont(HFONT hFont, bool bRedraw /* = true */) {
	SendMessage(WM_SETFONT,
		reinterpret_cast<WPARAM>(hFont), MAKELPARAM(bRedraw, 0));
}

inline bool CWindow::SetForegroundWindow() {
	AssertValidAsWindow();
	return toBoolean(::SetForegroundWindow(m_hWnd));
}

inline HICON CWindow::SetIcon(HICON hIcon, bool bBigIcon /* = true */) {
	return reinterpret_cast<HICON>(SendMessage(WM_SETICON, bBigIcon ? ICON_BIG : ICON_SMALL));
}

inline bool CWindow::SetLayeredWindowAttributes(COLORREF clrKey, BYTE bAlpha, DWORD dwFlags) {
	AssertValidAsWindow();
	if(m_hUser32Dll == 0)
		m_hUser32Dll = ::LoadLibrary(_T("User32.dll"));
	if(m_hUser32Dll != 0 && m_pfnSetLayeredWindowAttributes == 0)
		m_pfnSetLayeredWindowAttributes =
			reinterpret_cast<SLWA>(::GetProcAddress(m_hUser32Dll, "SetLayeredWindowAttributes"));
	return (m_pfnSetLayeredWindowAttributes != 0) ?
		toBoolean((*m_pfnSetLayeredWindowAttributes)(m_hWnd, clrKey, bAlpha, dwFlags)) : false;
}

inline bool CWindow::SetMenu(HMENU hMenu) {
	AssertValidAsWindow();
	return toBoolean(::SetMenu(m_hWnd, hMenu));
}

inline CWindow* CWindow::SetParent(HWND hwndNewParent) {
	AssertValidAsWindow();

	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::SetParent(m_hWnd, hwndNewParent)))
		return &objWindow;
	else
		return 0;
}

inline void CWindow::SetRedraw(bool bRedraw /* = true */) {
	SendMessage(WM_SETREDRAW, bRedraw);
}

inline bool CWindow::SetScrollInfo(int nBar, LPSCROLLINFO lpScrollInfo, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	return toBoolean(::SetScrollInfo(m_hWnd, nBar, lpScrollInfo, bRedraw));
}

inline int CWindow::SetScrollPos(int nBar, int nPos, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	return ::SetScrollPos(m_hWnd, nBar, nPos, bRedraw);
}

inline void CWindow::SetScrollRange(int nBar, int nMinPos, int nMaxPos, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	::SetScrollRange(m_hWnd, nBar, nMinPos, nMaxPos, bRedraw);
}

inline UINT CWindow::SetTimer(
		UINT nIDEvent, UINT nElapse, void (CALLBACK* lpfnTimer)(HWND, UINT, UINT, DWORD)) {
	AssertValidAsWindow();
	return ::SetTimer(m_hWnd, nIDEvent, nElapse, lpfnTimer);
}

inline bool CWindow::SetWindowContextHelpId(DWORD dwContextHelpId) {
	AssertValidAsWindow();
	return toBoolean(::SetWindowContextHelpId(m_hWnd, dwContextHelpId));
}

inline LONG CWindow::SetWindowLong(int nIndex, LONG dwNewLong) {
	AssertValidAsWindow();
	return ::SetWindowLong(m_hWnd, nIndex, dwNewLong);
}

inline bool CWindow::SetWindowPlacement(const WINDOWPLACEMENT* lpwndpl) {
	AssertValidAsWindow();
	return toBoolean(::SetWindowPlacement(m_hWnd, lpwndpl));
}

inline bool CWindow::SetWindowPos(HWND hwndInsertAfter, int x, int y, int cx, int cy, UINT nFlags) {
	AssertValidAsWindow();
	return toBoolean(::SetWindowPos(m_hWnd, hwndInsertAfter, x, y, cx, cy, nFlags));
}

inline bool CWindow::SetWindowPos(HWND hwndInsertAfter, LPRECT lpRect, UINT nFlags) {
	AssertValidAsWindow();
	return toBoolean(
		::SetWindowPos(
			m_hWnd, hwndInsertAfter,
			lpRect->left, lpRect->top,
			lpRect->right - lpRect->left,
			lpRect->bottom - lpRect->top, nFlags));
}

inline int CWindow::SetWindowRgn(const HRGN hRgn, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	return ::SetWindowRgn(m_hWnd, hRgn, bRedraw);
}

inline void CWindow::SetWindowText(const TCHAR* lpszText) {
	AssertValidAsWindow();
	::SetWindowText(m_hWnd, lpszText);
}

inline void CWindow::ShowCaret() {
	AssertValidAsWindow();
	::ShowCaret(m_hWnd);
}

inline void CWindow::ShowOwnedPopups(bool bShow /* = true */) {
	AssertValidAsWindow();
	::ShowOwnedPopups(m_hWnd, bShow);
}

inline void CWindow::ShowScrollBar(int nBar, bool bShow /* = true */) {
	AssertValidAsWindow();
	::ShowScrollBar(m_hWnd, nBar, bShow);
}

inline bool CWindow::ShowWindow(UINT nCmdShow) {
	AssertValidAsWindow();
	return toBoolean(::ShowWindow(m_hWnd, nCmdShow));
}

inline void CWindow::UnlockWindowUpdate() {
	AssertValidAsWindow();
	::LockWindowUpdate(0);
}

inline bool CWindow::UpdateLayeredWindow(HDC hDest, POINT* pptDest,
		SIZE* pSize, HDC hSrc, POINT* pptSrc, COLORREF clrKey, BLENDFUNCTION* pBlend, DWORD dwFlags) {
	AssertValidAsWindow();
	if(m_hUser32Dll == 0)
		m_hUser32Dll = ::LoadLibrary(_T("User32.dll"));
	if(m_hUser32Dll != 0 && m_pfnUpdateLayeredWindow == 0)
		m_pfnUpdateLayeredWindow =
			reinterpret_cast<ULW>(::GetProcAddress(m_hUser32Dll, "UpdateLayeredWindow"));
	return (m_pfnUpdateLayeredWindow != 0) ?
		toBoolean((*m_pfnUpdateLayeredWindow)(m_hWnd, hDest, pptDest, pSize, hSrc, pptSrc, clrKey, pBlend, dwFlags)) : false;
}

inline void CWindow::UpdateWindow() {
	AssertValidAsWindow();
	::UpdateWindow(m_hWnd);
}

inline void CWindow::ValidateRect(LPRECT lpRect) {
	AssertValidAsWindow();
	::ValidateRect(m_hWnd, lpRect);
}

inline void CWindow::ValidateRgn(HRGN hRgn) {
	AssertValidAsWindow();
	::ValidateRgn(m_hWnd, hRgn);
}

inline CWindow* CWindow::WindowFromPoint(POINT pt) {
	static CWindow	objWindow;

	if(objWindow.IsWindow())
		objWindow.Detach();
	if(objWindow.Attach(::WindowFromPoint(pt)))
		return &objWindow;
	else
		return 0;
}

inline LRESULT CALLBACK CWindow::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	CWindow*	pWnd = 0;

	if(message == WM_MATTACH) {
		pWnd = reinterpret_cast<CWindow*>(lParam);
		pWnd->SetWindowLong(GWL_USERDATA, lParam);
		assert(hWnd == pWnd->m_hWnd);
		return pWnd->DispatchEvent(WM_MATTACH, 0, 0L);
	} else {
		pWnd = reinterpret_cast<CWindow*>(::GetWindowLong(hWnd, GWL_USERDATA));
		return (pWnd != 0) ? pWnd->DispatchEvent(message, wParam, lParam) : false;
	}
}

inline void CWindow::OnActivate(UINT nState, HWND hwndPrevious, bool bMinimized) {
}

inline void CWindow::OnAttached() {
}

inline void CWindow::OnClose() {
	DestroyWindow();
}

inline bool CWindow::OnCommand(WORD wID, WORD wNotifyCode, HWND hwndCtrl) {
	return false;
}

inline bool CWindow::OnContextMenu(HWND hWnd, POINT pt) {
	return false;
}

inline void CWindow::OnDrawItem(UINT idCtrl, LPDRAWITEMSTRUCT lpDrawItem) {
}

inline void CWindow::OnDestroy() {
}

inline void CWindow::OnKillFocus(HWND hwndNew) {
}

inline void CWindow::OnLButtonDown(UINT nFlags, POINT pt) {
}

inline void CWindow::OnLButtonDblClk(UINT nFlags, POINT pt) {
}

inline void CWindow::OnLButtonUp(UINT nFlags, POINT pt) {
}

inline void CWindow::OnMeasureItem(UINT idCtrl, LPMEASUREITEMSTRUCT lpMI) {
}

inline void CWindow::OnMouseMove(UINT nFlags, POINT pt) {
}

inline bool CWindow::OnNotify(int idCtrl, LPNMHDR lpNMHDR) {
	return false;
}

inline void CWindow::OnPaint() {
}

inline bool CWindow::OnSetCursor(HWND hWnd, UINT nHitTest, UINT message) {
	return false;
}

inline void CWindow::OnSetFocus(HWND hwndOld) {
}

inline void CWindow::OnShowWindow(bool bShow, UINT nStatus) {
}

inline void CWindow::OnSize(UINT nType, int cx, int cy) {
}

inline void CWindow::OnSysColorChange() {
}

inline void CWindow::OnTimer(UINT nIDEvent) {
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _M_WINDOW_H_ */

/* [EOF] */