/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java
 ||
 || It is assumed that a Tamino database called "mydb" has
 || been created and is running.
 || For storing data the default collection ino:etc is used
 || which should only be used for examples/test code.
 || We do not recommend this for real applications.
 ||
 || The following tasks are performed:
 ||
 || - establish a connection to the Tamino database
 || - obtain a non-XML accessor
 || - start a local transaction mode and
 ||   o retrieve first object stored
 ||   o update object just inserted
 ||   o on success commit transaction otherwise roll back
 || - switch back to auto-commit mode
 || - query updated document
 || - delete document
 || - close the connection
 */
package com.softwareag.tamino.db.api.examples.greeting;


import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TDeleteException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TNonXMLObjectAccessor;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TNonXMLObject;
import com.softwareag.tamino.db.api.objectModel.TNonXMLObjectIterator;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;


public class ProcessNonXMLGreeting {
	
	public ProcessNonXMLGreeting(String databaseURI,String collection) throws TConnectionException {
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		connection = connectionFactory.newConnection( databaseURI );
		// Obtain the concrete TXMLObjectAccessor with an underlying JDOM object model
		accessor = connection.newNonXMLObjectAccessor( TAccessLocation.newInstance( collection ) );
	}
	
	private void performInsert(TNonXMLObject nonXMLObject) throws TException  {
		// TResponse represents an access response from Tamino
		TResponse response = null;
		// Insert the non-XML document
		try  {
			// Invoke the insert operation and obtain the response
			response = accessor.insert( nonXMLObject );
			// Show the collection, doctype and id
			System.out.println( "Insert succeeded, ino:collection:" + nonXMLObject.getCollection() +
							   ", ino:doctype:" + nonXMLObject.getDoctype() +
							   ", ino:id:" + nonXMLObject.getId() );
		}
		catch (TInsertException insertException)  {
			showAccessFailure( insertException );
			// Rethrow the exception
			throw insertException;
		}
	}
	
	private void performUpdate(TQuery query,Reader updatedContent) throws TException  {
		TLocalTransaction localTransaction = null;
		// Do the query and update within a transaction
		try  {
			// Switch to the local transaction mode
			localTransaction = connection.useLocalTransactionMode();
			// Invoke the query to obtain the document and to lock it
			TResponse response = accessor.query( query );
			// Obtain the TNonXMLObjectIterator from the response
			TNonXMLObjectIterator iterator = response.getNonXMLObjectIterator();
			if ( !iterator.hasNext() )
				return;
			TNonXMLObject nonXMLObject = iterator.next();
			nonXMLObject.readFrom( updatedContent );
			// Invoke the update
			response = accessor.update( nonXMLObject );
			System.out.println( "Update succeeded!" );
			// Commit the transaction
			localTransaction.commit();
		}
		// TQueryException and TUpdateException are both derived from TAccessorException,
		// so we simply use the base class here
		catch (TAccessorException accessorException)  {
			showAccessFailure( accessorException );
			localTransaction.rollback();
			throw accessorException;
		}
		finally  {
			// Switch back to the auto commit mode
			connection.useAutoCommitMode();
		}
	}
	
	private void performQuery(TQuery query) throws TException  {
		// Now lets make a query on the updated object
		try  {
			TResponse response = accessor.query( query );
			// Obtain the TNonXMLObjectIterator from the response
			TNonXMLObjectIterator iterator = response.getNonXMLObjectIterator();
			if ( !iterator.hasNext() )
				return;
			TNonXMLObject nonXMLObject = iterator.next();
			// Write the Non-XML content to a ByteArrayOutputStream
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			nonXMLObject.writeTo( outputStream );
			System.out.println( "Queried document, Content:" + outputStream );
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			throw queryException;
		}
	}
	
	private void performDelete(TQuery query) throws TException  {
		// Finally, lets delete the document again
		try  {
			TResponse response = accessor.delete( query );
			System.out.println( "Deleted the document!" );
		}
		catch (TDeleteException deleteException)  {
			showAccessFailure( deleteException );
			throw deleteException;
		}
	}
	
	// Build a JDOM Element from the given XML
	private Reader toPlainTextContent(String text) throws IOException  {
		return new StringReader( text );
	}
	
	// Show the reason for the access failure.
	private void showAccessFailure(TAccessorException accessorException)  {
		// Obtain an access failure message telling the exact reason if Tamino request failed
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			System.out.println( "Access failed:" + accessFailure );
		else
			System.out.println( "Access failed:" + accessorException.getMessage() );
	}
	
	private void show()  throws TException {
		try  {
			// Instantiate a TNonXMLObject with a JDOM object model
			TNonXMLObject nonXMLObject = TNonXMLObject.newInstance( toPlainTextContent( NON_XML ) ,
																   null ,
																   "ino:nonXML" ,
																   "NonXMLGreeting" ,
																   "text/plain"  );
			// Initiate the insert.
			performInsert( nonXMLObject );
			// Construct a query expression so that the inserted document can be referenced
			TQuery query = TQuery.newInstance( nonXMLObject.getDoctype() + "[@ino:id=\"" +
											  nonXMLObject.getId() + "\"]" );
			// Initiate the update
			performUpdate( query , toPlainTextContent( NON_XML + " updated :-)" ) );
			// Initiate the query
			performQuery( query );
			// Initiate the removal
			performDelete( query );
		}
		catch (IOException ioException)  {
			ioException.printStackTrace();
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		finally  {
			// Close the connection
			connection.close();
		}
	}
	
	public static void main(String[] args) throws TException  {
		ProcessNonXMLGreeting processNonXMLGreeting = new ProcessNonXMLGreeting( DATABASE_URI , "ino:etc" );
		processNonXMLGreeting.show();
	}
	
	// URI of the Tamino database, please edit accordingly
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
	
	// Non-XML document to be written to the connected database
	private final static String NON_XML = "Greeting by='XMLGreetingApplication':Hello World";
	
	// The database connection
	TConnection connection = null;
	
	// The accessor, here a TNonXMLObjectAccessor
	TNonXMLObjectAccessor accessor = null;
	
}
