/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java.
 ||
 || Assumes that there is a Tamino database called "mydb".
 || The example simply uses the default collection ino:etc.
 || This collection should only be used for examples/test code.
 || It shouldn't generally be used for real applications.
 ||
 || The example does the following:
 ||
 || establishes a connection to the Tamino database
 || obtains an XML accessor
 || inserts a new document of <Greeting>...</Greeting> into ino:etc
 || starts a local transaction mode
 || queries for the newly inserted document
 || updates the newly inserted document
 || commits the transaction if succeeded otherwise performs a rollback
 || switches back to auto-commit mode
 || queries for the updated document
 || Finally the document is deleted
 || closes the connection
 */
package com.softwareag.tamino.db.api.examples.greeting;


import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TDeleteException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.accessor.TXQuery;
import com.softwareag.tamino.db.api.accessor.TXQueryException;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectIterator;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectModel;
import com.softwareag.tamino.db.api.objectModel.dom4j.TDOM4JObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.StringReader;
import java.io.StringWriter;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;



public class ProcessXMLGreetingDOM4J {
	
	public ProcessXMLGreetingDOM4J(String databaseURI,String collection) throws TConnectionException {
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		connection = connectionFactory.newConnection( databaseURI );
		// Instantiate the specific TDOM4JObjectModel
		TXMLObjectModel dom4jObjectModel = TDOM4JObjectModel.getInstance();
		// Do the object model registration.
		TXMLObjectModel.register( dom4jObjectModel );
		// Obtain the concrete TXMLObjectAccessor with an underyling DOM4J object model
		accessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) , dom4jObjectModel );
	}
	
	private void performInsert(TXMLObject xmlObject) throws TException  {
		// TResponse represents an access response from Tamino.
		TResponse response = null;
		// Do the insert of the <Test></Test> document
		try  {
			// Invoke the insert operation and obtain the response.
			response = accessor.insert( xmlObject );
			// Show the collection, doctype and id
			System.out.println( "Insert succeeded, ino:collection:" + xmlObject.getCollection() + ", ino:doctype:" + xmlObject.getDoctype() +
							   ", ino:id:" + xmlObject.getId() );
		}
		catch (TInsertException insertException)  {
			showAccessFailure( insertException );
			// Rethrow the exception
			throw insertException;
		}
	}
	
	private void performUpdate(TQuery query,String updatedText) throws TException  {
		TLocalTransaction localTransaction = null;
		// Do the query and update within a transaction
		try  {
			// Switch to the local transaction mode
			localTransaction = connection.useLocalTransactionMode();
			// Invoke the query to obtain the document and to lock it
			TResponse response = accessor.query( query );
			// Obtain the TXMLObject from the response
			TXMLObject xmlObject = response.getFirstXMLObject();
			if ( xmlObject == null )
				return;
			// Obtain the DOM4J element and update its content
			Element element = (Element)xmlObject.getElement();
			element.setText( updatedText );
			// Invoke the update
			response = accessor.update( xmlObject );
			System.out.println( "Update succeeded!" );
			// Commit the transaction
			localTransaction.commit();
		}
		// TQueryException and TUpdateException are both derived from TAccessorException so we simply use the base class here
		catch (TAccessorException accessorException)  {
			showAccessFailure( accessorException );
			localTransaction.rollback();
			throw accessorException;
		}
		finally  {
			// Switch back to the auto commit mode
			connection.useAutoCommitMode();
		}
	}
	
	private void performXUpdate(TXQuery updateXQuery) throws TException  {
		TLocalTransaction localTransaction = null;
		// Do the query and update within a transaction
		try  {
			// Switch to the local transaction mode
			localTransaction = connection.useLocalTransactionMode();
			// Invoke the query to obtain the document and to lock it
			TResponse response = accessor.xquery( updateXQuery );
//			// Obtain the TXMLObject from the response
//			TXMLObject xmlObject = response.getFirstXMLObject();
//			if ( xmlObject == null )
//				return;
//			// Obtain the DOM4J element and update its content
//			Element element = (Element)xmlObject.getElement();
//			element.setText( updatedText );
//			// Invoke the update
//			response = accessor.update( xmlObject );
			System.out.println( "XUpdate succeeded!" );
			// Commit the transaction
			localTransaction.commit();
		}
		// TXQueryException and TUpdateException are both derived from TAccessorException so we simply use the base class here
		catch (TAccessorException accessorException)  {
			showAccessFailure( accessorException );
			localTransaction.rollback();
			throw accessorException;
		}
		finally  {
			// Switch back to the auto commit mode
			connection.useAutoCommitMode();
		}
	}
	
	private void performQuery(TQuery query) throws TException  {
		// Now lets make a query on the updated object
		try  {
			TResponse response = accessor.query( query );
			TXMLObject xmlObject = response.getFirstXMLObject();
			// Write the XML content to a StringWriter
			StringWriter stringWriter = new StringWriter();
			xmlObject.writeTo( stringWriter );
			System.out.println( "Queried document:" + stringWriter );
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			throw queryException;
		}
	}
	
	private void performXQuery(TXQuery xquery) throws TException  {
		// Now lets make a xquery on the updated object
		try  {
			TResponse response = accessor.xquery( xquery );
			TXMLObjectIterator iterator = response.getXMLObjectIterator();
			while ( iterator.hasNext() ) {
				TXMLObject xmlObject = iterator.next();
				// Write the XML content to a StringWriter
				StringWriter stringWriter = new StringWriter();
				xmlObject.writeTo( stringWriter );
				System.out.println( "XQueried document(s):" + stringWriter );
			}
		}
		catch (TXQueryException xqueryException)  {
			showAccessFailure( xqueryException );
			throw xqueryException;
		}
	}
	
	private void performDelete(TQuery query) throws TException  {
		// Finally, lets delete the document again
		try  {
			TResponse response = accessor.delete( query );
			System.out.println( "Deleted the document!" );
		}
		catch (TDeleteException deleteException)  {
			showAccessFailure( deleteException );
			throw deleteException;
		}
	}
	
	// Build a DOM4J Element from the given XML.
	private Element toDOM4J(String xml) throws DocumentException  {
		SAXReader saxReader = new SAXReader();
		Document document = saxReader.read( new StringReader( xml ) );
		return document.getRootElement();
	}
	
	// Show the reason for the access failure.
	private void showAccessFailure(TAccessorException accessorException)  {
		// Obtain an access failure message telling the exact reason if Tamino request failed.
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			System.out.println( "Access failed:" + accessFailure );
		else
			System.out.println( "Access failed:" + accessorException.getMessage() );
	}
	
	private void show()  throws TException {
		try  {
			// Instantiate a TXMLObject with an underyling DOM4J specific object model
			TXMLObject xmlObject = TXMLObject.newInstance( toDOM4J( XML ) );
			// Initiate the insert.
			performInsert( xmlObject );
			
			// Build a query so that the inserted document can be referenced
			TQuery query = TQuery.newInstance( xmlObject.getDoctype() + "[@ino:id=\"" + xmlObject.getId() + "\"]" );
			// Initiate the update
			performUpdate( query , "Hello World, updated :-)" );
			// Initiate the query
			performQuery( query );
			// Initiate a xquery
			TXQuery xquery = TXQuery.newInstance( "input()/Greeting" );
			performXQuery( xquery );
			// Initiate the removal
			performDelete( query );
		}
		catch (DocumentException docException)  {
			docException.printStackTrace();
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		finally  {
			// Close the connection.
			connection.close();
		}
	}
	
	public static void main(String[] args) throws TException  {
		// Change the database uri here to work with an appropiate one.
		ProcessXMLGreetingDOM4J processXMLGreeting = new ProcessXMLGreetingDOM4J( DATABASE_URI , "ino:etc" );
		processXMLGreeting.show();
	}
	
	// Constant for the database URI. Please edit to use your uri of interest.
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
	
	// Constant for the applications XML message that shall be written to the connected database.
	private final static String XML = "<Greeting by='ProcessGreetingApplication'>Hello World</Greeting>";
	
	// The database connection.
	private TConnection connection = null;
	
	// The concrete accessor, here a high level TXMLObjectAccessor.
	private TXMLObjectAccessor accessor = null;
	
}

