/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 * InsertConstraintCheckDOM4J.java
 */
package com.softwareag.tamino.db.api.examples.jazz;



import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TIsolationLevel;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectModel;
import com.softwareag.tamino.db.api.objectModel.dom4j.TDOM4JObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.FileInputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.XPath;


public class InsertConstraintCheckDOM4J {
	
	/** Creates new InsertUniqueDOM4J */
	public InsertConstraintCheckDOM4J (String databaseURI,String collection) throws TConnectionException {
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		connection = connectionFactory.newConnection( databaseURI );
		// Instantiate the specific TDOM4JObjectModel
		TXMLObjectModel dom4jObjectModel = TDOM4JObjectModel.getInstance();
		// Do the object model registration.
		TXMLObjectModel.register( dom4jObjectModel );
		// Obtain the concrete TXMLObjectAccessor with an underyling DOM4J object model
		accessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) , dom4jObjectModel );
		// Set isolation level to "shared"
		connection.setIsolationLevel(TIsolationLevel.SHARED) ;
	}
	
	private void performInsert(TXMLObject xmlObject) throws Exception  {
		try  {
			// Invoke the insert operation
			accessor.insert( xmlObject );
		} catch (TInsertException insertException)	{
			showAccessFailure( insertException );
		}
	}
	
	// process query
	private TResponse processQuery(String s) throws Exception {
		TQuery query = TQuery.newInstance( s );
		try {
			// Invoke the query operation.
			return accessor.query( query );
		}
		catch (TQueryException queryException) {
			// Tell about the reason for the failure.
			showAccessFailure(queryException);
			return null;
		}
	}
	
	// Show the reason for the access failure.
	private void showAccessFailure(TAccessorException accessorException) throws Exception {
		// Obtain an access failure message telling the exact reason if Tamino request failed.
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null ) {
			throw new Exception( "Access failed:" + accessFailure );
		} else {
			throw new Exception( "Access failed:" + accessorException.getMessage() );
		}
	}
	
	// Convert into date format
	private Date toDate(String s) throws java.text.ParseException {
		// determine formatting string
		String f = (s.indexOf("T",0) >= 0 ?
					"yyyy-MM-dd'T'HH:mm:ss" :
						(s.indexOf(":", 1) >= 0 ?
						 "HH:mm:ss" : "yyyy-MM-dd"));
		// check for explicit time zone
		int p = Math.max (s.indexOf("+", 8),s.indexOf("-", 8));
		if (p >= 0) {
			f += "z"; // indicate time zone in formatting string
			s = s.substring(0,p-1)+"GMT"+s.substring(p);   // keep SimpleDateFormat happy
		}
		else if (s.charAt(s.length()-1) == 'Z') {
			// check for UTC time zone
			f += "z";
			s = s.substring(0,s.length()-1) + "UTC";
		}
		// create SimpleDateFormat object
		SimpleDateFormat df = new SimpleDateFormat ( f );
		// and use it as a parser
		return df.parse(s);
	}
	
	
	
	// process transaction
	private void processTransaction(String filename)  throws Exception {
		TLocalTransaction myTransaction = null;
		boolean abortTransaction = false;
		
		// Read file into a DOM4J Tamino XML object.
		// Instantiate an empty TXMLObject instance related to the DOM4J object model.
		TXMLObject collaborationObject = TXMLObject.newInstance( TDOM4JObjectModel.getInstance() );
		// read from File
		collaborationObject.readFrom( new FileInputStream(filename ));
		// get DOM4J document
		Document collaborationDoc = (Document) collaborationObject.getDocument();
		// Set local transaction mode and get a transaction object
		myTransaction = connection.useLocalTransactionMode();
		// initialize start date
		Element startDateElement = null;
		//search with XPATH
		XPath xpathSelector = DocumentHelper.createXPath("//from");
		startDateElement = (Element) xpathSelector.selectSingleNode(collaborationDoc);
		if (startDateElement == null){
			// alternatively, get the "time" element
			xpathSelector = DocumentHelper.createXPath("//time");
			startDateElement  = (Element) xpathSelector.selectSingleNode(collaborationDoc);
		}
		// get start date value
		String startDateValue = startDateElement.getText();
		// convert to Date
		Date startDate = toDate(startDateValue);
		// Get jazzMusician elements
		xpathSelector = DocumentHelper.createXPath("//jazzMusician");
		List collaborateurs = xpathSelector.selectNodes(collaborationDoc);
		// now loop over the "jazzMusician" children
		Iterator iterator = collaborateurs.iterator();
		while(iterator.hasNext()) {
			// get a single "jazzMusician" child
			Element collaborateurElement = (Element) iterator.next();
			// check if this item has content
			if (collaborateurElement.nodeIterator().hasNext()) {
				// get the string content
				String collaborateurID = collaborateurElement.getText();
				// Perform query for jazzMusicians identified by collaborateurID
				TResponse response = processQuery("jazzMusician[@ID"+"='" + collaborateurID + "']");
				// Process the musician document if we have one
				if (!response.hasFirstXMLObject()) {
					abortTransaction = true;
					System.out.println("Error: Referenced jazzMusician "+collaborateurID+" does not exist");
				} else {
					// get first (and only) result document
					TXMLObject jazzMusicianObject = response.getFirstXMLObject();
					// Get top level DOM4J element
					Document jazzMusicianDoc = (Document) jazzMusicianObject.getDocument();
					// get birthDate
					xpathSelector = DocumentHelper.createXPath("//birthDate");
					Element birthDateElement = (Element) xpathSelector.selectSingleNode(jazzMusicianDoc);
					
					// get string value
					String birthDateValue = birthDateElement.getText();
					// convert to date
					Date birthDate = toDate(birthDateValue);
					// compare with startDate
					if (startDate.compareTo(birthDate) <= 0) {
						abortTransaction = true;
						// Report violation of integrity constraint
						System.out.println("Error: Collaboration start date before birth date of jazz musician "+collaborateurID);
					}
				}
			}
		}
		if (abortTransaction) {
			myTransaction.rollback();
			// Report abort of operation
			System.out.println("Error: Insert not performed");
		} else {
			performInsert( collaborationObject );
			myTransaction.commit();
			// Show the collection, doctype and id
			System.out.println("Message: Insert succeeded, ino:collection=" + collaborationObject.getCollection() + ", ino:doctype=" + collaborationObject.getDoctype() +", ino:id=" + collaborationObject.getId() );
		}
		connection.close();
	}
	
	/**
	 * @param args the command line arguments
	 */
	
	public static void main(String[] args) throws Exception  {
		try {
			InsertConstraintCheckDOM4J insertConstraintCheck = new InsertConstraintCheckDOM4J( DATABASE_URI , COLLECTION );
			// perform the transaction
			insertConstraintCheck.processTransaction(args[0]);
		} catch(Throwable thr) {
			thr.printStackTrace();
		}
	}
	
	// Constant for the database URI.
	private final static String DATABASE_URI = "http://localhost/tamino/jazz";
	
	// Constant for the collection.
	private final static String COLLECTION = "encyclopedia";
	
	
	// The database connection.
	private TConnection connection = null;
	
	// The concrete accessor, here a high level TXMLObjectAccessor.
	private TXMLObjectAccessor accessor = null;
}
