/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


package com.softwareag.tamino.db.api.examples.message;

import com.softwareag.tamino.db.api.objectModel.sax.TSAXDocument;
import com.softwareag.tamino.db.api.objectModel.sax.TSAXElement;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;


/**
 ** Message implements both TSAXElement and TSAXDocument. It represents a simple XML document
 ** of the form &lt;Message&gt;myContent&lt;/Message&gt;.
 **/
public class Message implements TSAXElement , TSAXDocument {
	
	// Constant for the tag name.
	public final static String TAG_NAME = "Message";
	
	/**
	 ** Appends the specified string content to the internal content.
	 **/
	public void appendContent(String content) {
		this.content.append( content );
	}
	
	/**
	 ** Appends the specified character array content to the internal content.
	 **/
	public void appendContent(char ch[], int start, int length) {
		content.append( ch , start , length );
	}
	
	/**
	 ** Appends the specified attribute with its value to the "Message" tag.
	 **/
	public void appendAttribute(String name,String value) {
		if ( name.equals( "ino:id" ) )
			setId( value );
		else if ( name.equals( "ino:docname" ) )
			setDocname( value );
		else {
			attributeNames.add( name );
			attributeValues.add( value );
		}
	}
	
	/**
	 ** Gets the current content.
	 **/
	public String getContent() {
		return content.toString();
	}
	
	/**
	 ** Gets the value for the specified attribute. If none is given null is returned.
	 **/
	public String getAttribute(String name) {
		int index = attributeNames.indexOf( name );
		return ( index != -1 ) ? (String)attributeValues.get( index ) : null;
	}
	
	/**
	 ** Gets all the attribute names by an iterator that iterates over the names as strings.
	 **/
	public Iterator getAttributeNames() {
		return attributeNames.iterator();
	}
	
	/**
	 ** Clears all the content and attributes currently given.
	 **/
	public void clear() {
		clearContent();
		clearAttributes();
	}
	
	/**
	 ** Clears the content currently given.
	 **/
	public void clearContent() {
		content.delete( 0 , content.length() );
	}
	
	/**
	 ** Clears the attributes currently given.
	 **/
	public void clearAttributes() {
		attributeNames.clear();
		attributeValues.clear();
		setId( "" );
		setDocname( "" );
	}
	
	/**
	 ** Gets the document root element as a TSAXElement instance. An implementation of
	 ** this method is not always required. If no meaningful implementation exists, null
	 ** may be returned. This would the case in which the processing of the SAX events
	 ** does not result in a tree structure containing elements.
	 **
	 ** @return a TSAXElement instance representing the document root element, null if no
	 **         such element exists.
	 **
	 **/
	public TSAXElement getRootElement() {
		return this;
	}
	
	/**
	 ** Sets the ino:docname on the underlying element. An implementation of this
	 ** method is required for a fully functional TXMLObject. This method corresponds
	 ** to the setDocname method of TXMLObject.
	 **
	 ** @param docname the ino:docname attribute of the XML document.
	 **/
	public void setDocname(String docname) {
		this.docname = ( docname != null ) ? docname : "";
	}
	
	/**
	 ** Sets the ino:id on the underlying element. An implementation of this
	 ** method is required for a fully functional TXMLObject. This method corresponds
	 ** to the setId method of TXMLObject.
	 **
	 ** @param id the ino:id attribute of the XML document.
	 **/
	public void setId(String id) {
		this.id = ( id != null ) ? id : "";
	}
	
	/**
	 ** Gets the doctype of the underlying SAX element if it exists. A doctype is required
	 ** for Tamino update and insert operations specifying a docname. This method returns
	 ** the doctype representing the default doctype used for these type of operations.
	 ** You can use the setDoctype method of TXMLObject to set the doctype explicitly.
	 **
	 ** @return the doctype of the sax element if it exist, otherwise an empty string.
	 **/
	public String getDoctype() {
		return doctype;
	}
	
	/**
	 ** Gets the ino:docname of the underlying SAX element, if it exists.
	 **
	 ** @return the docname of the sax element if it exist, otherwise an empty string.
	 **/
	public String getDocname() {
		return docname;
	}
	
	/**
	 ** Gets the ino:id if of the underlying SAX element, if it exists.
	 **
	 ** @return the ino:id of the sax element, otherwise an empty string.
	 **/
	public String getId() {
		return id;
	}
	
	/**
	 ** Serializes the document content as a character output stream into the given
	 ** writer. A meaninful implementation of this method is required if TXMLOject
	 ** instances of the SAX object modelo in question require serialization as a
	 ** character stream, e.g. for the writeTo method of TXMLObject.
	 **
	 ** @param writer a Writer instance to be used to write the document content as
	 **        a character output stream.
	 **
	 **/
	public void writeTo(Writer writer) {
		try {
			writer.write( "<" + getDoctype() );
			if ( id.length() > 0 )
				writeAttributeTo( writer , "ino:id" , getId() );
			if ( docname.length() > 0 )
				writeAttributeTo( writer , "ino:docname" , getDocname() );
			Iterator nameIter = getAttributeNames();
			while ( nameIter.hasNext() ) {
				String name = (String)nameIter.next();
				writeAttributeTo( writer , name , getAttribute( name ) );
			}
			writer.write( ">" );
			String cnt = getContent();
			if ( cnt.length() > 0 )
				writer.write( cnt );
			writer.write( "</" + getDoctype() + ">" );
		}
		catch (IOException ioException) {
		}
	}
	
	private void writeAttributeTo(Writer writer,String name,String value) throws IOException {
		writer.write( " " + name + "=\"" + value + "\"" );
	}
	
	// The ino:id.
	private String id = "";
	
	// The ino:docname.
	private String docname = "";
	
	// The doctype set to the tag name.
	private String doctype = TAG_NAME;
	
	// The list of all attribute names.
	private ArrayList attributeNames = new ArrayList();
	
	// The list of the attributes values.
	private ArrayList attributeValues = new ArrayList();
	
	// The content currently given.
	private StringBuffer content = new StringBuffer();
	
}
