/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


package com.softwareag.tamino.db.api.examples.message;

import java.util.ArrayList;
import java.util.Iterator;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;


/**
 ** MessageDefaultHandler extends the sax based DefaultHandler. Its purpose is to handle all events
 ** for the Message XML documents. For each such document MessageDefaultHandler instance is supposed
 ** to instantiate an internal Message instance. After completing event handling the MessageDefaultHandler
 ** provides access to the first Message by getFirstMessage and to the sequence of all established
 ** Messages by getMessageIterator.
 **/

public class MessageDefaultHandler extends DefaultHandler {
	
	/**
	 ** Default constructor.
	 **/
	public MessageDefaultHandler() {
	}
	
	/**
	 ** Handles the startDocument event. Clears the previous content.
	 **/
	public void startDocument() throws SAXException {
		reset();
	}
	
	/**
	 **	Receive notification of the beginning of an element.
	 **
	 **	@param      uri         The Namespace URI, or the empty string if the element has no Namespace URI or if Namespace processing is not being performed.
	 **	@param      localName   The local name (without prefix), or the empty string if Namespace processing is not being performed.
	 **	@param      qName       The qualified name (with prefix), or the empty string if qualified names are not available.
	 **	@param      atts         The attributes attached to the element. If there are no attributes, it shall be an empty Attributes object.
	 **
	 **	@exception  SAXException  Any SAX exception, possibly wrapping another exception.
	 **/
	public final void startElement(String uri,String localName,String qName,Attributes attributes) throws SAXException {
		if ( this.hasStarted == false && localName.equals( Message.TAG_NAME ) )  {
			this.hasStarted = true;
			message = new Message();
		}
		for ( int i = 0 ; i < attributes.getLength() ; ++i )
			message.appendAttribute( attributes.getQName( i ) , attributes.getValue( i ) );
	}
    
	/**
	 **	Receive notification of the ending of an element.
	 **
	 **	@param      uri         The Namespace URI, or the empty string if the element has no Namespace URI or if Namespace processing is not being performed.
	 **	@param      localName   The local name (without prefix), or the empty string if Namespace processing is not being performed.
	 **	@param      qName       The qualified name (with prefix), or the empty string if qualified names are not available.
	 **
	 **	@exception  SAXException  Any SAX exception, possibly wrapping another exception.
	 **/
	public final void endElement (String uri, String localName, String qName) throws SAXException {
		if ( this.hasStarted && localName.equals( Message.TAG_NAME ) ) {
			messages.add( message );
			message = null;
			this.hasStarted = false;
		}
	}
    
	/**
	 **	Receive notification of character data.
	 **
	 **	@param      ch      The characters from the XML document.
	 **	@param      start   The start position in the array.
	 **	@param      length  The number of characters to read from the array.
	 **
	 **	@exception  SAXException  Any SAX exception, possibly wrapping another exception.
	 **/
	public final void characters(char ch[], int start, int length) throws SAXException {
		if ( this.hasStarted )
			message.appendContent( ch , start , length );
	}
	
	/**
	 ** Gets the first message of the sequence that has been established during event handling. If none
	 ** is given, null is returned.
	 **/
	public Message getFirstMessage() {
		return messages.size() > 0 ? (Message)messages.get( 0 ) : null;
	}
	
	/**
	 ** Returns an Iterator instance that can be used to iterate over the sequence of Message instances
	 ** established during event handling.
	 **/
	public Iterator getMessageIterator() {
		return messages.iterator();
	}
	
	/**
	 ** Clears all the Message content currently given.
	 **/
	public void reset() {
		hasStarted = false;
		message = null;
		messages.clear();
	}
	
	// Indicates if the processing of a Message document has started.
	private boolean hasStarted = false;
	
	// The Message instance that needs to be processed.
	private Message message = null;
	
	// The container of all Message instances that have been established.
	private ArrayList messages = new ArrayList();
	
}
