/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java.
 ||
 || Assumes that there is a Tamino database called "mydb".
 || The example simply uses the default collection ino:etc.
 || This collection should only be used for examples/test code.
 || It shouldn't generally be used for real applications.
 ||
 || The example does the following:
 ||
 || establishes a connection to the Tamino database
 || obtains a system accessor and prints out some system information
 || obtains an XML accessor
 || reads some XML documents from files and inserts them into the collection ino:etc
 || creates two queries, one for all person documents and one for all persons named "Atkins"
 || queries all persons and list them
 || use a boolean query to check if any person named "Atkins" is stored in the database
 || if such persons are found, delete them
 || queries again all persons and list them
 || delete the rest of the persons
 || closes the connection
 */
package com.softwareag.tamino.db.api.examples.person;


import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TDeleteException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TSystemAccessor;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.io.TStreamWriteException;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectIterator;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectModel;
import com.softwareag.tamino.db.api.objectModel.dom4j.TDOM4JObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.List;
import org.dom4j.Element;


public class ProcessPersonsDOM4J {
	
	// use a sytem accessor, to check, if the database is alive and print some system information to stdout
	protected static boolean checkServerAndPrintSystemInformation(TConnection connection) throws TAccessorException {
		// Obtain the TSystemAccesor
		TSystemAccessor systemaccessor = connection.newSystemAccessor();
		
		if (!systemaccessor.isServerAlive()) {
			return false;
		} else {
			System.out.println( " server is alive" );
			System.out.println( "\nHere is some systeminformation" );
			System.out.println( "------------------------------\n" );
			System.out.println( "The Tamino server hosting " + DATABASE_URI + " is version " + systemaccessor.getServerVersion() );
			System.out.println( "(Server API version: " + systemaccessor.getServerAPIVersion() + ", Tamino API for Java version: " + systemaccessor.getAPIVersion() + ")\n" );
			return true;
		}
	}
	
	// Helpermethod to get the text of a given tagname from a DOM4J tree
	private static String getDOM4JElementTextByTagName(Element element, String tagname) {
		List list = element.elements( tagname );
		Element elem = (Element)list.get(0);
		return elem.node(0).getStringValue();
	}
	
	// List some data about a person to stdout
	private static void printPerson(TXMLObject xmlObject) throws TStreamWriteException {
		
		// Obtain the DOM4J root element
		Element e = (Element)xmlObject.getElement();
		
		String surname = getDOM4JElementTextByTagName(e, "surname");
		String firstname= getDOM4JElementTextByTagName(e, "firstname");
		System.out.print( surname + ", " + firstname);
		
		System.out.print( " (" );
		System.out.print( "ino:id=\"" + xmlObject.getId() +"\" "  );
		System.out.print( "collection=\"" + xmlObject.getCollection() +"\" " );
		System.out.print( "doctype=\"" + xmlObject.getDoctype() +"\" " );
		System.out.println( ")" );
	}
	
	// read a given filename into an TXMLObject, store it in the database and print the person
	protected static TXMLObject performInsertPersonFromFile(TXMLObjectAccessor accessor, String filename) throws TException, FileNotFoundException  {
		// Instantiate an empty TXMLObject instance related to the DOM4J object model.
		TXMLObject xmlObject = TXMLObject.newInstance( TDOM4JObjectModel.getInstance() );
		
		// Read a document out of the classpath and insert it into the database.
		InputStream myInputStream = (new ProcessPersonsDOM4J()).getClass().getResourceAsStream( filename );
		xmlObject.readFrom( myInputStream );
		
		try {
			// Invoke the insert operation and obtain the response.
			TResponse response = accessor.insert( xmlObject );
			
			System.out.print( "  Inserted: " );
			printPerson(xmlObject);
		}
		catch (TInsertException insertException)  {
			System.out.print( "\nCan't insert: " );
			printPerson( xmlObject );
			System.out.println( "  Reason: " + insertException.getAccessFailureException().getMessageLine() + "\n" );
		}
		return xmlObject;
	}
	
	// take a given query, put the count() function around and return the result as String
	private static String performCountQuery(TXMLObjectAccessor accessor, TQuery query) throws TException  {
		try  {
			// Build a new query string
			TQuery countquery = TQuery.newInstance( "count(" + query.getExpression() + ")" );
			return accessor.query( countquery ).getQueryContentAsString();
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			throw queryException;
		}
	}
	
	// perform a query and list all persons, which are returned
	protected static void performQueryAndListPersons(TXMLObjectAccessor accessor, TQuery query) throws TException  {
		// Now lets make the query
		try  {
			TResponse response = accessor.query( query );
			TXMLObjectIterator iterator = response.getXMLObjectIterator();
			System.out.print( "The query \"" + query + "\" returns " );
			if (!iterator.hasNext()) {
				System.out.println( "no documents!" );
			} else {
				System.out.println( performCountQuery( accessor, query ) + " documents, which are:" );
				while (iterator.hasNext()) {
					TXMLObject xmlObject = iterator.next();
					System.out.print( "  " );
					printPerson(xmlObject);
				}
			}
		} catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			throw queryException;
		}
	}
	
	// delete all documents for a given query with a given accessor
	protected static void performDelete(TXMLObjectAccessor accessor, TQuery query) throws TException  {
		// Finally, lets delete the document again
		try  {
			TResponse response = accessor.delete( query );
			System.out.println("\nDeleted all documents for query \"" + query + "\"");
		}
		catch (TDeleteException deleteException)  {
			showAccessFailure( deleteException );
			throw deleteException;
		}
	}
	
	// Show the reason for the access failure.
	private static void showAccessFailure(TAccessorException accessorException)  {
		// Obtain an access failure message telling the exact reason if Tamino request failed.
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			System.out.println( "Access failed with:" + accessFailure );
		else
			System.out.println( "Access failed:" + accessorException.getMessage() );
	}
	
	public static void main(String[] args) throws TException  {
		//TTrace.setActiveTrace(true);
		// Print out a program header to stdout
		System.out.println( "\nProcessPersonsDOM4J sample programm" );
		System.out.println( "==============================" );
		System.out.print( "Connecting to Tamino database " + DATABASE_URI + ", ..." );
		
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		TConnection connection = connectionFactory.newConnection( DATABASE_URI );
		
		// Check if the connection is available and print out some system information
		if ( !checkServerAndPrintSystemInformation( connection ) )
			return;
		
		// Instantiate the specific TDOM4JObjectModel
		TXMLObjectModel dom4jObjectModel = TDOM4JObjectModel.getInstance();
		// Do the object model registration.
		TXMLObjectModel.register( dom4jObjectModel );
		// Obtain a TXMLObjectAccessor to the "ino:etc" collection with an underyling DOM4J object model
		TXMLObjectAccessor accessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( "ino:etc" ) , dom4jObjectModel );
		
		// Print header for database operations
		System.out.println( "Insert and query and delete in default collection \"ino:etc\"" );
		System.out.println( "-----------------------------------------------------------\n" );
		
		try  {
			// Instantiate an empty TXMLObject instance related to the DOM4J object model.
			TXMLObject xmlObject = TXMLObject.newInstance( TDOM4JObjectModel.getInstance() );
			
			// read person documents from files and insert them into the "people" collection
			System.out.println( "Reading documents from file and insert into database\n" );
			
			performInsertPersonFromFile( accessor, "person1.xml" );
			performInsertPersonFromFile( accessor, "person2.xml" );
			performInsertPersonFromFile( accessor, "person3.xml" );
			performInsertPersonFromFile( accessor, "person4.xml" );
			xmlObject = performInsertPersonFromFile( accessor, "person5.xml" );
			
			// Build a query to reference all documents of doctype "person"
			TQuery queryall = TQuery.newInstance( xmlObject.getDoctype() );
			
			// Build a query to reference documents of doctype "person" with surname = "Atkins"
			TQuery querysome = TQuery.newInstance( xmlObject.getDoctype() + "[//surname='Atkins']" );
			
			System.out.println("");
			// Initiate a query to count and list all instances of a certain Doctype
			performQueryAndListPersons( accessor, queryall );
			
			// Create a boolean query and check if persons named "Atkins" are stored
			TQuery queryAtkins = TQuery.newInstance( "//surname='Atkins'" );
			String queryAtkinsResult = accessor.query( queryAtkins ).getQueryContentAsString();
			System.out.println( "\nThe query \"" + queryAtkins + "\" returns \"" +  queryAtkinsResult + "\"" );
			boolean AtkinsAvailable = new Boolean( queryAtkinsResult ).booleanValue();
			
			// Do some processing, if persons named "Atkins" are stored
			if (AtkinsAvailable) {
				System.out.println("So list and then delete all \"Atkins\" documents\n");
				// Initiate a query to count and list some instances of a certain Doctype
				performQueryAndListPersons( accessor, querysome);
				
				// Initiate the removal of all "Atkins"
				performDelete( accessor, querysome );
				
				System.out.println("");
				// Initiate a query to count and list all instances of a certain Doctype
				performQueryAndListPersons( accessor, queryall );
			}
			
			// Initiate the removal
			performDelete( accessor, queryall );
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		catch (FileNotFoundException filenotfoundException)  {
			filenotfoundException.printStackTrace();
		}
		
		// Close the connection.
		connection.close();
	}
	
	// Constant for the database URI. Please edit to use your uri of interest.
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
}

