/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java
 ||
 || It is assumed that a Tamino database called "mydb" has
 || been created and is running.
 || For storing data a collection "people" is created and used.
 ||
 || The following tasks are performed:
 ||
 || - establish a connection to the Tamino database
 || - obtain a system accessor and print out system information
 || - obtain an XML Schema (TSD3) accessor
 || - read an XML Schema from a file in the class path and insert
 ||   it into the database, creating the collection "people"
 || - obtain an XML accessor
 || - read XML documents from files and insert them into the database
 ||   collection "people ; one insertion will fail
 || - construct a query expression to query all person documents
 || - query all person documents
 || - delete all person documents
 || - close the connection

 */
package com.softwareag.tamino.db.api.examples.person;

import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TSchemaDefinition3Accessor;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.examples.person.ProcessPersons;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.dom.TDOMObjectModel;
import java.io.FileNotFoundException;
import java.io.InputStream;


public class ProcessPersonsWithSchema {
	
	public static void main(String[] args) throws TException  {
		
		// Print out a program header to stdout
		System.out.println( "\nProcessPersonsWithSchema sample programm" );
		System.out.println( "========================================" );
		System.out.print( "Connecting to Tamino database " + DATABASE_URI + ", ..." );
		
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// and obtain the connection to the database
		TConnection connection = connectionFactory.newConnection( DATABASE_URI );
		
		// Check if the connection is available and print out some system information
		if ( !ProcessPersons.checkServerAndPrintSystemInformation( connection ) )
			return;
		
		// Print header for database operations
		System.out.println( "Create collection \"people\" insert and delete some documents" );
		System.out.println( "-----------------------------------------------------------\n" );
		
		// Instantiate an empty TXMLObject instance using the DOM object model
		TXMLObject schemaXmlObject = TXMLObject.newInstance( TDOMObjectModel.getInstance() );
		
		try {
			
			System.out.println( "Reading TSD3 schema from file and insert into database" );
			
			// Read XML Schema (TSD3) from a file into a TXMLObject
			// This is not used in this sample!
//			schemaXmlObject.readFrom( new FileReader( "person.tsd" ) );
			
			// Read a document present in the class path and insert it into the database.
			InputStream myInputStream = (new
										 ProcessPersonsWithSchema()).getClass().getResourceAsStream( "person.tsd" );
			schemaXmlObject.readFrom( myInputStream );
			
			// Obtain a SchemaAccessor, here a TSD3 Accessor
			TSchemaDefinition3Accessor TSD3Accessor =
			connection.newSchemaDefinition3Accessor( TDOMObjectModel.getInstance() );
			
			// define new collection and new schema in database
			TSD3Accessor.define( schemaXmlObject );
			System.out.println( "\n  Inserted the schema for collection \"people\" and doctype \"person\"" );
			
			// print the XML Schema to stdout
//			schemaXmlObject.writeTo(System.out);
			
			// Obtain a TXMLObjectAccessor to the "people" collection using the DOM object model
			TXMLObjectAccessor peopleAccessor = connection.newXMLObjectAccessor(
				TAccessLocation.newInstance( "people" ) ,
				TDOMObjectModel.getInstance() );
			
			// Instantiate an empty TXMLObject instance using the DOM object model
			TXMLObject xmlObject = TXMLObject.newInstance( TDOMObjectModel.getInstance() );
			
			// read person documents from files and insert them into the "people" collection
			System.out.println( "\nReading documents from files and insert into database\n" );
			
			ProcessPersons.performInsertPersonFromFile( peopleAccessor, "person1.xml" );
			ProcessPersons.performInsertPersonFromFile( peopleAccessor, "person2.xml" );
			ProcessPersons.performInsertPersonFromFile( peopleAccessor, "person3.xml" );
			ProcessPersons.performInsertPersonFromFile( peopleAccessor, "person4.xml" );
			xmlObject = ProcessPersons.performInsertPersonFromFile( peopleAccessor, "person5.xml" );
			
			// Build a query to reference all documents of doctype "person"
			TQuery queryall = TQuery.newInstance( xmlObject.getDoctype() );
			
			System.out.println("");
			// Initiate a query to count and list all instances of a certain Doctype
			ProcessPersons.performQueryAndListPersons( peopleAccessor, queryall );
			
			// Initiate the removal of all documents
			ProcessPersons.performDelete( peopleAccessor, queryall );
			
			// delete collection with all content
			TSD3Accessor.undefine( "people", null );
			System.out.println("\nDeleted collection \"people\" with all content");
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		catch (FileNotFoundException filenotfoundException)  {
			filenotfoundException.printStackTrace();
		}
		
		// Close the connection
		connection.close();
	}
	
	// URI of the Tamino database, please edit accordingly
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
}

