/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java.
 ||
 || Assumes that there is a Tamino database called "mydb".
 || The example simply uses the default collection ino:etc.
 || This collection should only be used for examples/test code.
 || It shouldn't generally be used for real applications.
 ||
 || The example does the following:
 ||
 || establishes a connection to the Tamino database
 || obtains a system accessor and prints out some system information
 || obtains an XML Schema (TSD3) accessor
 || reads a XMl Schema from a file and inserts it into the database, creating the collection "people"
 || obtains an XML accessor
 || reads some XML documents from files and inserts them into the collection "people", one insertion fails
 || creates a query for all person documents
 || queries all persons and list them
 || delete all persons
 || closes the connection
 */
package com.softwareag.tamino.db.api.examples.person;


import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TSchemaDefinition3Accessor;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectModel;
import com.softwareag.tamino.db.api.objectModel.dom4j.TDOM4JObjectModel;
import java.io.FileNotFoundException;
import java.io.InputStream;


public class ProcessPersonsWithSchemaDOM4J {
	
	public static void main(String[] args) throws TException  {
		
		// Print out a program header to stdout
		System.out.println( "\nProcessPersonsWithSchemaDOM4J sample programm" );
		System.out.println( "========================================" );
		System.out.print( "Connecting to Tamino database " + DATABASE_URI + ", ..." );
		
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// and obtain the connection to the database
		TConnection connection = connectionFactory.newConnection( DATABASE_URI );
		
		// Check if the connection is available and print out some system information
		if ( !ProcessPersonsDOM4J.checkServerAndPrintSystemInformation( connection ) )
			return;
		
		// Print header for database operations
		System.out.println( "Create collection \"people\" insert and delete some documents" );
		System.out.println( "-----------------------------------------------------------\n" );
		
		// Instantiate an empty TXMLObject instance related to the DOM4J object model.
		TXMLObject schemaXmlObject = TXMLObject.newInstance( TDOM4JObjectModel.getInstance() );
		
		try {
			
			System.out.println( "Reading TSD3 schema from file and insert into database" );
			
			// Read XML Schema (TSD3) from a file into a TXMLObject
//			schemaXmlObject.readFrom( new FileReader( "person.tsd" ) );
			
			// Read a document out of the classpath and insert it into the database.
			InputStream myInputStream = (new ProcessPersonsWithSchemaDOM4J()).getClass().getResourceAsStream( "person.tsd" );
			schemaXmlObject.readFrom( myInputStream );
			
			// Obtain a SchemaAccessor, here a TSD3 Accessor
			TSchemaDefinition3Accessor TSD3Accessor = connection.newSchemaDefinition3Accessor( TDOM4JObjectModel.getInstance() );
			
			// define new collection and new schema in database
			TSD3Accessor.define( schemaXmlObject );
			System.out.println( "\n  Inserted the schema for collection \"people\" and doctype \"person\"" );
			
			// print the XML Schema to stdout
			// schemaXmlObject.writeTo(System.out);
			
			// Instantiate the specific TDOM4JObjectModel
			TXMLObjectModel dom4jObjectModel = TDOM4JObjectModel.getInstance();
			// Do the object model registration.
			TXMLObjectModel.register( dom4jObjectModel );
			// Obtain a TXMLObjectAccessor to the "people" collection with an underyling DOM object model
			TXMLObjectAccessor peopleAccessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( "people" ) , TDOM4JObjectModel.getInstance() );
			
			// Instantiate an empty TXMLObject instance related to the DOM4J object model.
			TXMLObject xmlObject = TXMLObject.newInstance( TDOM4JObjectModel.getInstance() );
			
			// read person documents from files and insert them into the "people" collection
			System.out.println( "\nReading documents from files and insert into database\n" );
			
			ProcessPersonsDOM4J.performInsertPersonFromFile( peopleAccessor, "person1.xml" );
			ProcessPersonsDOM4J.performInsertPersonFromFile( peopleAccessor, "person2.xml" );
			ProcessPersonsDOM4J.performInsertPersonFromFile( peopleAccessor, "person3.xml" );
			ProcessPersonsDOM4J.performInsertPersonFromFile( peopleAccessor, "person4.xml" );
			xmlObject = ProcessPersonsDOM4J.performInsertPersonFromFile( peopleAccessor, "person5.xml" );
			
			// Build a query to reference all documents of doctype "person"
			TQuery queryall = TQuery.newInstance( xmlObject.getDoctype() );
			
			System.out.println("");
			// Initiate a query to count and list all instances of a certain Doctype
			ProcessPersonsDOM4J.performQueryAndListPersons( peopleAccessor, queryall );
			
			// Initiate the removal of all documents
			ProcessPersonsDOM4J.performDelete( peopleAccessor, queryall );
			
			// delete collection with all content
			TSD3Accessor.undefine( "people", null );
			System.out.println("\nDeleted collection \"people\" with all content");
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		catch (FileNotFoundException filenotfoundException)  {
			filenotfoundException.printStackTrace();
		}
		
		// Close the connection.
		connection.close();
	}
	
	// Constant for the database URI. Please edit to use your uri of interest.
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
}

