/* This file contains EUC-JP text */
#include <gtk/gtk.h>
#include <anthy/anthy.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "fossil.h"

#define WORDS_STRUCT 0
#define WORD_COLUMN_UTF8 1
#define YOMI_COLUMN_UTF8 2
#define WTYPE_COLUMN 3
#define FREQ_COLUMN 4
#define CHECK_COLUMN 5
#define FIX_COLUMN 6
#define NR_COLUMNS 7

#define MAX_CANDIDATES 1000
#define NR_BUCKETS 32768

#define CHECKED_ITEM_FILE "checked_words"

struct words {
    char *index;
    char *wt;
    int freq;
    char *word;
    /**/
    int is_top;
    int checked;
    int fixed;
    /**/
    struct words *next;
    struct words *next_chain;
};

struct dict {
    struct words *bucket[NR_BUCKETS];
};

struct app_win {
    /**/
    GtkWidget *top;
    GtkWidget *next_button;
    GtkWidget *save_button;
    GtkWidget *quit_button;
    /**/
    GtkWidget *text_entry;
    /**/
    GtkWidget *words_view;
    GtkTreeModel *words_model;
    /**/
    GtkWidget *note_book;
    /**/
    GtkWidget *res_view;
    GtkTreeModel *res_model;
    /**/
    int argc;
    char **argv;
    /**/
    struct dict dict;
    struct words words;
    /**/
    anthy_context_t ac;
    char *cands[MAX_CANDIDATES];
    /**/
    GtkWidget *conf_win;
};

static guchar *
euc_to_utf8 (const guchar *str)
{
    GError *error = NULL;
    gchar *result;
 
    result = g_convert (str, -1,
			"UTF-8", "EUC-JP",
			NULL, NULL, &error);
    if (!result) {
	g_warning ("g_convert fail(%s)\n", error->message);
	g_error_free (error);
    }
 
    return result;
}

static int
str_hash(char *str, int h)
{
    while (*str) {
	h += (*str) * (*str);
	h += 17;
	h |= h >> 16;
	str++;
    }
    return h;
}

static int
word_hash(char *index, char *word, char *wt)
{
    int h = 0;
    h = str_hash(index, h);
    h = str_hash(word, h);
    h = str_hash(wt, h);
    return h % NR_BUCKETS;
}

static struct words *
find_word(struct dict *dict, char *index,
	  char *word, char *wt)
{
    int b = word_hash(index, word, wt);
    struct words *w;
    for (w = dict->bucket[b]; w; w = w->next_chain) {
	if (!strcmp(index, w->index) &&
	    !strcmp(word, w->word) &&
	    !strcmp(wt, w->wt)) {
	    return w;
	}
    }
    return NULL;
}

static void
fill_wt(struct words *word, char *wt)
{
    char *tmp = strchr(wt, '*');
    if (tmp) {
	*tmp = 0;
	tmp ++;
	word->wt = strdup(wt);
	word->freq = atoi(tmp);
    } else {
	word->wt = strdup(wt);
	word->freq = 1;
    }
}

static void
add_word(struct app_win *win,
	 char *index, char *wt, char *word)
{
    struct words w_tmp;
    struct words *w;
    int b;
    fill_wt(&w_tmp, wt);
    w = find_word(&win->dict, index, word, w_tmp.wt);
    /**/
    if (w) {
	if (w_tmp.freq > w->freq) {
	    w->freq = w_tmp.freq;
	}
	return ;
    }
    /**/
    w = malloc(sizeof(struct words));
    *w = w_tmp;
    w->index = strdup(index);
    w->word = strdup(word);
    w->checked = 0;
    w->fixed = 0;
    w->is_top = 0;
    /* link */
    w->next = win->words.next;
    win->words.next = w;
    /**/
    b = word_hash(index, word, w->wt);
    w->next_chain = win->dict.bucket[b];
    win->dict.bucket[b] = w;
}

static int
load_words_fp(FILE *fp, struct app_win *win)
{
    char buf[1024];
    int nr = 0;
    while (fgets(buf, 1024, fp)) {
	char *str = buf;
	char *index, *tmp;
	char *wt = "";
	if (buf[0] == '#') {
	    continue;
	}
	buf[strlen(buf) - 1] = 0;
	index = strsep(&str, " ");
	while ((tmp = strsep(&str, " "))) {
	    if (tmp[0] == '#') {
		if (tmp[1] != '_') {
		    wt = tmp;
		}
	    } else {
		add_word(win, index, wt, tmp);
		nr ++;
	    }
	}
    }
    return nr;
}

static void
update_model(struct app_win *win)
{
    struct words *w;
    GtkTreeIter iter;
    int i;
    for (w = win->words.next; w; w = w->next) {
	if (w->checked) {
	    continue;
	}
	gtk_list_store_append(GTK_LIST_STORE(win->words_model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(win->words_model), &iter,
			   WORDS_STRUCT, w,
			   YOMI_COLUMN_UTF8, euc_to_utf8(w->index),
			   WORD_COLUMN_UTF8, euc_to_utf8(w->word),
			   WTYPE_COLUMN, w->wt,
			   FREQ_COLUMN, w->freq,
			   CHECK_COLUMN, FALSE,
			   FIX_COLUMN, FALSE,
			   -1);
    }
}

static void
load_words(struct app_win *win)
{
    int i;
    int nr = 0;
    for (i = 1; i < win->argc; i++) {
	char *fn = win->argv[i];
	FILE *fp = fopen(fn, "r");
	if (!fp) {
	    continue;
	}
	nr += load_words_fp(fp, win);
	fclose(fp);
    }
    printf("%d words\n", nr);
}

static void
quit(GtkWidget *widget, gpointer data)
{
    gtk_main_quit();
}

static void
next_cb(GtkWidget *widget, struct app_win *win)
{
    GtkTreeIter iter;
    GtkTreeSelection *selection;
    GValue value = {0, };
    gboolean checked;
    struct words *w;
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(win->words_view));
    if (!selection ||
	!gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	return ;
    }
    gtk_list_store_set (GTK_LIST_STORE (win->words_model), &iter,
			CHECK_COLUMN, 1, -1);
    gtk_tree_model_get_value(win->words_model, &iter,
			     WORDS_STRUCT, &value);
    w = g_value_get_pointer(&value);
    w->checked = 1;
    g_value_unset(&value);
    /**/
    gtk_tree_model_iter_next(win->words_model, &iter);
    gtk_tree_selection_select_iter(selection, &iter);
}

static void
save_cb(GtkWidget *widget, struct app_win *win)
{
    struct words *w;
    FILE *fp = fopen(CHECKED_ITEM_FILE, "w");
    if (!fp) {
	printf("failed to open %s.\n", CHECKED_ITEM_FILE);
	return ;
    }
    for (w = win->words.next; w; w = w->next) {
	if (w->checked) {
	    char *fix = "ok";
	    if (w->fixed) {
		if (w->is_top) {
		    fix = "down";
		} else {
		    fix = "up";
		}
	    }
	    fprintf(fp, "%s %s %s %s\n", w->index, w->word, w->wt, fix);
	}
    }
    fclose(fp);
}

static void
build_check_buttons(struct app_win *win, GtkWidget *hbox)
{
    int i;
    GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(hbox), vbox);
    /**/
    win->next_button = gtk_button_new_with_label("next");
    gtk_container_add(GTK_CONTAINER(vbox), win->next_button);
    gtk_widget_show(win->next_button);
    win->save_button = gtk_button_new_with_label("save");
    gtk_container_add(GTK_CONTAINER(vbox), win->save_button);
    gtk_widget_show(win->save_button);
    win->quit_button = gtk_button_new_with_label("quit");
    gtk_container_add(GTK_CONTAINER(vbox), win->quit_button);
    gtk_widget_show(win->quit_button);
    /**/
    g_signal_connect(G_OBJECT(win->next_button), "clicked",
		     G_CALLBACK(next_cb), win);
    g_signal_connect(G_OBJECT(win->save_button), "clicked",
		     G_CALLBACK(save_cb), win);
    g_signal_connect(G_OBJECT(win->quit_button), "clicked",
		     G_CALLBACK(quit), NULL);
    /**/
    gtk_widget_show(vbox);
}

static GtkTreeModel *
create_words_model()
{
    GtkListStore *model;
    model = gtk_list_store_new(NR_COLUMNS,
			       G_TYPE_POINTER,
			       G_TYPE_STRING,
			       G_TYPE_STRING,
			       G_TYPE_STRING,
			       G_TYPE_UINT,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN);

    return GTK_TREE_MODEL(model);
}

static void
fix_toggled(GtkCellRendererToggle *toggle,
	    gchar *path_str,
	    struct app_win *win)
{
    GtkTreeIter iter;
    GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
    gboolean fixed;
    struct words *w;
    GValue value = {0, };

    gtk_tree_model_get_iter (win->words_model, &iter, path);
    gtk_tree_model_get (win->words_model, &iter, FIX_COLUMN, &fixed, -1);

    fixed = !fixed;
    /**/
    gtk_tree_model_get_value(win->words_model, &iter,
			     WORDS_STRUCT, &value);
    w = g_value_get_pointer(&value);
    w->fixed = fixed;
    g_value_unset(&value);
    /* set to model */
    gtk_list_store_set (GTK_LIST_STORE (win->words_model), &iter,
			FIX_COLUMN, fixed, -1);
    gtk_tree_path_free (path);
}

static void
check_toggled(GtkCellRendererToggle *toggle,
	      gchar *path_str,
	      struct app_win *win)
{
    GtkTreeIter iter;
    GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
    gboolean fixed;
    struct words *w;
    GValue value = {0, };

    gtk_tree_model_get_iter (win->words_model, &iter, path);
    gtk_tree_model_get (win->words_model, &iter, CHECK_COLUMN, &fixed, -1);

    fixed = !fixed;
    /**/
    gtk_tree_model_get_value(win->words_model, &iter,
			     WORDS_STRUCT, &value);
    w = g_value_get_pointer(&value);
    w->checked = fixed;
    g_value_unset(&value);
    /* set to model */
    gtk_list_store_set (GTK_LIST_STORE (win->words_model), &iter,
			CHECK_COLUMN, fixed, -1);
    gtk_tree_path_free (path);

}

static void
add_columns(struct app_win *win, GtkTreeView *treeview)
{
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    /**/
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("word",
						      renderer,
						      "text", WORD_COLUMN_UTF8,
						      NULL);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 100);
    gtk_tree_view_column_set_sort_column_id(column, WORD_COLUMN_UTF8);
    gtk_tree_view_append_column (treeview, column);
    /**/
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("yomi",
						      renderer,
						      "text", YOMI_COLUMN_UTF8,
						      NULL);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 100);
    gtk_tree_view_column_set_sort_column_id(column, YOMI_COLUMN_UTF8);
    gtk_tree_view_append_column (treeview, column);
    /**/
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("wtype",
						      renderer,
						      "text", WTYPE_COLUMN,
						      NULL);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 50);
    gtk_tree_view_column_set_sort_column_id(column, WTYPE_COLUMN);
    gtk_tree_view_append_column (treeview, column);
    /**/
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("freq",
						      renderer,
						      "text", FREQ_COLUMN,
						      NULL);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 50);
    gtk_tree_view_column_set_sort_column_id(column, FREQ_COLUMN);
    gtk_tree_view_append_column (treeview, column);
    /**/
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes ("fix!",
						       renderer,
						       "active",
						       FIX_COLUMN,
						       NULL);
    g_signal_connect(renderer, "toggled",
		     G_CALLBACK(fix_toggled), win);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 50);
    gtk_tree_view_column_set_sort_column_id(column, CHECK_COLUMN);
    gtk_tree_view_append_column (treeview, column);
    /**/
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes ("Checked?",
						       renderer,
						       "active",
						       CHECK_COLUMN,
						       NULL);
    g_signal_connect(renderer, "toggled",
		     G_CALLBACK(check_toggled), win);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 100);
    gtk_tree_view_column_set_sort_column_id(column, CHECK_COLUMN);
    gtk_tree_view_append_column (treeview, column);
}

static void
row_activate_cb(GtkTreeView *tree_view,
		GtkTreePath *path,
		GtkTreeViewColumn *column)
{
    printf("row activate\n");
}

static void
set_convert_result(struct app_win *win, struct words *w,
		   const char *index_str,
		   const char *word_str, const char *wt)
{
    char buf[1024];
    char buf2[1024];
    struct anthy_segment_stat ss;
    int i;
    char *tail = syusi_tail(wt);
    sprintf(buf, "%s%s", index_str, tail);
    printf("%s\n", buf);
    anthy_set_string(win->ac, buf);
    anthy_get_segment_stat(win->ac, 0, &ss);
    /**/
    sprintf(buf2, "%s%s", word_str, tail);
    /**/
    gtk_list_store_clear(GTK_LIST_STORE(win->res_model));
    for (i = 0; i < ss.nr_candidate; i++) {
	GtkTreeIter iter;
	anthy_get_segment(win->ac, 0, i, buf, 1024);
	free(win->cands[i]);
	win->cands[i] = euc_to_utf8(buf);
	gtk_list_store_append(GTK_LIST_STORE(win->res_model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(win->res_model), &iter,
			   0, win->cands[i],
			   -1);
	/**/
	if (i == 0) {
	    if (!strcmp(buf2, buf)) {
		w->is_top = 1;
		printf("top\n");
	    } else {
		w->is_top = 0;
		printf("not top\n");
	    }
	}
    }
}

static void
selection_cb(GtkTreeSelection *selection,
	     struct app_win *win)
{
    GtkTreeIter iter;
    GValue value = {0, };
    struct words *w;
    const char *s;
    if (!gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	return ;
    }

    gtk_tree_model_get_value(win->words_model, &iter,
			     WORDS_STRUCT, &value);
    w = g_value_get_pointer(&value);
    g_value_unset(&value);
    if (w) {
	set_convert_result(win, w, w->index, w->word, w->wt);
    }
}

static void
build_tree_view(struct app_win *win, GtkWidget *hbox)
{
    GtkWidget *sw;
    GtkTreeSelection *selection;

    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
					 GTK_SHADOW_ETCHED_IN);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				    GTK_POLICY_NEVER,
				    GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (hbox), sw, TRUE, TRUE, 0);

    win->words_model = create_words_model();
    win->words_view = gtk_tree_view_new_with_model(win->words_model);
    add_columns(win, GTK_TREE_VIEW(win->words_view));
    gtk_container_add(GTK_CONTAINER(sw), win->words_view);
    gtk_widget_show(win->words_view);

    gtk_widget_show(sw);

    g_signal_connect(win->words_view, "row_activated",
		     G_CALLBACK(row_activate_cb), NULL);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(win->words_view));
    g_signal_connect(selection, "changed",
		     G_CALLBACK(selection_cb), win);
}

static void
build_text_entry(struct app_win *win, GtkWidget *vbox)
{
    /**/
    win->text_entry = gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (vbox), win->text_entry, FALSE, TRUE, 0);
    gtk_widget_show(win->text_entry);
    /**/
}

static void
build_result_view(struct app_win *win)
{
    GtkWidget *sw;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    GtkWidget *label = gtk_label_new("result");
    win->res_model = GTK_TREE_MODEL(gtk_list_store_new(1, G_TYPE_STRING));
    win->res_view = gtk_tree_view_new_with_model(win->res_model);
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("word",
						      renderer,
						      "text", 0,
						      NULL);
    gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				     GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 100);
    gtk_tree_view_append_column (GTK_TREE_VIEW(win->res_view), column);
    gtk_widget_show(label);
    gtk_widget_show(win->res_view);

    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
					 GTK_SHADOW_ETCHED_IN);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				    GTK_POLICY_NEVER,
				    GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(sw), win->res_view);
    gtk_widget_show(sw);
    gtk_notebook_append_page(GTK_NOTEBOOK(win->note_book),
			     sw, label);
}

static void
build_note_book(struct app_win *win, GtkWidget *vbox)
{
    win->note_book = gtk_notebook_new();
    gtk_box_pack_start (GTK_BOX (vbox), win->note_book, TRUE, TRUE, 0);
    /**/
    build_result_view(win);
    /**/
    gtk_widget_show(win->note_book);
}

static void
build_ui(struct app_win *win)
{
    GtkWidget *hbox, *vbox;
    win->top = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    hbox = gtk_hbox_new(FALSE, 0);
    /**/
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
    build_text_entry(win, vbox);
    build_tree_view(win, vbox);
    build_note_book(win, vbox);
    /* right pane */
    gtk_container_add(GTK_CONTAINER(win->top), hbox);
    build_check_buttons(win, hbox);
    /**/
    gtk_widget_show(vbox);
    gtk_widget_show(hbox);

    gtk_window_set_default_size (GTK_WINDOW (win->top), 500, 600);
    gtk_widget_show(win->top);
}

static void
load_checked(struct app_win *win)
{
    FILE *fp;
    char buf[1024];
    int nr = 0;
    fp = fopen(CHECKED_ITEM_FILE, "r");
    if (!fp) {
	printf("no checked\n");
	return ;
    }
    while (fgets(buf, 1024, fp)) {
	char index[1024];
	char word[1024];
	char wt[1024];
	char fix[1024];
	struct words *w;
	if (sscanf(buf, "%s %s %s %s", index, word, wt, fix) != 4) {
	    continue;
	}
	w = find_word(&win->dict, index, word, wt);
	if (w) {
	    nr++;
	    w->checked = 1;
	}
    }
    fclose(fp);
    printf("%d checked\n", nr);
}

static void
init_anthy(struct app_win *win)
{
    int i;
    anthy_init();
    anthy_set_personality("");
    win->ac = anthy_create_context();
    for (i = 0; i < MAX_CANDIDATES; i++) {
	win->cands[i] = NULL;
    }
}

static void
start_cb(GtkWidget *button, struct app_win *win)
{
    build_ui(win);
    load_words(win);
    load_checked(win);
    update_model(win);
    gtk_widget_hide(win->conf_win);
}

static void
build_conf_win(struct app_win *win)
{
    GtkWidget *vbox;
    GtkWidget *button;
    win->conf_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(win->conf_win), vbox);
    button = gtk_button_new_with_label("start");
    gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    gtk_widget_show(vbox);
    gtk_widget_show(win->conf_win);
    g_signal_connect(button, "clicked",
		     G_CALLBACK(start_cb), win);
}

int
main(int argc, char **argv)
{
    struct app_win win;
    int i;
    init_anthy(&win);
    gtk_init(&argc, &argv);
    win.argc = argc;
    win.argv = argv;
    win.words.next = NULL;
    /**/
    for (i = 0; i < NR_BUCKETS; i++) {
	win.dict.bucket[i] = NULL;
    }
    /**/
    build_conf_win(&win);
    gtk_main();
    return 0;
}
