#include <zlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include "reader.h"
#include "storage.h"
#include "word.h"
#include "misc.h"

static void
do_puts(FILE *fp, gzFile zfp, char *buf)
{
  if (fp) {
    fputs(buf, fp);
  } else {
    gzputs(zfp, buf);
  }
}

void
WordSet::add_word(const char *name, Stamp stamp,
		  const char *attr, bool approve,
		  const char *idx, const char *word)
{
  Word *w = find_word(idx, word);
  Attr *a = w->find_attr(attr);
  if (!a) {
    a = new Attr(w, attr);
    w->m_attrs.push_back(a);
  }
  Sign *s;
  s = a->add_sign(name, stamp, approve);
  if (s) {
    m_sign_array.push_back(s);
  }
}

Word *
WordSet::find_word(const char *idx, const char *word)
{
  int hash = word_hash(idx, word) % NR_PAGES;
  //
  std::list <Word *>::iterator it;
  for (it = m_word_hash_array[hash].begin();
       it != m_word_hash_array[hash].end();
       it++) {
    Word *w = *it;
    if (!strcmp(idx, w->m_index) &&
	!strcmp(word, w->m_word)) {
      return w;
    }
  }
  // create new
  Word *w = new Word();
  w->m_index = strdup(idx);
  w->m_word = strdup(word);
  m_word_array.push_back(w);
  m_word_hash_array[hash].push_back(w);
  return w;
}

void
WordSet::write_words(FILE *fp, gzFile zfp)
{
  std::list<Sign *>::iterator it;
  for (it = m_sign_array.begin(); it != m_sign_array.end(); it++) {
    Sign *s = *it;
    Attr *a = s->m_owner;
    Word *w = a->m_owner;
    char stamp_buf[10];
    s->m_stamp.get_str(stamp_buf);
    char buf[1000];
    sprintf(buf, "%s %s %s %s %s %s\n",
	    w->m_index, w->m_word,
	    stamp_buf, s->m_user, a->m_attr,
	    (s->m_is_ok ? "ok" : "ng"));
    do_puts(fp, zfp, buf);
  }
}

Page::Page(const char *name, int idx)
{
  char buf[strlen(name) + 20];
  sprintf(buf, "%s/words/%.2x", name, idx);
  m_file_name = strdup(buf);
  m_idx = idx;
}

void
Page::read()
{
  gzFile fp = gzopen(m_file_name, "r");
  if (!fp) {
    return ;
  }
  char buf[1024];
  while (do_gets(fp, buf, 1024)) {
    DictLine dl(buf);
    if (dl.nr != 6) {
      continue;
    }
    Word *w = find_word(dl.index, dl.word);
    if (!w) {
      continue;
    }
    Attr *a = w->find_attr(dl.attr);
    if (!a) {
      a = new Attr(w, dl.attr);
      w->m_attrs.push_back(a);
    }

    Sign *sign;
    sign = a->add_sign(dl.user, dl.stamp, dl.approve);
    if (sign) {
      m_sign_array.push_back(sign);
    }
  }
  gzclose(fp);
}

void
Page::write()
{
  gzFile zfp = gzopen(m_file_name, "wb");
  if (zfp) {
    write_words(NULL, zfp);
    gzclose(zfp);
  }
}

void
Page::dump()
{
  write_words(stdout, NULL);
}

void
Page::print_index(FILE *fp)
{
  std::list<Sign *>::iterator it;
  Stamp newest(0);
  for (it = m_sign_array.begin(); it != m_sign_array.end(); it++) {
    if ((*it)->m_stamp.get_stamp() > newest.get_stamp()) {
      newest = (*it)->m_stamp;
    }
  }
  char buf[10];
  newest.get_str(buf);

  fprintf(fp,"%d %d %s",
	  m_word_array.size(),
	  m_sign_array.size(),
	  buf);
}

Storage::Storage(const char *name)
{
  m_dir_name = strdup(name);
  int i;
  for (i = 0; i < NR_PAGES; i++) {
    m_pages[i] = NULL;
  }
}

Storage::~Storage()
{
  free(m_dir_name);
}

void
Storage::check()
{
  mkdir(m_dir_name, 0777);
  char buf[strlen(m_dir_name) + 20];
  sprintf(buf, "%s/words", m_dir_name);
  mkdir(buf, 0777);
}

void
Storage::update()
{
  int i;
  for (i = 0; i < NR_PAGES; i++) {
    if (m_pages[i]) {
      m_pages[i]->write();
    }
  }
}

void
Storage::dump()
{
  int i;
  for (i = 0; i < NR_PAGES; i++) {
    Page *p = get_page(i);
    p->dump();
  }
}

void
Storage::update_index()
{
  char buf[strlen(m_dir_name) + 20];
  sprintf(buf, "%s/index", m_dir_name);
  FILE *fp = fopen(buf, "w");
  if (!fp) {
    return ;
  }
  //
  int i;
  for (i = 0; i < NR_PAGES; i++) {
    Page *p = get_page(i);
    fprintf(fp, "%.2x ", i);
    p->print_index(fp);
    fprintf(fp, "\n");
  }
  fclose(fp);
}

void
Storage::add_word(const char *name, Stamp stamp,
		  const char *attr, bool approve,
		  const char *index, const char *word)
{
  int hash = word_hash(index, word);
  int idx = hash % NR_PAGES;
  Page *p = get_page(idx);
  p->add_word(name, stamp, attr, approve, index, word);
}

Page *
Storage::get_page(int idx)
{
  if (!m_pages[idx]) {
    Page *p = new Page(m_dir_name, idx);
    p->read();
    m_pages[idx] = p;
  }
  return m_pages[idx];
}
