;; SKK is a Japanese input method
;;
;; SKKϤϲξ֤ǹ
;;  ľ direct
;;   kanji
;;  Ѵ converting
;;  ꤬ okuri
;;  ѿ latin
;;  ѱѿ wide-latin
;;
;; parent/child context is not used now
;;
(require "japanese.scm")
(require "generic-key.scm")


(define skk-dic-file-name "/usr/share/skk/SKK-JISYO.L")
(define skk-personal-dic-filename
  (string-append (getenv "HOME") "/.skk-jisyo"))
(define skk-dic-init nil)
;; configs
(define skk-use-candidate-window? #t)
(define skk-candidate-op-count 0)
(define skk-use-recursive-learning? #f)
;; key
(define skk-latin-key
  (lambda (key key-state)
    (or
     (generic-off-key key key-state)
     (= (to-lower-char key) 108))))
(define skk-wide-latin-key
  (lambda (key key-state)
    (and (= (to-lower-char key) 108)
	 (shift-key-mask key-state))))
(define skk-begin-conv-key
  (lambda (key key-state)
    (= key 32)))
(define skk-on-key
  (lambda (key key-state)
    (or
     (and (= (to-lower-char key) 106)
	  (control-key-mask key-state))
     (generic-on-key key key-state))))
(define skk-commit-key
  (lambda (key key-state)
    (generic-commit-key key key-state)))
(define skk-next-candidate
  (lambda (key key-state)
    (generic-next-candidate-key key key-state)))
(define skk-prev-candidate
  (lambda (key key-state)
    (or
     (generic-prev-candidate-key key key-state)
     (= (to-lower-char key) 120))))

(define skk-kana-toggle
  (lambda (key key-state)
    (= (to-lower-char key) 113)))
(define skk-cancel-key
  (lambda (key key-state)
    (generic-cancel-key key key-state)))

(define skk-backspace-key
  (lambda (key key-state)
    (generic-backspace-key key key-state)))

;; access
(define skk-context-parent-context
  (lambda (c)
    (car (nthcdr 11 c))))

(define skk-context-set-parent-context!
  (lambda (c cnt)
    (set-car! (nthcdr 11 c) cnt)))

(define skk-context-child-context
  (lambda (c)
    (car (nthcdr 10 c))))

(define skk-context-set-child-context!
  (lambda (c cnt)
    (set-car! (nthcdr 10 c) cnt)))

(define skk-context-candidate-window
  (lambda (c)
    (car (nthcdr 9 c))))

(define skk-context-set-candidate-window!
  (lambda (c cnt)
    (set-car! (nthcdr 9 c) cnt)))

(define skk-context-candidate-op-count
  (lambda (c)
    (car (nthcdr 8 c))))

(define skk-context-set-candidate-op-count!
  (lambda (c cnt)
    (set-car! (nthcdr 8 c) cnt)))

(define skk-context-rk-context
  (lambda (c)
    (car (nthcdr 7 c))))

(define skk-context-set-rk-context!
  (lambda (c rkc)
    (set-car! (nthcdr 7 c) rkc)))

(define skk-context-nth
  (lambda (c)
    (car (nthcdr 6 c))))

(define skk-context-set-nth!
  (lambda (c nth)
    (set-car! (nthcdr 6 c) nth)))

(define skk-context-okuri
  (lambda (c)
    (car (nthcdr 4 c))))

(define skk-context-set-okuri!
  (lambda (c tail)
    (set-car! (nthcdr 4 c) tail)))

(define skk-context-okuri-head
  (lambda (c)
    (car (nthcdr 3 c))))

(define skk-context-set-okuri-head!
  (lambda (c okuri-head)
    (set-car! (nthcdr 3 c) okuri-head)))

(define skk-context-set-head!
  (lambda (c head)
    (set-car! (nthcdr 2 c) head)))

(define skk-context-head
  (lambda (c)
    (car (nthcdr 2 c))))

(define skk-context-kana-mode
  (lambda (sc)
    (car (nthcdr 1 sc))))

(define skk-context-set-kana-mode!
  (lambda (sc mode)
    (set-car! (nthcdr 1 sc) mode)))

(define skk-context-state
  (lambda (c)
    (car (nthcdr 0 c))))

(define skk-context-set-state!
  (lambda (c s)
    (set-car! (nthcdr 0 c) s)))
;; state kana head okuri tail candidates nth rk


(define skk-find-root-context
  (lambda (sc)
    (let ((pc (skk-context-parent-context sc)))
      (if pc
	  (skk-find-root-context pc)
	  sc))))

(define skk-find-descendant-context
  (lambda (sc)
    (let ((csc (skk-context-child-context sc)))
      (if csc
	  (skk-find-descendant-context csc)
	  sc))))

(define skk-current-context
  (lambda (c)
    (skk-find-descendant-context (context-data c))))

(define skk-flush
  (lambda (sc)
    (rk-flush (skk-context-rk-context sc))
    (skk-context-set-state! sc 'skk-state-direct)
    (skk-context-set-head! sc '())
    (skk-context-set-okuri-head! sc "")
    (skk-context-set-okuri! sc '())
    (skk-context-set-candidate-op-count! sc 0)
    (skk-context-set-candidate-window! sc #f)
    (skk-context-set-child-context! sc ())
    (skk-context-set-parent-context! sc ())))

(define skk-context-new
  (lambda ()
    (if (null? skk-dic-init)
	(begin
	  (set! skk-dic-init #t)
	  (if skk-use-recursive-learning?
	   (require "skk-editor.scm"))
	  (skk-lib-dic-open skk-dic-file-name)
	  (skk-lib-read-personal-dictionary skk-personal-dic-filename)))
    (let ((c 
	   (copy-list 
	    '(skk-state-latin #t "" "" "" () () () () () () () ()))))
      (skk-context-set-head! c ())
      (skk-context-set-rk-context! c
				   (rk-context-new ja-rk-rule #t #f))
      (skk-flush c)
      (skk-context-set-state! c 'skk-state-latin)
      c)))

(define skk-make-string
  (lambda (sl kana)
    (if sl
	(string-append (skk-make-string (cdr sl) kana)
		       (if kana
			   (caar sl)
			   (cdar sl)))
	"")))

(define skk-context-kana-toggle
  (lambda (sc)
    (let ((s (skk-context-kana-mode sc)))
      (set! s (not s))
      (skk-context-set-kana-mode! sc s))))

(define skk-get-string-by-mode
  (lambda (sc res)
    (if res
	(if (skk-context-kana-mode sc)
	    (car res)
	    (cdr res))
	nil)))

(define skk-get-nth-candidate
  (lambda (sc n)
    (skk-lib-get-nth-candidate
     n
     (skk-make-string (skk-context-head sc)
		      #t)
     (skk-context-okuri-head sc)
     (skk-make-string
      (skk-context-okuri sc) #t))))

(define skk-get-current-candidate
  (lambda (sc)
    (skk-get-nth-candidate
     sc
     (skk-context-nth sc))))

(define skk-commit-raw
  (lambda (sc id key key-state)
    (if (not (skk-context-parent-context sc))
	(im-commit-raw id))))

(define skk-commit
  (lambda (sc id str)
    (im-commit id str)))

(define skk-begin-conversion
  (lambda (sc id)
    (let ((res))
       ;; get residual 'n'
       (if (= (skk-context-state sc) 'skk-state-kanji)
 	  (let ((tmp (rk-push-key-last! (skk-context-rk-context sc))))
 	    (if tmp
 		(skk-context-set-head! sc
 				       (cons tmp (skk-context-head sc))))))
       ;;
      (set! res
	    (skk-lib-get-entry
	     (skk-make-string (skk-context-head sc) #t)
	     (skk-context-okuri-head sc)
	     (skk-make-string
	      (skk-context-okuri sc) #t)))
      (if res
	  (begin
	    (skk-context-set-nth! sc 0)
	    (skk-context-set-state!
	     sc 'skk-state-converting))
	  (skk-flush sc))
      ())))

(define skk-do-update-preedit
  (lambda (id sc)
    (let ((rkc (skk-context-rk-context sc))
	  (stat (skk-context-state sc))
	  (csc (skk-context-child-context sc)))
      (if (= stat 'skk-state-kanji)
	  (im-pushback-preedit id preedit-reverse ""))
      (if (or
	   (= stat 'skk-state-okuri)
	   (= stat 'skk-state-converting))
	  (im-pushback-preedit id preedit-reverse ""))
      (if (or
	   (= stat 'skk-state-kanji)
	   (= stat 'skk-state-okuri))
	  (let ((h (skk-make-string 
		    (skk-context-head sc)
		    (skk-context-kana-mode sc))))
	    (if (string? h)
		(im-pushback-preedit
		 id preedit-reverse
		 h))))
      (if (= stat 'skk-state-converting)
	  (begin
	    (im-pushback-preedit
	     id preedit-reverse
	     (skk-get-current-candidate sc))))
      (if (= stat 'skk-state-okuri)
          (im-pushback-preedit id preedit-reverse "*"))
      (im-pushback-preedit
       id preedit-reverse
       (skk-make-string (skk-context-okuri sc)
			(skk-context-kana-mode sc)))
      (im-pushback-preedit id preedit-reverse
                           (rk-pending rkc))
      ;; child context's preedit
      (if csc
	  (begin
	    (im-pushback-preedit id preedit-reverse "[")
	    (skk-do-update-preedit id csc)
	    (im-pushback-preedit id preedit-reverse "]")
	    )))))

(define skk-update-mode
  (lambda (id sc)
    (let ((stat (skk-context-state sc))
	  (mode))
      (set! mode
	    (if (skk-context-kana-mode sc)
		1
		2))
      (if (= stat 'skk-state-latin)
	  (set! mode 0))
      (if (= stat 'skk-state-wide-latin)
	  (set! mode 3))
      (im-update-mode id mode))))

(define skk-update-preedit
  (lambda (id sc)
    (im-clear-preedit id)
    (skk-do-update-preedit id (skk-find-root-context sc))
    (im-update-preedit id)))

(define skk-proc-state-direct-no-preedit
  (lambda (key key-state id sc rkc)
    (if (skk-wide-latin-key key key-state)
	(begin
	  (skk-context-set-state! sc 'skk-state-wide-latin)
	  (rk-flush rkc)
	  (skk-update-mode id sc)
	  #f)
	#t)
    (if (skk-latin-key key key-state)
	(begin
	  (skk-context-set-state! sc 'skk-state-latin)
	  (rk-flush rkc)
	  (skk-update-mode id sc)
	  #f)
	#t)
    (if (skk-kana-toggle key key-state)
	(begin 
	  (skk-context-kana-toggle sc)
	  (skk-update-mode id sc)
	  #f)
	#t)
    #t))

(define skk-proc-state-direct
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (id (context-id c))
	   (key-str (charcode->string (to-lower-char key)))
	   (rkc (skk-context-rk-context sc))
	   (res))
      (set! res nil)
      (and
       (if (string-equal? (rk-pending rkc) "")
	   (skk-proc-state-direct-no-preedit key key-state id sc rkc)
	   #t)
       (if (skk-cancel-key key key-state)
	   (begin
	     (skk-flush sc)
	     #f)
	   #t)
       (if (skk-backspace-key key key-state)
	   (if (not (rk-backspace rkc))
	       (begin
		 (skk-commit-raw sc id key key-state)
		 #f)
	       #f)
	   #t)
       (if (or
	    (control-key-mask key-state)
	    (= key 32))
	   (begin
	     (skk-commit-raw sc id key key-state)
	     #f)
	   #t)
       (if (and
	    (shift-key-mask key-state)
	    (alphabet-char? key))
	   (begin
	     (skk-context-set-state! sc 'skk-state-kanji)
	     (skk-update-mode id sc)
	     (set! key (to-lower-char key))
	     (set! key-str (charcode->string key))
	     #t)
	   #t)
       (if (and
	    (not (skk-context-head sc))
	    (not (string-find (rk-expect rkc) key-str))
	    (not (rk-pending rkc)))
	   (begin
	     (skk-commit-raw sc id key key-state)
	     (skk-flush sc)
	     (skk-update-preedit id sc)
	     #f)
	   #t)
       (if (symbol? key)
	   (begin
	     (skk-flush sc)
	     (skk-commit-raw sc (context-id c) key key-state)
	     #f)
	   #t)
       (begin
	 (set! res
	       (rk-push-key!
		rkc
		key-str))
	 #t))
      ;; update state
      (if (= (skk-context-state sc) 'skk-state-kanji)
	  (if res
	      (skk-context-set-head! sc (cons res ()))
	      (set! res ())))
      (if (= (skk-context-state sc) 'skk-state-direct)
	  (skk-get-string-by-mode sc res)
	  nil))))

(define skk-proc-state-kanji
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (id (context-id c))
	   (rkc (skk-context-rk-context sc))
	   (stat (skk-context-state sc))
	   (res))
      (and
       (if (skk-begin-conv-key key key-state)
	   (begin
	     (skk-begin-conversion sc id)
	     #f)
	   #t)
       (if (skk-cancel-key key key-state)
	   (begin
	     (skk-flush sc)
	     #f)
	   #t)
       (if (skk-backspace-key key key-state)
	   (begin
	     (if (not (rk-backspace rkc))
		 (if (> (length (skk-context-head sc)) 0)
		     (skk-context-set-head!
		      sc (cdr (skk-context-head sc)))
		     (begin
		       (skk-context-set-state! sc 'skk-state-direct)
		       (skk-flush sc))))
	     #f)
	   #t)
       (if (skk-commit-key key key-state)
	   (begin
	     (skk-commit sc id (skk-make-string
				(skk-context-head sc)
				(skk-context-kana-mode sc)))
	     (skk-flush sc)
	     (skk-context-set-state! sc 'skk-state-direct)
	     (skk-update-mode id sc)
	     #f)
	   #t)
       (if (and (shift-key-mask key-state)
		(skk-context-head sc))
	   (begin
	     (skk-context-set-state! sc 'skk-state-okuri)
	     (skk-update-mode id sc)
	     (set! key (to-lower-char key))	     
	     (skk-context-set-okuri-head! sc
					  (charcode->string key))
	     #t)
	   #t)
       (begin
	 (set! key (to-lower-char key))	     
	 (set! stat (skk-context-state sc))
	 (set! res
	       (rk-push-key!
		rkc
		(charcode->string key)))
	 (if (and res (= stat 'skk-state-kanji))
	     (begin
	       (skk-context-set-head! sc
				      (cons
				       res
				       (skk-context-head sc)))))
	 (if (and res (= stat 'skk-state-okuri))
	     (begin
	       (skk-context-set-okuri! sc
				      (cons res ()))
	       (skk-begin-conversion sc id)))))
      nil)))

(define skk-setup-child-context
  (lambda (sc)
    (let ((csc (skk-context-new)))
      (skk-context-set-child-context! sc csc)
      (skk-context-set-parent-context! csc sc)
      (skk-context-set-state! csc 'skk-state-direct))))

(define skk-check-candidate-window-begin
  (lambda (sc id)
    (if
     (and
      (not
       (skk-context-candidate-window sc))
      skk-use-candidate-window?
      (> (skk-context-candidate-op-count sc)
	 skk-candidate-op-count))
     (begin
       (skk-context-set-candidate-window! sc #t)
       (im-begin-candidate
	id
	(skk-lib-get-nr-candidates
	 (skk-make-string (skk-context-head sc)
			  #t)
	 (skk-context-okuri-head sc)
	 (skk-make-string
	  (skk-context-okuri sc) #t))
	0)))))

(define skk-change-candidate-index
  (lambda (sc id incr)
    (if incr
	(begin
	  (skk-context-set-nth! sc
				(+ 1 (skk-context-nth sc)))
	  (skk-context-set-candidate-op-count!
	   sc
	   (+ 1 (skk-context-candidate-op-count sc))))
	(begin
	  (if (> (skk-context-nth sc) 0)
	      (skk-context-set-nth! sc (- (skk-context-nth sc) 1))
	      (skk-context-set-nth! sc (- (skk-lib-get-nr-candidates
					   (skk-make-string
					    (skk-context-head sc) #t)
					   (skk-context-okuri-head sc)
					   (skk-make-string
					    (skk-context-okuri sc) #t))
					  1)))))
    (if (not (skk-get-current-candidate sc))
	(begin
	  (skk-context-set-nth! sc 0)
	  (if skk-use-recursive-learning?
	      (skk-setup-child-context sc))))
    (if (not (skk-context-child-context sc))
	(begin
	  ;; Windowɽ򳫻Ϥ뤫
	  (skk-check-candidate-window-begin sc id)
	  ;;
	  (if (skk-context-candidate-window sc)
	      (im-update-candidate id (skk-context-nth sc)))))
    #f))


(define skk-proc-state-converting
  (lambda (c key key-state)
    (let ((sc (skk-current-context c))
	  (id (context-id c))
	  (res ()))
      (and
       (if (skk-next-candidate key key-state)
	   (skk-change-candidate-index sc id #t)
	   #t)
       (if (skk-prev-candidate key key-state)
	   (skk-change-candidate-index sc id #f)
	   #t)
       (if (skk-cancel-key key key-state)
	   (begin
	     ;; back to kanji mode
	     ()
	     (if (skk-context-candidate-window sc)
		 (im-end-candidate id))
	     (skk-context-set-state! sc 'skk-state-kanji)
	     (skk-context-set-okuri-head! sc "")
	     (if (car (skk-context-okuri sc))
		 (skk-context-set-head! sc
					(cons (car (skk-context-okuri sc))
					      (skk-context-head sc))))
	     (skk-context-set-okuri! sc ())
	     #f)
	   #t)
       (if (skk-commit-key key key-state)
	   (begin
	     (set! res (skk-get-current-candidate sc))
	     (set!
	      res
	      (string-append res
			     (skk-make-string (skk-context-okuri sc)
					      (skk-context-kana-mode sc))))
	     (skk-lib-commit-candidate
	      (skk-make-string (skk-context-head sc) #t)
	      (skk-context-okuri-head sc)
	      (skk-make-string
	       (skk-context-okuri sc) #t)
	      (skk-context-nth sc))
	     (if (skk-context-candidate-window sc)
		 (im-end-candidate id))
	     (skk-flush sc)
	     (skk-context-set-state! sc 'skk-state-direct)
	     (skk-update-mode id sc)
	     #f)
	   #t)
       (begin
	 (skk-context-set-state! sc 'skk-state-direct)
	 (skk-update-mode id sc)
	 (set! res (skk-get-current-candidate sc))
	 (if (skk-context-candidate-window sc)
	     (im-end-candidate id))
	 (set!
	  res
	  (string-append res 
			 (skk-make-string
			  (skk-context-okuri sc)
			  (skk-context-kana-mode sc))))
	 (skk-flush sc)
	 (let ((res2 (skk-proc-state-direct c key key-state)))
	   (set!
	    res
	    (string-append res 
			   (skk-make-string
			    (skk-context-okuri sc)
			    (skk-context-kana-mode sc))))
	   (if (string? res2)
	       (set! res
		     (string-append res res2))))))
      res)))

(define skk-proc-state-okuri
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (rkc (skk-context-rk-context sc))
	   (id (context-id c))	   
	   (res))
      (and
       (if (skk-cancel-key key key-state)
	   (begin
	     (rk-flush rkc)
	     (skk-context-set-state! sc 'skk-state-kanji)
	     #f)
	   #t)
       (if (skk-backspace-key key key-state)
	   (begin
	     (if (not (rk-backspace rkc))
		 (begin
		   (print (cdr (skk-context-okuri sc)))
		   (if (cdr (skk-context-okuri sc))
		       (skk-context-set-okuri! sc
			(cdr (skk-context-okuri sc)))
		       (begin
			 (skk-context-set-okuri! sc '())
		       (skk-context-set-state! sc 'skk-state-kanji)))))
	     #f)
	   #t)
       (begin
	 (set! res
	       (rk-push-key!
		rkc
		(charcode->string (to-lower-char key))))
	 (if res
	     (begin
	       (skk-context-set-okuri!
		sc
		(cons res (skk-context-okuri sc)))
	       (if (string-equal? (rk-pending rkc) "")
		   (skk-begin-conversion sc id))))))
      ())))

(define skk-proc-state-latin
  (lambda (c key key-state)
    (let ((sc (skk-current-context c))
	  (id (context-id c)))
      (if
       (skk-on-key key key-state)
       (begin
	 (skk-context-set-state! sc 'skk-state-direct)
	 (skk-update-mode (context-id c) sc))
       (skk-commit-raw sc (context-id c) key key-state))
      ())))

(define skk-proc-state-wide-latin
  (lambda (c key key-state)
    (let* ((w (jp-wide (charcode->string key)))
	   (id (context-id c))
	   (sc (skk-current-context c)))
      (if (skk-on-key key key-state)
	  (begin
	    (skk-flush sc)
	    (skk-update-mode id sc))
	  (if w
	      (skk-commit sc id w)
	      (skk-commit-raw sc id key key-state)))
      ())))

(define skk-push-key
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (state (skk-context-state sc))
	   (fun)
	   (res))
      (if (= state 'skk-state-direct)
	  (set! fun skk-proc-state-direct))
      (if (= state 'skk-state-kanji)
	  (set! fun skk-proc-state-kanji))
      (if (= state 'skk-state-converting)
	  (set! fun skk-proc-state-converting))
      (if (= state 'skk-state-okuri)
	  (set! fun skk-proc-state-okuri))
      (if (= state 'skk-state-latin)
	  (set! fun skk-proc-state-latin))
      (if (= state 'skk-state-wide-latin)
	  (set! fun skk-proc-state-wide-latin))
      (set! res (fun c key key-state))
      (if res
	  (skk-commit sc (context-id c) res))
      (skk-update-preedit
       (context-id c) sc)
      )))

(define skk-init-handler
  (lambda (id arg)
    (let* ((c (find-context id)))
      (set-context-data! c
			 (skk-context-new))
      (im-clear-mode-list id)
      (im-pushback-mode-list id "RAW")
      (im-pushback-mode-list id "Ҥ餬")
      (im-pushback-mode-list id "")
      (im-pushback-mode-list id "ѱѿ")
      (im-update-mode-list id)
      (im-update-mode id 0))))

(define skk-press-key-handler
  (lambda (id key state)
    (let ((c (find-context id)))
      (skk-push-key c key state))))

(define skk-release-key-handler
  (lambda (id key state)
    ()))

(define skk-reset-handler
  (lambda (id)
    ()))

(define skk-mode-handler
  (lambda (id mode)
    (let* ((c (find-context id))
	   (sc (context-data c)))
      (skk-flush sc)
      (if (= mode 0)
	  (skk-context-set-state! sc 'skk-state-latin))
      (if (= mode 1)
	  (begin
	    (skk-context-set-state! sc 'skk-state-direct)
	    (skk-context-set-kana-mode! sc #t)))
      (if (= mode 2)
	  (begin
	    (skk-context-set-state! sc 'skk-state-direct)
	    (skk-context-set-kana-mode! sc #f)))
      (if (= mode 3)
	  (skk-context-set-state! sc 'skk-state-wide-latin))
      (skk-update-preedit id sc)
      ())))

(define skk-get-candidate-handler
  (lambda (id idx)
    (let* ((c (find-context id))
	   (sc (context-data c))
	   (cand (skk-get-nth-candidate sc idx))
	   (okuri (skk-context-okuri sc)))
      (if okuri
	  (string-append cand
			 (skk-make-string okuri t))
	  cand))))


(define skk-set-candidate-index-handler
  (lambda (id idx)
    (let* ((c (find-context id))
	   (sc (context-data c)))
      (skk-context-set-nth! sc idx)
      (skk-update-preedit  id sc))))

(register-im
 'skk
 "ja"
 "EUC-JP"
 nil
 skk-init-handler
 nil
 skk-mode-handler
 skk-press-key-handler
 skk-release-key-handler
 skk-reset-handler
 skk-get-candidate-handler
 skk-set-candidate-index-handler
 nil)
