﻿// project created on 2007/09/08 at 20:20
using System;
using System.IO;
using System.Collections.Generic;
using NaGet.Net;
using NaGet.Packages;
using NaGet.Packages.Install;
using NaGet.SubCommands;

namespace AllGet
{

	class DownloadListener
	{
		private string line = null;
		
		/// <summary>
		/// lineのコンソール上における長さ
		/// </summary>
		private int lineWidth = 0;
	
		public void OnDownload(object sender, DownloadEventArgs a) {
			if (line != null) {
				lock (line) {
					int newpos = (lineWidth <= 0)? 0 : Console.CursorLeft - lineWidth;
					if (newpos < 0) newpos = 0;
					
					// 空白で埋める
					Console.CursorLeft = newpos;
					for (int i = 0; i < lineWidth && Console.CursorLeft < Console.BufferWidth; i++) {
						Console.Write(' ');
					}
					Console.CursorLeft = newpos;
					
					line = null; lineWidth = 0;
				}
			}
			
			switch (a.Type) {
			case DownloadEventType.INITED:
				line = "starting...";
				break;
			case DownloadEventType.CONNECTED:
			case DownloadEventType.DOWNLOADING:
				line = a.TaskMessage;
				break;
			case DownloadEventType.COMPLETED:
				//
				line = null;
				break;
			case DownloadEventType.ERROR:
				Console.Write("interrupted! ERROR!");
				line = null;
				break;
			}
			
			if (line != null) {
				int posOld = Console.CursorLeft;
				Console.Write(line);
				lineWidth = Console.CursorLeft - posOld;
			}
		}
	}

	class MainClass
	{
		private Downloader downloader;
		
		private PackageListsManager pkgListMan;
		
		public MainClass()
		{
			downloader = new Downloader();
			DownloadListener dl = new DownloadListener();
			downloader.DownloadEventRaised += new EventHandler<DownloadEventArgs>(dl.OnDownload);
			
			pkgListMan = new PackageListsManager();
		}
		
		public void Update()
		{
			update(true);
		}
		
		public void LocalUpdate()
		{
			update(false);
		}
		
		public void update(bool downloadPackageListFlag)
		{
			NaGet.SubCommands.NaGetUpdate tasks = new NaGet.SubCommands.NaGetUpdate(pkgListMan, downloadPackageListFlag);
			tasks.Downloader = this.downloader;
			tasks.TaskSetRaised += delegate(object sender, NaGetTaskSetEventArgs e) { 
				switch (e.Type) {
//					case NaGetTaskSetEventType.COMPLETED_TASKSET
//						break;
					case NaGetTaskSetEventType.COMPLETED:
						Console.WriteLine("Done.");
						break;
					case NaGetTaskSetEventType.STARTED_TASKSET:
						Console.Write("  " + e.TaskMessage);
						break;
					case NaGetTaskSetEventType.COMPLETED_TASKSET:
						Console.WriteLine(" ... Done. [{0}%]", (int) e.TaskProgressPercent);
						break;
					case NaGetTaskSetEventType.INFO:
						Console.WriteLine("  " + e.TaskMessage);
						break;
					case NaGetTaskSetEventType.ERROR:
					case NaGetTaskSetEventType.WARNING:
						Console.WriteLine("  [Error]" + e.TaskMessage);
						break;
				}
			};
			
			tasks.Run();
		}
		
		public void Upgrade()
		{
			PackageList<Package> packageList = pkgListMan.AvailablePkgList;
			
			VersionComparetor verComp = new VersionComparetor();
			
			foreach (InstalledPackage pkg in pkgListMan.InstalledPkgList.Packages) {
				Package avaiablePkg = packageList.GetPackageForName(pkg.Name);
				
				if (avaiablePkg != null) {
					if (verComp.Compare(pkg.Version, avaiablePkg.Version) < 0 &&
					    pkgListMan.InstalledPkgList.GetPackageForPackage(pkg.Name, avaiablePkg.Version) == null) {
						Console.ForegroundColor = ConsoleColor.Green;
					}
					
					Console.WriteLine("{0} ({1}) => ({2})", pkg.Name, pkg.Version, avaiablePkg.Version);
					Console.ResetColor();
				}
			}
			foreach (InstalledPackage pkg in pkgListMan.SystemInstalledPkgList.Packages) {
				Package avaiablePkg = packageList.GetPackageForName(pkg.Name);
				
				if (avaiablePkg != null) {
					if (verComp.Compare(pkg.Version, avaiablePkg.Version) < 0 &&
					    pkgListMan.SystemInstalledPkgList.GetPackageForPackage(pkg.Name, avaiablePkg.Version) == null) {
						Console.ForegroundColor = ConsoleColor.Green;
					}
					
					Console.WriteLine("{0} ({1})@sys => ({2})", pkg.Name, pkg.Version, avaiablePkg.Version);
					Console.ResetColor();
				}
			}
		}
		
		public void Search(string keys)
		{
			foreach(Package package in pkgListMan.AvailablePkgList.Search(keys)) {
				Console.WriteLine("{0} ({1}) - {2}", package.Name, package.Version, package.Summary);
			}
			
			foreach(InstalledPackage package in pkgListMan.InstalledPkgList.Search(keys)) {
				Console.WriteLine("{0} ({1})[@install] - {2}", package.Name, package.Version, package.Summary);
			}
			
			foreach(InstalledPackage package in pkgListMan.SystemInstalledPkgList.Search(keys)) {
				Console.WriteLine("{0} ({1})[@sys] - {2}", package.Name, package.Version, package.Summary);
			}
		}
		
		public void Show(string packagename)
		{
			PackageList<Package> allPkgs = new PackageList<Package>();
			allPkgs.AddPackages(pkgListMan.GetAllPackages());
			
			foreach (Package pkg in allPkgs.GetPackagesForName(packagename)) {
				Console.WriteLine("Package: {0}", pkg.Name);
				Console.WriteLine("Version: {0}", pkg.Version);
				if (pkg.Url != null) Console.WriteLine("Website: {0}", pkg.Url.Href);
				if (pkg.Tags != null) Console.WriteLine("Tag: {0}", pkg.Tags.ToLower());
				Console.WriteLine("Type: {0}", pkg.Type);
				if (pkg.License != null) Console.WriteLine("License: {0}", pkg.License);
				if (pkg is InstalledPackage) Console.WriteLine("State: Installed");
				
				if (pkg.Description != null) {
					Console.WriteLine("Description:");
					Console.WriteLine(pkg.Description);
				}
				Console.WriteLine();
			}
		}
		
		public void Download(string[] packagenames)
		{
			Installation[] installations = null;
			{
				List<Package> downloadList = new List<Package>();
			
				foreach(string packagename in packagenames) {
					Package foundPackage = pkgListMan.AvailablePkgList.GetPackageForName(packagename);
					if (foundPackage == null) {
						Console.WriteLine("E: Couldn't find package "+packagename);
						Environment.Exit(1);
					}
					if (! downloadList.Contains(foundPackage)) {
						downloadList.Add(foundPackage);
					}
				}
				
				installations = new Installation[downloadList.Count];
				for (int i = 0; i < installations.Length; i++) {
					installations[i] = new Installation((Package) downloadList[i]);
				}
			}
			
			Console.WriteLine("The following packages will be downloaded:");
			Console.WriteLine("  {0}", Installation.ToString(installations));
			if (AllGet.Util.Confirm("Do you want to continue [Y/n]?", true) == false) {
				Console.WriteLine("Abort.");
				Environment.Exit(0);
			}
			
			{
				int i = 0;
				foreach (Installation inst in installations) {
					i++;
					
					if (! inst.IsInstallablePackage()) {
						Console.WriteLine("E:{0} {1} can not be installed.", i, inst.ToString());
						continue;
					}
					
					try {
						Console.Write("Get:{0} {1}", i, inst.ToString());
						inst.Download(downloader);
						Console.WriteLine(" ...Done");
					} catch (Exception e) {
						Console.WriteLine("E: "+e.Message);
						throw new ApplicationException(string.Format("Failed to download {0}", inst.ToString()), e);
					}
				}
				
				i = 0;
				foreach (Installation inst in installations) {
					i++;
					
					if (inst.IsInstallablePackage() && inst.VerifyHashValues() == false) {
						Console.WriteLine("Verify:{0} {1} does not match hash value!", i, inst.ToString());
					}
				}
			}
		}
		
		public void Install(string[] packagenames)
		{
			if (! NaGet.Utils.IsAdministrators()) {
				Console.WriteLine("W: you are not administrators!");
			}
			
			Installation[] installations = null;
			{
				List<Package> downloadList = new List<Package>();
			
				foreach(string packagename in packagenames) {
					Package foundPackage = pkgListMan.AvailablePkgList.GetPackageForName(packagename);
					if (foundPackage == null) {
						Console.WriteLine("E: Couldn't find package "+packagename);
						Environment.Exit(1);
					}
					if (! downloadList.Contains(foundPackage)) {
						downloadList.Add(foundPackage);
					}
				}
				
				installations = new Installation[downloadList.Count];
				for (int i = 0; i < installations.Length; i++) {
					installations[i] = new Installation((Package) downloadList[i]);
					
					if (! installations[i].IsInstallablePackage()) {
						Console.WriteLine("E: Can not install package {0} to your PC.", installations[i].ToString());
						Environment.Exit(1);
					}
				}
			}
			
			Console.WriteLine("The following packages will be downloaded:");
			Console.WriteLine("  {0}", Installation.ToString(installations));
			if (AllGet.Util.Confirm("Do you want to continue [Y/n]?", true) == false) {
				Console.WriteLine("Abort.");
				Environment.Exit(0);
			}
			
			NaGet.SubCommands.NaGetInstall tasks = new NaGet.SubCommands.NaGetInstall(pkgListMan, installations);
			tasks.Downloader = this.downloader;
			tasks.TaskSetRaised += delegate(object sender, NaGetTaskSetEventArgs e) { 
				switch (e.Type) {
//					case NaGetTaskSetEventType.COMPLETED_TASKSET
//						break;
					case NaGetTaskSetEventType.COMPLETED:
						Console.WriteLine("Done.");
						break;
					case NaGetTaskSetEventType.STARTED_TASKSET:
						Console.Write("  " + e.TaskMessage);
						break;
					case NaGetTaskSetEventType.COMPLETED_TASKSET:
						Console.WriteLine(" ... Done. [{0}%]", (int) e.TaskProgressPercent);
						break;
					case NaGetTaskSetEventType.INFO:
						Console.WriteLine("  " + e.TaskMessage);
						break;
					case NaGetTaskSetEventType.ERROR:
					case NaGetTaskSetEventType.WARNING:
						Console.WriteLine("  [Error] " + e.TaskMessage);
						break;
				}
			};
			
			tasks.Run();
		}
		
		public void Remove(string[] packagenames)
		{
			if (! NaGet.Utils.IsAdministrators()) {
				Console.WriteLine("W: you are not administrators!");
			}
			
			this.LocalUpdate();
			
			PackageList<InstalledPackage> installedPkgList = new PackageList<InstalledPackage>();
			installedPkgList.AddPackages(pkgListMan.GetAllInstalledPackages());
			
			Uninstallation[] uninstallations = new Uninstallation[packagenames.Length];
			for (int i = 0; i < packagenames.Length; i++) {
				InstalledPackage pkg = installedPkgList.GetPackageForName(packagenames[i]);
				
				if (pkg == null) {
					Console.WriteLine("E: could not found package " + packagenames[i]);
					Environment.Exit(1);
				}
				
				uninstallations[i] = new Uninstallation(pkg);
			}
			
//			if (AllGet.Util.Confirm("Do you want to continue [Y/n]?", true) == false) {
//				Console.WriteLine("Abort.");
//				Environment.Exit(0);
//			}
			
			NaGet.SubCommands.NaGetUninstall tasks = new NaGet.SubCommands.NaGetUninstall(pkgListMan, uninstallations);
			tasks.TaskSetRaised += delegate(object sender, NaGetTaskSetEventArgs e) { 
				switch (e.Type) {
//					case NaGetTaskSetEventType.COMPLETED_TASKSET
//						break;
					case NaGetTaskSetEventType.COMPLETED:
						Console.WriteLine("Done.");
						break;
					case NaGetTaskSetEventType.STARTED_TASKSET:
						Console.Write("  " + e.TaskMessage);
						break;
					case NaGetTaskSetEventType.COMPLETED_TASKSET:
						Console.WriteLine(" ... Done. [{0}%]", (int) e.TaskProgressPercent);
						break;
					case NaGetTaskSetEventType.INFO:
						Console.WriteLine("  " + e.TaskMessage);
						break;
					case NaGetTaskSetEventType.ERROR:
					case NaGetTaskSetEventType.WARNING:
						Console.WriteLine("  [Error] " + e.TaskMessage);
						break;
				}
			};
			
			tasks.Run();
		}
		
		public void CleanCache(string[] packages)
		{
			if (! Directory.Exists(NaGet.Env.ArchiveFolderPath)) {
				return;	
			}
			
			int i = 0;
			if (packages.Length == 0) {
				foreach (string folder in Directory.GetDirectories(NaGet.Env.ArchiveFolderPath)) {
					Directory.Delete(folder, true);
					i ++;
				}
			} else {
				foreach (string package in packages) {
					foreach (string folder in Directory.GetDirectories(NaGet.Env.ArchiveFolderPath, package+"*", SearchOption.TopDirectoryOnly)) {
						Directory.Delete(folder, true);
						i ++;
					}
				}
			}
			if (i > 0) {
				Console.WriteLine("... Done.");
			}
		}
		
		public void Help()
		{
			string executeFileName = System.AppDomain.CurrentDomain.FriendlyName;
			Console.Write("Usage:");
			
			Console.WriteLine("\t{0} update|localupdate", executeFileName);
			Console.WriteLine("\t{0} search|show pkg1 [pkg2 ...]", executeFileName);
			Console.WriteLine("\t{0} cleancache [pkg ...]", executeFileName);
			Console.WriteLine("\t{0} download pkg1 [pkg2 ...]", executeFileName);
			Console.WriteLine("\t{0} install|uninstall pkg1 [pkg2 ...]", executeFileName);
			Console.WriteLine();
			Console.WriteLine("{0} is a simple command line interface for downloading and "+
			                  "installing packages. The most frequently used commands are update "+
			                  "and install.", executeFileName);
			Console.WriteLine();
			Console.WriteLine("Commands:");
			Console.WriteLine("   update - Retrieve new lists of packages");
			Console.WriteLine("   localupdate - Update installed-soft-list only");
			Console.WriteLine("   search - Search the package list for not a regex pattern");
			Console.WriteLine("   show - Show package detail");
			Console.WriteLine("   cleancache - Clear cached archived file(s)");
			Console.WriteLine("   download - Download only - do NOT install or unpack archives");
			Console.WriteLine("   install - Install new packages");
			Console.WriteLine("   remove - Uninstall packages");
			Console.WriteLine();
		}
		
		public void FooBar()
		{
			foreach (InstalledPackage pkg in pkgListMan.SystemInstalledPkgList.Packages) {
				Console.WriteLine("{0} : {1}", pkg.Name, pkg.UninstallInfo.UninstallString);
			}
		}
		
		public void Hoge()
		{
			foreach (UninstallInformation uInfo in RegistriedUninstallers.Uninstallers) {
				if (! uInfo.IsOSPatch && ! uInfo.IsSystemComponent) {
					Console.WriteLine("{0}", uInfo.DisplayName);
				}
			}
		}
		
		public static void Main(string[] args)
		{
			// アーカイブSYSTEM32をパスに足す
			NaGet.Utils.AddDirectoryToPath(NaGet.Env.ArchiveSystem32);
			
			MainClass mc = new MainClass();
			
			if (args.Length == 0) {
				mc.Help();
			} else if (args[0] == "update") {
				if (args.Length != 1) {
					Console.WriteLine("E: The update command takes no arguments");
					Environment.Exit(1);
				}
				mc.Update();
			} else if (args[0] == "localupdate") {
				if (args.Length != 1) {
					Console.WriteLine("E: The update command takes no arguments");
					Environment.Exit(1);
				}
				mc.LocalUpdate();
			} else if (args[0] == "search") {
				if (args.Length <= 1) {
					Console.WriteLine("E: You must give exactly one pattern");
					Environment.Exit(1);
				}
				
				mc.Search(string.Join(" ", args, 1, args.Length - 1));
			} else if (args[0] == "show") {
				if (args.Length <= 1) {
					Console.WriteLine("E: You must give exactly one pattern");
					Environment.Exit(1);
				}
				
				for (int i = 1; i < args.Length; i++) {
					mc.Show(args[i]);
				}
			} else if (args[0] == "download") {
				if (args.Length <= 1) {
					Console.WriteLine("E: Invalid operation download");
					Environment.Exit(1);
				}
				
				string[] packages = new string[args.Length - 1];
				Array.Copy(args, 1, packages, 0, packages.Length);
				
				mc.Download(packages);
			} else if (args[0] == "install") {
				if (args.Length <= 1) {
					Console.WriteLine("E: Invalid operation install");
					Environment.Exit(1);
				}
				
				string[] packages = new string[args.Length - 1];
				Array.Copy(args, 1, packages, 0, packages.Length);
				
				mc.Install(packages);
			} else if (args[0] == "upgrade") {
				mc.Upgrade();
			} else if (args[0] == "remove") {
				if (args.Length <= 1) {
					Console.WriteLine("E: Invalid operation remove");
					Environment.Exit(1);
				}
				
				string[] packages = new string[args.Length - 1];
				Array.Copy(args, 1, packages, 0, packages.Length);
				
				mc.Remove(packages);
			} else if (args[0] == "cleancache") {
				string[] packages = new string[args.Length - 1];
				Array.Copy(args, 1, packages, 0, packages.Length);
				
				mc.CleanCache(packages);
			} else if (args[0] == "foobar") {
				mc.FooBar();
			} else if (args[0] == "hoge") {
				mc.Hoge();
			} else {
				mc.Help();
				
				Environment.Exit(1);
			}
		}
	}
}
