﻿using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.Win32;

namespace NaGet.Packages.Install
{
	/// <summary>
	/// レジストリ登録されているアンインストーラに関するユーティリティ
	/// </summary>
	public sealed class RegistriedUninstallers
	{
		/// <summary>
		/// アンインストーラのレジストリの格納されているルートキーの文字列表現
		/// </summary>
		public const string UninstallersKey = @"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall";
		
		/// <summary>
		/// アンインストーラのレジストリの格納されているルートキーの文字列表現
		/// </summary>
		public const string UninstallersKeyWow6432 = @"SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall";
		
		/// <summary>
		/// アンインストーラのレジストリのキーを返す。
		/// </summary>
		public static IEnumerable<RegistryKey> RegistryKeies {
			get {
				RegistryKey key = null;
				
				// HKLM
				
				// UninstallersKey
				try {
					key = Registry.LocalMachine.OpenSubKey(UninstallersKey, false);
				} catch (System.Security.SecurityException) {
				} 
				if (key != null) {
					yield return key;
					key.Close();
				}
				
				// UninstallersKeyWow6432
				try {
					key = Registry.LocalMachine.OpenSubKey(UninstallersKeyWow6432, false);
				} catch (System.Security.SecurityException) {
				} 
				if (key != null) {
					yield return key;
					key.Close();
				}
				
				// HKCU
				
				// UninstallersKey
				try {
					key = Registry.CurrentUser.OpenSubKey(UninstallersKey, false);
				} catch (System.Security.SecurityException) {
				} 
				if (key != null) {
					yield return key;
					key.Close();
				}
				
				// UninstallersKeyWow6432
				try {
					key = Registry.CurrentUser.OpenSubKey(UninstallersKeyWow6432, false);
				} catch (System.Security.SecurityException) {
				} 
				if (key != null) {
					yield return key;
					key.Close();
				}
			}
		}
	
		
		/// <summary>
		/// アンインストーラをイテレートする
		/// </summary>
		public static IEnumerable<UninstallInformation> Uninstallers {
			get {
				foreach (RegistryKey regkey in RegistryKeies) {
					foreach (string key in regkey.GetSubKeyNames()) {
						UninstallInformation info;
						using (RegistryKey subregkey = regkey.OpenSubKey(key, false)) {
							info = UninstallInformation.NewInstance(subregkey);
						}
						
						if (info.IsOSPatch || info.IsSystemComponent || string.IsNullOrEmpty(info.DisplayName) ) {
							continue;
						}
						
						yield return info;
					}
				}
			}
		}
		
		/// <summary>
		/// レジストリを走査してインストール済みのソフトを検出する
		/// </summary>
		/// <param name="list">
		/// 参照するパッケージリスト
		/// </param>
		/// <returns>
		/// インストール済みのパッケージを返すイテレータ
		/// </returns>
		public static IEnumerable<InstalledPackage> DetectInstalledPackages(PackageList<Package> list)
		{
			foreach (UninstallInformation info in RegistriedUninstallers.Uninstallers) {
				foreach (Package pkg in list.Packages) {
					if (pkg.Type != InstallerType.ARCHIVE && pkg.UninstallerKey != null) {
						Match match = Regex.Match(info.DisplayName, pkg.UninstallerKey);
						
						if (match.Success) {
							yield return InstalledPackage.PackageConverter(pkg, info);
							
							break;
						}
					}// else continue;
				}
			}
		}
		
		/// <summary>
		/// パッケージに対応するインストールパッケージを返す。
		/// インストール終了確認などに使用。
		/// </summary>
		/// <param name="pkg">対応するパッケージ</param>
		/// <returns>インストール情報</returns>
		public static InstalledPackage GetInstalledPackageFor(Package pkg)
		{
			if (pkg.Type == InstallerType.ARCHIVE || pkg.Type == InstallerType.CANNOT_INSTALL) {
				return null;
			}
			
			foreach (UninstallInformation info in RegistriedUninstallers.Uninstallers) {
				Match match = Regex.Match(info.DisplayName, pkg.UninstallerKey);
				
				if (match.Success) {
					return InstalledPackage.PackageConverter(pkg, info);
				}
			}
			return null;
		}
	}
}
