﻿// Vistaの効果を有効にするフラグ（不要の場合はコメントアウト）
#define USE_VISTA_EFFECTS

using System;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Drawing;

namespace AppliStation.Util
{
	/// <summary>
	/// Win32ネイティブメソッドを叩いてGUI操作するための関数群のクラス
	/// </summary>
	public sealed class NativeMethods
	{
		/// <summary>
		/// 呼び出し禁止
		/// </summary>
		private NativeMethods()
		{
		}

		/// <summary>
		/// WindowsVista向け、プログレスバーステータス(色)を設定する
		/// </summary>
		/// <param name="progBar">対象のプログレスバー</param>
		/// <param name="state">状態。(1:Normal,2:Error,3:Paused)</param>
		public static void ProgressBar_SetState(ProgressBar progBar, uint state)
		{
#if USE_VISTA_EFFECTS
			try {
				// status := (PBST_NORMAL | PBST_ERROR | PBST_PAUSED)
				// SendMessage(progressBar.Handle, PBM_SETSTATE, state, 0);
				SendMessage(progBar.Handle, 0x410, new IntPtr(state), IntPtr.Zero);
			} catch (Exception) {
			}
#endif
		}
		
		#region タスクバーおよびタイトルバーのフラッシュ
		
		/// <summary>
		/// タスクバーおよびタイトルバーボタンのフラッシュの設定フラグ
		/// </summary>
		public enum FlashFlag : uint {
			/// <summary>
			/// 点滅の停止
			/// </summary>
			Stop = 0,
			/// <summary>
			/// タイトルバーを点滅
			/// </summary>
			Caption = 1,
			/// <summary>
			/// タスクバーボタンを点滅
			/// </summary>
			Tray = 2,
			/// <summary>
			/// タイトルバーとタスクバーボタンを点滅
			/// </summary>
			All = 3,
			/// <summary>
			/// Stopが設定されるまで点滅する
			/// </summary>
			Timer = 4,
			/// <summary>
			/// フォアグラウンドの状態になるまで点滅
			/// </summary>
			TimerNoFG = 12,
		}
		
		[StructLayout(LayoutKind.Sequential)]
		struct FLASHWINFO
		{
		    public int cbSize;
		    public IntPtr hWnd;
		    public FlashFlag dwFlags;
		    public uint uCount;
		    public uint dwTimeout;
		}
		
		/// <summary>
		/// タスクバーおよびタイトルバーボタンを点滅させる
		/// </summary>
		/// <param name="form">対象フォーム</param>
		/// <param name="flag">点滅パラメータフラグ</param>
		/// <param name="count">点滅回数</param>
		/// <param name="timeout">点滅の間隔(ミリ秒)</param>
		/// <returns></returns>
		public static bool Form_FlashWindow(Form form, FlashFlag flag, uint count, uint timeout)
		{
			try {
				FLASHWINFO info = new FLASHWINFO();
				info.cbSize = Marshal.SizeOf(typeof(FLASHWINFO));
				info.hWnd = form.Handle;
				info.dwFlags = flag;
				info.uCount = count;
				info.dwTimeout = timeout;
				
				return FlashWindowEx(ref info) == 0;
			} catch (Exception) {
				return false;
			}
		}
		
		[DllImport("user32.dll")]
		static extern Int32 FlashWindowEx(ref FLASHWINFO pwfi);
		
		#endregion
		
		#region ListView関連
		
		/// <summary>
		/// リストビューにダブルバッファでの描画をするか否かを設定する
		/// </summary>
		/// <remarks>マウスでの選択に半透明ツールを採用するか否かもこの設定に依存</remarks>
		/// <param name="listView">対象のリストビュー</param>
		/// <param name="bEnable">ダブルバッファでの描画をするとき<code>true</code></param>
		public static void ListView_SetDoubleBuffer(ListView listView, bool bEnable)
		{
			try {
				// SendMessage(listView.Handle, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_DOUBLEBUFFER, bEnable? LVS_EX_DOUBLEBUFFER:0);
				SendMessage(listView.Handle, 0x1036, new IntPtr(0x00010000), new IntPtr((bEnable)? 0x00010000u:0x0u));
			} catch (Exception) {
			}
		}
		
		/// <summary>
		/// 選択されたアイテムの部分の背景にグラデーションがかかった感じになる、
		/// Vista以降でのエクスプローラの見た目をListViewに反映させる。
		/// </summary>
		/// <remarks>Vista未満のバージョンでは何もしない。</remarks>
		/// <param name="listView">対象のListView</param>
		public static void ListView_EnableVistaExplorerTheme(ListView listView)
		{
#if USE_VISTA_EFFECTS
			// Vista未満はなにもしない
			OperatingSystem os = Environment.OSVersion;
			if (os.Platform != PlatformID.Win32NT || os.Version.Major < 6) return;
			
			try {
				SetWindowTheme(listView.Handle, "explorer", null);
			} catch (Exception) {
			}
#endif
		}
		
		/// <summary>
		/// ヘッダに"すべて選択"に似たチェックボックスを作るか否かを指定する
		/// </summary>
		/// <remarks>このオプションを設定するとVistaエクスプローラでの「チェックボックスを使用して項目を選択する」と同様の動作になる</remarks>
		/// <param name="listView">対象のListBox</param>
		/// <param name="bAutoCheckSelect">チェックボックスを使用して項目を選択するとき<code>true</code></param>
		public static void ListView_SetAutoCheckSelect(ListView listView, bool bAutoCheckSelect)
		{
#if USE_VISTA_EFFECTS
			try {
				// SendMessage(listView.Handle, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_AUTOCHECKSELECT, bAutoCheckSelect?LVS_EX_AUTOCHECKSELECT:0);
				SendMessage(listView.Handle, 0x1036, new IntPtr(0x08000000), new IntPtr((bAutoCheckSelect)?0x08000000u:0x0u));
			} catch (Exception) {
			}
#endif
		}
		
		#region ColumnHeaderのソートの三角印用
		
		[StructLayout(LayoutKind.Sequential, CharSet=CharSet.Auto)]
		private struct HD_ITEM
		{
			public uint    mask;
			public int     cxy;
			[MarshalAs(UnmanagedType.LPTStr)]public string   pszText;
			public IntPtr   hbm;
			public int     cchTextMax;
			public int     fmt;
			[MarshalAs(UnmanagedType.LPTStr)]public string  lParam;
			public int     iImage;        // index of bitmap in ImageList
			public int     iOrder;
		}
		
		/// <summary>
		/// WinXP以降、ソートの矢印を表示
		/// </summary>
		/// <param name="listView">対象のListView</param>
		/// <param name="column">表示する矢印のヘッダ</param>
		/// <param name="order">ソートの昇順・降順</param>
		public static void ColumnHeader_SetSortState(ListView listView, int column, SortOrder order)
		{
			try {
				// SendMessage(hWnd, LVM_GETHEADER, NULL, NULL);
				IntPtr hWnd = SendMessage(listView.Handle, 0x101F, IntPtr.Zero, IntPtr.Zero);
				
				HD_ITEM hdi = new HD_ITEM();
				hdi.mask = 0x0004; // HDI_FORMAT;
				for (int i = 0; i < listView.Columns.Count; i++) {
					// SendMessage(hWnd, HDM_GETITEMW, i, &hdi);
					SendMessage(hWnd, 0x120b, new IntPtr(i), ref hdi);
					
					const int HDF_SORTUP = 0x400;
					const int HDF_SORTDOWN = 0x200;
					
					if (i != column || order == SortOrder.None) {
						hdi.fmt = hdi.fmt & ~(HDF_SORTUP | HDF_SORTDOWN);
					} else if (order == SortOrder.Ascending) { // 昇順
						hdi.fmt = hdi.fmt & ~HDF_SORTDOWN | HDF_SORTUP;
					} else if (order == SortOrder.Descending) { // 降順
						hdi.fmt = hdi.fmt & ~HDF_SORTUP | HDF_SORTDOWN;
					}
					
					// SendMessage(hWnd, HDM_SETITEMW, i, &hdi);
					SendMessage(hWnd, 0x120c, new IntPtr(i), ref hdi);
				}
			} catch (Exception) {
			}
		}
		
		/// <summary>
		/// ヘッダ部のサイズを返す
		/// </summary>
		/// <param name="listView">ListView</param>
		/// <returns>ヘッダ部のクライアントサイズ</returns>
		public static Size ColumnHeader_GetSize(ListView listView)
		{
			LRECT lrect;
			
			try {
				// SendMessage(hWnd, LVM_GETHEADER, NULL, NULL);
				IntPtr hWnd = SendMessage(listView.Handle, 0x101F, IntPtr.Zero, IntPtr.Zero);
				GetClientRect(hWnd, out lrect);
			} catch {
				lrect.Left = lrect.Right = lrect.Top = lrect.Bottom = 0;
			}
			
			Size size = new Size(lrect.Right - lrect.Left,
			                lrect.Bottom - lrect.Top);
			return size;
		}
		
		#endregion
		
		#endregion
		
		#region EnableWindow(コメントアウト)
//		/// <summary>
//		/// 指定されたコントロール(ウィンドウ)への、
//		/// キーボード入力およびマウス入力を有効化または無効化
//		/// </summary>
//		/// <param name="control">対象のコントロールのハンドラ</param>
//		/// <param name="bEnable">有効にするか無効にするかを指定</param>
//		/// <returns>直前にウィンドウが無効状態だった場合はtrueを返す</returns>
//		public static bool Control_EnableWindow(Control ctrl, bool bEnable)
//		{
//			try {
//				return EnableWindow(ctrl.Handle, bEnable);
//			} catch {
//				ctrl.Enabled = bEnable;
//				return true;
//			}
//		}
//		
//		/// <summary>
//		/// 指定されたコントロール(ウィンドウ)への、
//		/// キーボード入力およびマウス入力を有効化または無効化
//		/// </summary>
//		/// <param name="control">対象のコントロールのハンドラ</param>
//		/// <param name="bEnable">有効にするか無効にするかを指定</param>
//		/// <returns>直前にウィンドウが無効状態だった場合はtrueを返す</returns>
//		[DllImport("user32.dll")]
//		public static extern bool EnableWindow(IntPtr hWnd, bool bEnable);
		#endregion
		
		#region RichTextBox関連
		
		[StructLayout(LayoutKind.Sequential)]
		private struct CHARFORMAT2
		{
			public int cbSize;
			public uint dwMask;
			public uint dwEffects;
			public int yHeight;
			public int yOffset;
			public int crTextColor;
			public byte bCharSet;
			public byte bPitchAndFamily;
			[MarshalAs(UnmanagedType.ByValArray, SizeConst=32)]
			public char[] szFaceName;
			public short wWeight;
			public short sSpacing;
			public int crBackColor;
			public int LCID;
			public uint dwReserved;
			public short sStyle;
			public short wKerning;
			public byte bUnderlineType;
			public byte bAnimation;
			public byte bRevAuthor;
			public byte bReserved1;
		}
		
		/// <summary>
		/// リッチテキストの選択部分に文字フォーマット効果を与える
		/// </summary>
		/// <param name="richTextBox">対象のリッチテキストボックス</param>
		/// <param name="dwMask">有効な文字フォーマット効果</param>
		/// <param name="dwEffect">文字フォーマット効果のフラグ</param>
		public static void RichTextBox_SetSelectionFormat(RichTextBox richTextBox, uint dwMask, uint dwEffect)
		{
			CHARFORMAT2 cfmt = new CHARFORMAT2();
			cfmt.cbSize = Marshal.SizeOf(typeof(CHARFORMAT2));
			cfmt.dwMask = dwMask;
			cfmt.dwEffects = dwEffect;
			
			//SendMessage(richTextBox.Handle, EM_SETCHARFORMAT, SCF_SELECTION, ref cfmt);
			SendMessage(richTextBox.Handle, 0x0444, new IntPtr(0x0001), ref cfmt);
		}
		/// <summary>
		/// リッチテキストにテキストリンクを追加する
		/// </summary>
		/// <param name="richTextBox">対象のリッチテキストボックス</param>
		/// <param name="text">追加するテキスト</param>
		public static void RichTextBox_AddTextLink(RichTextBox richTextBox, string text)
		{
			int pos = richTextBox.TextLength;
			richTextBox.Select(pos,0);
			richTextBox.SelectedText = text;
			richTextBox.Select(pos, text.Length);
			
			// RichTextBox_SetSelectionFormat(richTextBox, CFM_LINK, CFE_LINK);
			RichTextBox_SetSelectionFormat(richTextBox, 0x00000020, 0x0020);
			
			richTextBox.Select(richTextBox.TextLength, 0);
		}
		
		#endregion
		
		#region アイコン関連
		
		[DllImport("shell32.dll")]
		private static extern IntPtr ExtractIcon(IntPtr hInst, string lpszExeFileName, int nIconIndex);
		
		[DllImport("user32.dll", SetLastError=true)]
		[return: MarshalAs(UnmanagedType.Bool)]
		private static extern bool DestroyIcon(IntPtr hIcon);
		
		[StructLayout(LayoutKind.Sequential, CharSet=CharSet.Unicode)]
		private struct SHSTOCKICONINFO
		{
			public Int32 cbSize;
			public IntPtr hIcon;
			public Int32 iSysImageIndex;
			public Int32 iIcon;
			[MarshalAs(UnmanagedType.ByValTStr, SizeConst=260)] public string path;
		}
		
		[DllImport("shell32.dll", CharSet=CharSet.Unicode, ExactSpelling=true, SetLastError=true)]
		private static extern void SHGetStockIconInfo(uint siid, uint uFlags, ref SHSTOCKICONINFO sii);
		
		/// <summary>
		/// アイコンファイル（実行ファイル・DLL）を開いてアイコンを作る
		/// </summary>
		/// <remarks>内部でコピーされるのでWin32APIのDestroyIconを使わないでいいが、やや遅い動作</remarks>
		/// <param name="form">ハンドラ</param>
		/// <param name="lpszExeFileName">対象ファイル</param>
		/// <param name="nIconIndex">アイコンインデックス</param>
		/// <returns>生成されたアイコン</returns>
		public static Icon ExtractIcon(Form form, string lpszExeFileName, uint nIconIndex)
		{
			Icon ico = null;
			
			if (! System.IO.File.Exists(lpszExeFileName)) {
				ico = Icon.ExtractAssociatedIcon(lpszExeFileName); // ExtractAssociatedIconに例外を吐いてもらう
			} else {
				IntPtr hInst = (form != null)? form.Handle : IntPtr.Zero;
				IntPtr hIcon = IntPtr.Zero;
				
				try {
					hIcon = ExtractIcon(hInst, lpszExeFileName, (int) nIconIndex);
					if ((hIcon != IntPtr.Zero) && (hIcon.ToInt32() != 2)) {
						ico = (Icon) Icon.FromHandle(hIcon).Clone();
						DestroyIcon(hIcon);
					}
				} catch (System.Runtime.InteropServices.COMException) {
					// ExtraIconのP/Invoke失敗時用
				}
			}
			return ico;
		}
		
		/// <summary>
		/// ユーザ昇格が必要か設定し、設定必要なときシールドアイコンを表示する。
		/// </summary>
		/// <param name="button">対象のボタン</param>
		/// <param name="required">ユーザ昇格が必要か否か、すなわちシールドアイコンを表示するか</param>		
		public static void Button_SetElevationRequiredState(Button button, bool required)
		{
#if USE_VISTA_EFFECTS
			if (Environment.OSVersion.Version.Major >= 6) {
				button.FlatStyle = FlatStyle.System;
				// SendMessage(hWnd, BCM_SETSHIELD, 0, required);
				SendMessage(button.Handle, 0x160C, IntPtr.Zero, new IntPtr((required)? 1u : 0u));
			} else { // Legacy OS
#endif
				// FlatStyle.System に設定されている場合、Image プロパティに割り当てられているイメージは表示されない対策
				if (button.FlatStyle == FlatStyle.System) {
					button.FlatStyle = FlatStyle.Standard;
				}
				
				button.TextImageRelation = TextImageRelation.ImageBeforeText;
				if (required) {
					button.ImageAlign = ContentAlignment.MiddleLeft;
					using (Icon ico = new Icon(SystemIcons.Shield, 16, 16)) {
						button.Image = ico.ToBitmap();
					}
				} else {
					button.Image = null;
				}
				button.AutoSize = true;
#if USE_VISTA_EFFECTS
			}
#endif
		}
		
		/// <summary>
		/// シェルのソトックアイコンのハンドラを取得する。
		/// </summary>
		/// <param name="siid">ストックID</param>
		/// <param name="isSmall">アイコンサイズは小さいのであればtrueを指定</param>
		/// <returns>アイコンのハンドラ。存在しない場合、null</returns>
		private static IntPtr SHGetStockIconHandle(uint siid, bool isSmall)
		{
#if USE_VISTA_EFFECTS
			try {
				SHSTOCKICONINFO sii = new SHSTOCKICONINFO();
				sii.cbSize = Marshal.SizeOf(typeof(SHSTOCKICONINFO));
				sii.hIcon = IntPtr.Zero;
				
				//SHGetStockIconInfo(siid, SHGFI_ICON | ((isSmall)? SHGFI_SMALLICON : SHGFI_LARGEICON), ref sii);
				SHGetStockIconInfo(siid, 0x100u | ((isSmall)? 0x1u : 0x0u), ref sii);
				
				return sii.hIcon;
			} catch (Exception) {
				return IntPtr.Zero;
			}
#else
			return IntPtr.Zero;
#endif
		}
		
		/// <summary>
		/// ユーザ昇格が必要か設定し、設定必要なときシールドアイコンを表示する。
		/// </summary>
		/// <param name="label">対象のリンクラベル</param>
		/// <param name="required">ユーザ昇格が必要か否か、すなわちシールドアイコンを表示するか</param>		
		public static void LinkLabel_SetElevationRequiredState(LinkLabel label, bool required)
		{
			if (required) {
				IntPtr iconHandle = IntPtr.Zero;
				
				//SHGetStockIconHandle(SIID_SHIELD, true);
				iconHandle = SHGetStockIconHandle(77, true);
				if (iconHandle != IntPtr.Zero) {
					label.Image = Bitmap.FromHicon(iconHandle);
				} else {
					using (Icon ico = new Icon(SystemIcons.Shield, 16, 16)) {
						label.Image = ico.ToBitmap();
					}
				}
				
				label.ImageAlign = ContentAlignment.MiddleLeft;
				label.Padding = new Padding(label.Image.Width, label.Padding.Top, label.Padding.Right, label.Padding.Bottom);
			} else {
				label.Image = null;
				label.Padding = new Padding(0);
			}
		}
		
		#endregion
		
		[DllImport("user32.dll", CharSet = CharSet.Auto)]
		internal static extern IntPtr SendMessage(IntPtr hWnd, UInt32 Msg, IntPtr wParam, IntPtr lParam);
		
		[DllImport("user32.dll", CharSet=CharSet.Auto)]
		private static extern IntPtr SendMessage(IntPtr hWnd, UInt32 Msg, IntPtr wParam, ref HD_ITEM lParam);
		
		[DllImport("user32.dll", CharSet=CharSet.Auto)]
		private static extern IntPtr SendMessage(IntPtr hWnd, Int32 Msg, IntPtr wParam, ref CHARFORMAT2 lParam);

		[DllImport("uxtheme.dll", ExactSpelling=true, CharSet=CharSet.Unicode)]
		internal static extern int SetWindowTheme(IntPtr hWnd, String pszSubAppName, String pszSubIdList);
		
		[Serializable]
		[StructLayout(LayoutKind.Sequential)]
		private struct LRECT {
			public int Left;
			public int Top;
			public int Right;
			public int Bottom;
		}
		
		[DllImport("user32.dll")]
		private static extern bool GetClientRect(IntPtr hWnd, out LRECT lpRect);
	}
}
