﻿using System;
using System.Drawing;
using System.Windows.Forms;
using NaGet.Packages;
using NaGet.Packages.Install;
using System.IO;
using System.Collections.Generic;

using AppliStation.PackageInfo;

namespace AppliStation
{
	/// <summary>
	/// Description of PackageListViewForm.
	/// </summary>
	public partial class PackageListViewForm : Form
	{
		private PackageListsManager pkgListsMan = null;
		
		public PackageListViewForm()
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			
			installToolStripMenuItem.Font = new Font(installToolStripMenuItem.Font, FontStyle.Bold);
			uninstallToolStripMenuItem.Font = new Font(uninstallToolStripMenuItem.Font, FontStyle.Bold);
			
			pkgListsMan = new PackageListsManager();
			packageListView.Data = pkgListsMan;
			
			this.Icon = Icon.ExtractAssociatedIcon(Application.ExecutablePath);
			AppliStation.Util.NativeMethods.ListView_EnableVistaExplorerTheme(packageListView);
			AppliStation.Util.NativeMethods.ListView_SetDoubleBuffer(packageListView, true);
		}
		
		private void ShowInfoToDetailBoxFor(Package pkg)
		{
			detailBox.Clear();
			
			detailBox.SelectionFont = new Font(detailBox.Font.FontFamily, 12);
			detailBox.SelectedText += string.Format("{0} ({1})\r\n", pkg.Name, pkg.Version);
			if (! string.IsNullOrEmpty(pkg.Tags) ) {
				detailBox.SelectionFont = new Font(detailBox.Font.FontFamily, 8);
				detailBox.SelectedText += "タグ:";
				foreach (string tag in pkg.Tags.Split(' ')) {
					detailBox.AppendText(" ");
					AppliStation.Util.NativeMethods.RichTextBox_AddTextLink(detailBox, tag);
				}
				detailBox.AppendText("\r\n");
			}
			
			// インストール済みパッケージの場合
			InstalledPackage iPkg = pkg as InstalledPackage;
			if (iPkg != null) {
				System.Text.StringBuilder sb = new System.Text.StringBuilder();
				if (iPkg.UninstallInfo.InstallDate != null) {
					sb.AppendFormat("インストールした日: {0:d}  ", iPkg.UninstallInfo.InstallDate.Value);
				}
				if (iPkg.UninstallInfo.EstimatedSize > 0) {
					sb.AppendFormat("サイズ: {0}  ", NaGet.Utils.FormatSize(iPkg.UninstallInfo.EstimatedSize*1024));
				}
				
				if (sb.Length > 0) {
					detailBox.SelectionFont = new Font(detailBox.Font.FontFamily, 8);
					detailBox.SelectedText += sb.ToString();
					detailBox.SelectedText += "\r\n";
				}
			}
			detailBox.SelectionFont = detailBox.Font;
			if (pkg.Url != null && pkg.Url.Href != null) {
				detailBox.SelectedText += "公式サイト: ";
				AppliStation.Util.NativeMethods.RichTextBox_AddTextLink(detailBox, pkg.Url.Href);
				detailBox.SelectedText += "\r\n";
			}
			detailBox.SelectedText += pkg.Summary;
		}
		
		void PackageListViewSelectedIndexChanged(object sender, EventArgs e)
		{
			updateSelectedPackages();
		}
		
		private void updateSelectedPackages()
		{
			uint installPkgCount = 0;
			uint uninstallPkgCount = 0;
			bool installBtnEnabled, uninstallBtnEnabled;
			
			foreach (Package pkg in packageListView.SelectedPackages) {
				if (pkg is InstalledPackage) {
					uninstallPkgCount ++;
				} else {
					installPkgCount ++;
				}
			}
			
			uint pkgCount = installPkgCount + uninstallPkgCount;
			installBtnEnabled = (installPkgCount > 0) && (uninstallPkgCount == 0);
			uninstallBtnEnabled = (uninstallPkgCount == 1) && (installPkgCount == 0);
			
			// ToolStrip
			informationToolStripDropDownButton.Visible = (pkgCount == 1);
			uninstallToolStripButton.Visible = uninstallBtnEnabled;
			installToolStripButton.Visible = installBtnEnabled;
			// MenuStrip
			webResourceToolStripMenuItem.Visible = (pkgCount == 1);
			uninstallToolStripMenuItem.Visible = uninstallBtnEnabled;
			installToolStripMenuItem.Visible = installBtnEnabled;
			
			// detailBoxのメッセージ設定
			switch (pkgCount) {
				case 0:
					int count = packageListView.Items.Count;
					detailBox.Clear();
					detailBox.Text = (count > 0) ? string.Format("{0}個のソフトがあります。", count)
						: "該当するソフトがありません。";
					break;
				case 1:
					ShowInfoToDetailBoxFor(packageListView.SelectedPackage);
					break;
				default: // case 2 and over:
					detailBox.Clear();
					detailBox.Text = (installBtnEnabled)? string.Format("{0}個のソフトが選択されています。", installPkgCount) :
						(uninstallBtnEnabled)? string.Format("{0}個のインストール済みのソフトが選択されています。", uninstallPkgCount) :
						string.Format("{0}個のソフトが選択されています。\r\n(うち{1}個はインストール済み、{2}個はインストール可能)", pkgCount, uninstallPkgCount, installPkgCount);
					break;
			}
			
			try {
				detailBox.Select(0, 0);
				detailBox.ScrollToCaret();
			} catch (System.Runtime.InteropServices.COMException) {
				// ScrollToCaretでこけることがある
			}
		}

		void PackageListViewItemActivate(object sender, EventArgs e)
		{
			Package pkg = packageListView.SelectedPackage;
			
			if (pkg != null) {
				if (pkg is InstalledPackage) {
					UninstallToolStripButtonClick(sender, e);
				} else {
					InstallToolStripButtonClick(sender, e);
				}
			}
		}
		
		void Form_OnLoad(object sender, EventArgs e)
		{
			packageListViewImageList.Images.Add("installed", Icon.ExtractAssociatedIcon(Application.ExecutablePath));
			
			updatePackageFilterToolStripMenuItemCheckState();
			notInstalledPackageFilterToolStripMenuItem.Image = packageListViewImageList.Images["available-new"];
			installedASPackageFilterToolStripMenuItem.Image = packageListViewImageList.Images["installed"];
			installedSysPackageFilterToolStripMenuItem.Image = packageListViewImageList.Images["sys"];
			
			packageListView.UpdateItems();
			updateSelectedPackages();
		}
		
		#region PackageFilter関連
		
		/// <summary>
		/// packgageFilterToolStripのリストを更新する。
		/// </summary>
		private void updatePackageFilterToolStripMenuItemCheckState()
		{
			ToolStripMenuItem selected = getCheckedPackageFilterToolStripItem();
			
			foreach (ToolStripMenuItem item in packageFilterToolStripDropDownButton.DropDown.Items) {
				item.Checked = (selected == item);
			}
			
			packageFilterToolStripDropDownButton.Text = selected.Text;
			packageFilterToolStripDropDownButton.Image = selected.Image;
			packageFilterToolStripDropDownButton.ToolTipText = selected.ToolTipText;
		}
		
		private ToolStripMenuItem getCheckedPackageFilterToolStripItem()
		{
			switch (packageListView.FilteringType) {
				case PackageListViewPkgTypeFilter.NotInstalled:
					return notInstalledPackageFilterToolStripMenuItem;
				case PackageListViewPkgTypeFilter.InstalledAS:
					return installedASPackageFilterToolStripMenuItem;
				case PackageListViewPkgTypeFilter.InstalledSys:
					return installedSysPackageFilterToolStripMenuItem;
				default:
					return allPackageFilterToolStripMenuItem;
			}
		}
		
		void AnyPackageFilterToolStripMenuItemClicked(object sender, EventArgs e)
		{
			if (sender == notInstalledPackageFilterToolStripMenuItem) {
				packageListView.FilteringType = PackageListViewPkgTypeFilter.NotInstalled;
			} else if (sender == installedASPackageFilterToolStripMenuItem) {
				packageListView.FilteringType = PackageListViewPkgTypeFilter.InstalledAS;
			} else if (sender == installedSysPackageFilterToolStripMenuItem) {
				packageListView.FilteringType = PackageListViewPkgTypeFilter.InstalledSys;
			} else {
				packageListView.FilteringType = PackageListViewPkgTypeFilter.All;
			}
			
			updatePackageFilterToolStripMenuItemCheckState();
			searchTextBox.FireTextChangedTrigger();
		}
		
		#endregion
		
		void DetailBoxLinkClicked(object sender, LinkClickedEventArgs e)
		{
			if (System.Text.RegularExpressions.Regex.IsMatch(e.LinkText, "^https?://")){
				/* URLの場合はブラウザ起動 */
				try {
					System.Diagnostics.Process.Start(e.LinkText);
				} catch (System.ComponentModel.Win32Exception) {
					MessageBox.Show(string.Format("{0}を開くのに失敗しました。", e.LinkText), "ブラウザ起動エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			} else {
				/* それ以外はタグとみなして検索 */
				searchTextBox.Text = e.LinkText;
			}
		}
		
		internal void updateActionInvoke(bool downloadPackageListsFlag)
		{
			AppliStation.Util.ExecutionProgressViewer prog = new AppliStation.Util.ExecutionProgressViewer();
			prog.Shown += delegate(object sender2, EventArgs e2) {
				NaGet.SubCommands.NaGetUpdate tasks = new NaGet.SubCommands.NaGetUpdate(pkgListsMan, downloadPackageListsFlag);
				prog.SetTaskSet(tasks);
				prog.Refresh();
				prog.StartTaskSet();
			};
			prog.Text = "リストの更新";
			prog.ShowDialog(this);
		}
			
		void UpdateToolStripMenuItemClick(object sender, EventArgs e)
		{
			updateActionInvoke(true);
			
			UpdatePackageList();
		}
		
		void LocalupdateToolStripMenuItemClick(object sender, EventArgs e)
		{
			updateActionInvoke(false);
			
			UpdatePackageList();
		}
		
		void OptionToolStripMenuItemClick(object sender, EventArgs e)
		{
			UserPrefForm userPrefForm = new UserPrefForm();
			DialogResult result = userPrefForm.ShowDialog(this);
			
			if (result == DialogResult.OK) {
				if (userPrefForm.IsRepositoryListSettingChanged) {
					updateActionInvoke(true);
					UpdatePackageList();
				}
			}
		}
		
		#region searchTextBoxまわり
		
		void SearchTextBoxKeyPress(object sender, KeyPressEventArgs e)
		{
			switch (e.KeyChar) {
				case (char)Keys.Enter:
					searchTextBox.FireTextChangedTrigger();
					break;
				case (char)Keys.Escape:
					searchTextBox.Text = string.Empty;
					break;
			}
		}
		
		void SearchTextBoxTextChangedTriggerFired(object sender, EventArgs e)
		{
			packageListView.FilteringKeyword = searchTextBox.Text;
		}
		
		#endregion
		
		internal void installActionInvoke(Installation[] insts)
		{
			AppliStation.Util.ExecutionProgressViewer prog = new AppliStation.Util.ExecutionProgressViewer();
			prog.Shown += delegate(object sender2, EventArgs e2) {
				NaGet.SubCommands.NaGetInstall tasks = new NaGet.SubCommands.NaGetInstall(pkgListsMan, insts);
				prog.SetTaskSet(tasks);
				prog.Refresh();
				prog.StartTaskSet();
			};
			prog.Text = string.Format("ソフトウェアのインストール");
			prog.ShowDialog(this);
		}
		
		void InstallToolStripButtonClick(object sender, EventArgs e)
		{
			InstallationConfirmForm confirm = new InstallationConfirmForm();
			confirm.PkgListsManager = pkgListsMan;
			confirm.Installations = Installation.ConvertInstallations( NaGet.Utils.IEnumerable2Array(packageListView.SelectedPackages) );
			confirm.UseRunas = confirm.GetShouldUseRunas();
			DialogResult result = confirm.ShowDialog(this);
			
			if (result == DialogResult.OK) {
				Installation[] insts = confirm.CheckedInstallations;
				
				if (confirm.UseRunas) {
					installRunasActionInvoke(insts);
				} else {
					installActionInvoke(insts);
				}
				
				UpdatePackageList();
			}
		}
		
		public void installRunasActionInvoke(Installation[] insts)
		{
			this.setWindowEnabled(false);
			
			string tmpfileName = Path.GetTempFileName();
			try {
				NaGet.Utils.PutSerializeObject(tmpfileName, insts);
				
				System.Diagnostics.ProcessStartInfo procInfo = new System.Diagnostics.ProcessStartInfo();
				procInfo.FileName = Application.ExecutablePath;
				procInfo.Arguments = string.Format("--noupdate --cmd=install \"--instsref={0}\"", tmpfileName);
				procInfo.Verb = "runas";
				procInfo.WorkingDirectory = Environment.CurrentDirectory;
				
				System.Diagnostics.Process hProc = System.Diagnostics.Process.Start(procInfo);
				hProc.EnableRaisingEvents = true;
				hProc.SynchronizingObject = this;
				hProc.Exited += delegate(object sender, EventArgs e) {
					UpdatePackageList();
					
					this.setWindowEnabled(true);
					this.BringToFront();
					
					if (File.Exists(tmpfileName)) {
						File.Delete(tmpfileName);
					}
				};
			} catch (System.ComponentModel.Win32Exception ex) {
				MessageBox.Show(ex.Message, "インストール", MessageBoxButtons.OK, MessageBoxIcon.Error);
				
				if (File.Exists(tmpfileName)) {
					File.Delete(tmpfileName);
				}				this.setWindowEnabled(true);
			}
		}
		
		internal void uninstallActionInvoke(InstalledPackage[] pkgs)
		{
			AppliStation.Util.ExecutionProgressViewer prog = new AppliStation.Util.ExecutionProgressViewer();
			prog.Shown += delegate(object sender2, EventArgs e2) {
				NaGet.SubCommands.NaGetUninstall tasks = new NaGet.SubCommands.NaGetUninstall(pkgListsMan, pkgs);
				prog.SetTaskSet(tasks);
				prog.Refresh();
				prog.StartTaskSet();
			};
			prog.Text = string.Format("ソフトウェアのアンインストール");
			prog.ShowDialog(this);
		}
		
		internal void uninstallRunasActionInvoke(InstalledPackage[] pkgs)
		{			
			this.setWindowEnabled(false);
			
			string tmpfileName = Path.GetTempFileName();
			try {
				NaGet.Utils.PutSerializeObject(tmpfileName, pkgs);
				
				System.Diagnostics.ProcessStartInfo procInfo = new System.Diagnostics.ProcessStartInfo();
				procInfo.FileName = Application.ExecutablePath;
				procInfo.Arguments = string.Format("--noupdate --cmd=uninstall \"--pkgsref={0}\"", tmpfileName);
				procInfo.Verb = "runas";
				procInfo.WorkingDirectory = Environment.CurrentDirectory;
				
				System.Diagnostics.Process hProc = System.Diagnostics.Process.Start(procInfo);
				hProc.EnableRaisingEvents = true;
				hProc.SynchronizingObject = this;
				hProc.Exited += delegate(object sender, EventArgs e) {
					UpdatePackageList();
					
					this.setWindowEnabled(true);
					this.BringToFront();
					
					if (File.Exists(tmpfileName)) {
						File.Delete(tmpfileName);
					}
				};
			} catch (System.ComponentModel.Win32Exception ex) {
				MessageBox.Show(ex.Message, "アンインストール", MessageBoxButtons.OK, MessageBoxIcon.Error);
				
				this.setWindowEnabled(true);
				if (File.Exists(tmpfileName)) {
					File.Delete(tmpfileName);
				}
			}
		}
		
		void UninstallToolStripButtonClick(object sender, EventArgs e)
		{
			PackageUninstallConfirmForm confirm = new PackageUninstallConfirmForm();
			confirm.UninstallPackage = (InstalledPackage) packageListView.SelectedPackage;
			confirm.UseRunas = confirm.GetShouldUseRunas();
			DialogResult result = confirm.ShowDialog(this);
			
			if (result == DialogResult.OK) {
				InstalledPackage[] instPkgs = new InstalledPackage[]{confirm.UninstallPackage};
				
				if (confirm.UseRunas) {
					uninstallRunasActionInvoke(instPkgs);
				} else {
					uninstallActionInvoke(instPkgs);
				}
				
				UpdatePackageList();
			}
		}
		
		
		void WebOfficialMenuItemClick(object sender, EventArgs e)
		{
			Package pkg = packageListView.SelectedPackage;
			if (pkg != null) {
				string linkURL = pkg.Url.Href;
				
				if (! string.IsNullOrEmpty(linkURL)) {
					try {
						System.Diagnostics.Process.Start(linkURL);
					} catch (System.ComponentModel.Win32Exception) {
						MessageBox.Show(string.Format("{0}を開くのに失敗しました。", linkURL), "ブラウザ起動エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
					}
				}
			}
		}
		
		void WebGoogleSearchMenuItemClick(object sender, EventArgs e)
		{
			Package pkg = packageListView.SelectedPackage;
			if (pkg != null) {
				string q = System.Web.HttpUtility.UrlEncode(pkg.Name, System.Text.Encoding.UTF8);
				string googleURL = @"http://www.google.co.jp/search?q="+q;
				
				try {
					System.Diagnostics.Process.Start(googleURL);
				} catch (System.ComponentModel.Win32Exception) {
					MessageBox.Show("Googleを開くのに失敗しました。", "ブラウザ起動エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			}
		}
		
		void OpenInstalledDirectoryStripMenuItemClick(object sender, EventArgs e)
		{
			InstalledPackage pkg = (InstalledPackage) packageListView.SelectedPackage;
			if (pkg != null) {
				if (pkg.Type == InstallerType.ARCHIVE || pkg.Type == InstallerType.ITSELF) {
					System.Diagnostics.Process.Start(Path.Combine(NaGet.Env.ArchiveProgramFiles, pkg.Name));
				} else if (Directory.Exists(pkg.discoverInstalledLocation())) {
					System.Diagnostics.Process.Start(pkg.discoverInstalledLocation());
				}
			}
		}
		
		
		void PropertiesCommonMenuItemClick(object sender, EventArgs e)
		{
			if (packageListView.SelectedItems.Count >= 5) {
				string msg = string.Format("{0}個のプロパティダイアログを開こうとしています。\n継続しますか?", packageListView.SelectedItems.Count);
				if (MessageBox.Show(msg, "プロパティ", MessageBoxButtons.OKCancel, MessageBoxIcon.Question) != DialogResult.OK) {
					return;
				}
			}
			
			foreach (Package pkg in packageListView.SelectedPackages) {
				PackageInfoForm form = new PackageInfoForm();
				form.SelectedObject = pkg;
				form.Text = string.Format("{0}({1})のプロパティ", pkg.Name, pkg.Version);
				
				form.Show(this);
			}
		}
		
		private void updateLauncherMenuItem(AppliStation.Util.ToolStripPetitLauncherMenuItem launcherMenuItem)
		{
			bool selectionIsOnlyOne = packageListView.SelectedItems.Count == 1;
			
			if (selectionIsOnlyOne) {
				Package pkg = packageListView.SelectedPackage;
				InstalledPackage iPkg = pkg as InstalledPackage;
				
				// インストール済みパッケージのとき
				if (iPkg != null) {
					bool launcherMenuItemVisible = 
						(pkg.Type == InstallerType.ARCHIVE) ||
						(pkg.Type == InstallerType.ITSELF) ||
						Directory.Exists(iPkg.discoverInstalledLocation());
					
					launcherMenuItem.Visible = launcherMenuItemVisible;
					if (launcherMenuItemVisible) {
						launcherMenuItem.BaseFolderPath = iPkg.discoverInstalledLocation();
					}
				} else {
					launcherMenuItem.Visible = false;
				}
			} else {
				launcherMenuItem.Visible = false;
			}
		}
		
		void PackageListContextMenuStripOpening(object sender, System.ComponentModel.CancelEventArgs e)
		{
			bool selectionIsOnlyOne = packageListView.SelectedItems.Count == 1;
			bool hasSelection = packageListView.SelectedItems.Count > 0;
			
			updateSelectedPackages();
			
			// インストール先のフォルダの設定
			updateLauncherMenuItem(installedDirectoryToolStripMenuItem);
			
			if (packageListView.View == View.Details) {
				// ヘッダ部がクリックされたとき、パッケージが選択されていないものとして扱って処理をする。
				int headerHeight = AppliStation.Util.NativeMethods.ColumnHeader_GetSize(packageListView).Height;
				if (packageListView.PointToClient(packageListContextMenuStrip.Location).Y < headerHeight) {
					selectionIsOnlyOne = hasSelection = false;
					installToolStripMenuItem.Visible = false;
					uninstallToolStripMenuItem.Visible = false;
					installedDirectoryToolStripMenuItem.Visible = false;
				}
			}
			
			packageListContextMenuStripSeparator.Visible = selectionIsOnlyOne;
			webResourceToolStripMenuItem.Visible = selectionIsOnlyOne;
			propertiesToolStripMenuItem.Visible = hasSelection;
			columnToolStripMenuItem.Visible = (! hasSelection) && (packageListView.View == View.Details);
		}
		
		void InformationToolStripDropDownButtonDropDownOpening(object sender, EventArgs e)
		{
			bool selectionIsOnlyOne = packageListView.SelectedItems.Count == 1;
			bool hasSelection = packageListView.SelectedItems.Count > 0;
			
			// インストール先のフォルダの設定
			updateLauncherMenuItem(installedDirectoryMenuItem);
			
			webResourceMenuItem.Visible = selectionIsOnlyOne;
			propertiesMenuItem.Visible = hasSelection;
		}
		
		void WebResourceCommonContextMenuStripOpening(object sender, System.ComponentModel.CancelEventArgs e)
		{
			Package pkg = packageListView.SelectedPackage;
			webOfficialMenuItem.Enabled = (pkg != null && pkg.Url != null && !string.IsNullOrEmpty(pkg.Url.Href));
			// webGoogleSearchMenuItem always active.
		}
		
		private IEnumerable<Package> getUpdatedPackages(PackageList<InstalledPackage> installedPkgList, PackageList<Package> avaiablePkgList, IComparer<string> verComp)
		{
			foreach (InstalledPackage pkg in installedPkgList) {
				Package avaiablePkg = avaiablePkgList.GetPackageForName(pkg.Name);
				
				if (avaiablePkg != null) {
					if (verComp.Compare(pkg.Version, avaiablePkg.Version) < 0 &&
					    installedPkgList.GetPackageForPackage(pkg.Name, avaiablePkg.Version) == null) {
						
						yield return avaiablePkg;
					}
				}
			}
		}
		
		void UpgradeToolStripButtonClick(object sender, EventArgs e)
		{
			List<Package> pkgs;
			VersionComparetor verComp = new VersionComparetor();
			PackageList<Package> avaiablePackageList = pkgListsMan.AvailablePkgList;
			
			pkgs = NaGet.Utils.MergeList(
				getUpdatedPackages(pkgListsMan.InstalledPkgList, avaiablePackageList, verComp),
				getUpdatedPackages(pkgListsMan.SystemInstalledPkgList, avaiablePackageList, verComp)
			);
			
			if (pkgs.Count <= 0) {
				MessageBox.Show(this, "更新されたソフトはありません", "ソフトの更新");
				return;
			}
			
			InstallationConfirmForm confirm = new InstallationConfirmForm();
			confirm.PkgListsManager = pkgListsMan;
			confirm.Installations = Installation.ConvertInstallations(pkgs.ToArray());
			DialogResult result = confirm.ShowDialog(this);
			
			if (result == DialogResult.OK) {
				Installation[] insts = confirm.CheckedInstallations;
				
				if (confirm.UseRunas) {
					installRunasActionInvoke(insts);
				} else {
					installActionInvoke(insts);
				}
				
				UpdatePackageList();
			}
		}
		
		protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
		{
			if ((int)keyData == (int)Keys.Control + (int)Keys.E) {
				searchTextBox.SelectAll();
				searchTextBox.Focus();
				return true;
			}
			
			return base.ProcessCmdKey(ref msg, keyData);
		}
		
		public void UpdatePackageList()
		{
			pkgListsMan.LoadPackageLists();
			packageListView.UpdateItems();
			updateSelectedPackages();
		}
		
		/// <summary>
		/// 自ウィンドウの有効無効(Enabled)を(必要あればInvokeして)実行する 
		/// </summary>
		/// <param name="enabled">有効か否か。Enabledの値に入れられる</param>
		private void setWindowEnabled(bool enabled)
		{
			MethodInvoker process = (MethodInvoker) delegate() {
				this.Enabled = enabled;
			};
			
			if (InvokeRequired) {
				Invoke(process);
			} else {
				process.Invoke();
			}
		}
		
		void ColumnCommonToolStripMenuItemClick(object sender, EventArgs e)
		{
			packageListView.BeginUpdate();
			ColumnHeader sortcolumn = packageListView.SortColumn;
			
			// 列の追加と削除
			foreach (ToolStripItem item in columnToolStripMenuItem.DropDownItems) {
				ToolStripMenuItem menu = item as ToolStripMenuItem;
				if (menu != null) {
					bool exists = false;
					// 列が存在しているがチェックが外れていたら削除する。
					foreach (ColumnHeader header in packageListView.Columns) {
						if (header.Tag == menu.Tag) {
							exists = true;
							if (sortcolumn == header) {
								packageListView.SortColumn = sortcolumn = null;
							}
							if (menu.Checked == false) {
								packageListView.Columns.Remove(header);
							}
							break;
						}
					}
					
					// 列が存在していなく、チェックがされているなら追加する。
					if (menu.Checked && !exists) {
						ColumnHeader header = new ColumnHeader();
						header.Text = menu.Text;
						header.Tag = menu.Tag;
						packageListView.Columns.Add(header);
					}
				}
			}
			
			AppliStation.Util.NativeMethods.ColumnHeader_SetSortState(packageListView, (sortcolumn != null)? sortcolumn.Index : -1, SortOrder.None);
			
			packageListView.UpdateItems();
			
			packageListView.EndUpdate();
		}
	}

}
