﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using NaGet.SubCommands.SubTask;
using NaGet.Tasks;

namespace NaGet.SubCommands
{
	/// <summary>
	/// NaGetタスク処理のパック
	/// </summary>
	public abstract class NaGetTaskSet2 : Task
	{
		/// <summary>
		/// サブタスクのハンドラ
		/// </summary>
		public virtual event EventHandler<TaskEventArgs> SubTaskEventRaised;
		
		/// <summary>
		/// タスク処理中の質問のハンドラ
		/// </summary>
		public event NaGetTaskQueryHandler TaskQueryRaised;
		
		/// <summary>
		/// サブタスクのリスト
		/// </summary>
		protected IList<NaGetSubTask> subTasks;
		
		/// <summary>
		/// 文字列で表現した作業一覧リスト
		/// </summary>
		protected IList<string> taskSetNames;
		
		/// <summary>
		/// 現在実行中のサブタスクのインデックス
		/// </summary>
		private int currentSubTaskIndex = -1;
		
		/// <summary>
		/// 終了したか
		/// </summary>
		protected bool isDone = false;
		
		/// <summary>
		/// キャンセルされたか
		/// </summary>
		protected bool isCancelled = false;
		
		/// <summary>
		/// キャンセルが呼ばれたか
		/// </summary>
		protected bool cancelCalled = false;
		
		/// <summary>
		/// キャンセル呼び出し時のイベント。
		/// </summary>
		protected event Action<object> onCancelCalled;
		
		public NaGetTaskSet2()
		{
		}
		
		/// <summary>
		/// キャンセル処理を行う
		/// </summary>
		/// <returns>成功したか否か</returns>
		public override bool Cancel()
		{
			if (Cancelable) {
				if (! cancelCalled && ! isDone) {
					cancelCalled = true;
					if (onCancelCalled != null) {
						onCancelCalled(null);
					}
					
					if (currentSubTask != null && currentSubTask.Cancelable) {
						return currentSubTask.Cancel();
					} else {
						return true;
					}
				} else {
					return false;
				}
			} else {
				return false;
			}
		}
		
		public virtual int CurrentTaskSetIndex {
			get { return currentSubTaskIndex; }
		}
		
		public override bool Done {
			get { return isDone; }
		}
		
		public virtual bool Cancelled {
			get { return isCancelled; }
		}
		
		public override bool Running {
			get { return CurrentTaskSetIndex >= 0 && !Done; }
		}
		
		/// <summary>
		/// 文字列で表現した作業一覧リスト
		/// </summary>
		public virtual IList<string> TaskSetNames {
			get {
				return new ReadOnlyCollection<string>(taskSetNames);
			}
		}
		
		/// <summary>
		/// 現在の進捗を戻す。
		/// </summary>
		/// <param name="type">作業の状態</param>
		/// <param name="subTaskProgress">サブタスクの進捗</param>
		/// <returns>現在の進捗</returns>
		protected virtual float GetProgressPercent(TaskEventType type, float subTaskProgress)
		{
			if (CurrentTaskSetIndex >= 0) {
				if (subTaskProgress >= 0) {
					return (CurrentTaskSetIndex * 100 + subTaskProgress) / taskSetNames.Count;
				}
				switch (type) {
					case TaskEventType.STARTED:
						return 0;
					case TaskEventType.COMPLETED:
						return 100;
					case TaskEventType.COMPLETED_SUBTASK:
						return ((CurrentTaskSetIndex+1) * 100) / taskSetNames.Count;
					default:
						return (CurrentTaskSetIndex * 100) / taskSetNames.Count;
				}
			}
			
			return -1;
		}
		
		#region フラグ処理の便利メソッド
		
		/// <summary>
		/// 開始時に関するフラグの処理を行う
		/// </summary>
		protected virtual void NotifyStarted()
		{
			currentSubTaskIndex = 0;
			isDone = false;
		}
		
		/// <summary>
		/// キャンセル中断時に関するフラグの処理を行う
		/// </summary>
		protected virtual void NotifyCancelled()
		{
			isCancelled = true;
			isDone = true;
		}
		
		/// <summary>
		/// 成功終了時に関するフラグの処理を行う
		/// </summary>
		protected virtual void NotifyCompleted()
		{
			isDone = true;
		}
		
		/// <summary>
		/// サブタスク実行を次へ。
		/// </summary>
		protected virtual void NotifyGoToNextSubTask()
		{
			currentSubTaskIndex ++;
		}
		
		/// <summary>
		/// サブタスク実行をジャンプする。
		/// </summary>
		/// <param name="subTaskIndex">ジャンプ先のサブタスク番号</param>
		protected virtual void NotifyGoToSubTask(int subTaskIndex)
		{
			currentSubTaskIndex = subTaskIndex;
		}
		
		#endregion
		
		#region サブタスク実行時における便利メソッド
				
		protected virtual string currentSubTaskName {
			get { return taskSetNames[currentSubTaskIndex]; }
		}
		
		protected virtual NaGetSubTask currentSubTask {
			get { return subTasks[currentSubTaskIndex]; }
		}
		
		protected virtual bool hasMoreSubTask {
			get { return currentSubTaskIndex < taskSetNames.Count; }
		}
		
		#endregion
		
		#region TaskEvent便利メソッド
		
		protected virtual void RaiseTaskSetEvent(TaskEventType type, string message)
		{
			RaiseTaskSetEvent(type, message, GetProgressPercent(type, -1));
		}
		
		protected virtual void ReceivedErrorData(object sender, NaGet.Utils.AnyDataEventArgs<string> e)
		{
			if (! string.IsNullOrEmpty(e.Data)) {
				RaiseTaskSetEvent(TaskEventType.WARNING, e.Data);
			}
		}
		
		protected virtual void ReceivedOutputData(object sender, NaGet.Utils.AnyDataEventArgs<string> e)
		{
			if (! string.IsNullOrEmpty(e.Data)) {
				RaiseTaskSetEvent(TaskEventType.INFO, e.Data);
			}
		}
		
		protected virtual NaGetTaskQueryResult RaiseTaskSetQueryEvent(string message, NaGetTaskQueryResult selection)
		{
			if (TaskQueryRaised != null) {
				return TaskQueryRaised(this, new NaGetTaskQueryArgs(message, selection));
			}
			return NaGetTaskQueryResult.CANCELED_AUTOMATICALLY;
		}
				
		#endregion
		
		#region サブタスク初期化・登録便利メソッド
		
		protected void initSubTask()
		{
			taskSetNames = new List<string>();
			subTasks = new List<NaGetSubTask>();
		}
		
		/// <summary>
		/// サブタスクを登録する
		/// </summary>
		/// <param name="name">名称</param>
		/// <param name="task">サブタスクオブジェクト</param>
		protected void registSubTask(string name, NaGetSubTask task)
		{
			taskSetNames.Add(name);
			subTasks.Add(task);
			
			// サブタスクのイベントをSubTaskEventRaised,TaskEventRaisedに転送
			task.TaskEventRaised += new EventHandler<TaskEventArgs>(subtaskEventTransformer);
		}
		
		/// <summary>
		/// サブタスクのイベントをSubTaskEventRaised,TaskEventRaisedに転送する関数
		/// </summary>
		/// <param name="sender">送信元のサブタスク</param>
		/// <param name="e">イベントオブジェクト</param>
		private void subtaskEventTransformer(object sender, TaskEventArgs e)
		{
			if (SubTaskEventRaised != null) {
				SubTaskEventRaised(sender, e);
			}
			if (e.ProgressPercent >= 0) {
				RaiseTaskSetEvent(TaskEventType.PING, null, GetProgressPercent(TaskEventType.PING, e.ProgressPercent));
			}
		}
		
		#endregion

	}
}
