﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Globalization;
using System.Security.Principal;
using System.Security.AccessControl;
using System.Reflection;
using System.Diagnostics;
using Microsoft.Win32;

namespace NaGet
{

	/// <summary>
	/// 雑多な便利メソッドを集めたクラス
	/// </summary>
	public static class Utils
	{
		#region 汎用的なオブジェクト操作メソッド
		
		/// <summary>
		/// オブジェクトのフィールドをコピーしてクローン化する
		/// </summary>
		/// <param name="from">コピー元</param>
		/// <param name="target">コピー先。コピー元のクラスと同一か継承している型でなければならない</param>
		public static void FieldCopy<T,U>(T from, ref U target) where U : T
		{
			foreach(FieldInfo member in typeof(T).GetFields()) {
				try {
					member.SetValue(target, member.GetValue(from));
				} catch (FieldAccessException) {} // アクセス不能は無視
			}
		}
		
		#endregion
		
		#region リスト関連関数
		
		/// <summary>
		/// イテレータを結合して、返す
		/// </summary>
		/// <param name="enus">元となる複数のイテレータ</param>
		/// <returns>結合されたイテレータ</returns>
		public static IEnumerable<T> MergeEnumerable<T>(params IEnumerable<T>[] enus)
		{
			foreach (IEnumerable<T> enu in enus) {
				if (enu == null) continue;
				
				foreach (T elem in enu) {
					yield return elem;
				}
			}
		}

		/// <summary>
		/// イテレータを結合して、返す
		/// </summary>
		/// <param name="enus">元となる複数のイテレータ</param>
		/// <returns>結合されたイテレータ</returns>
		public static IEnumerable<T> MergeEnumerable<T>(params IEnumerator<T>[] enus)
		{
			foreach (IEnumerator<T> enu in enus) {
				if (enu == null) continue;
				
				try {
					while (enu.MoveNext()) {
						yield return enu.Current;
					}
				} finally {
					enu.Dispose();
				}
			}
		}
		
		/// <summary>
		/// イテレータを結合して、Listとして返す
		/// </summary>
		/// <param name="enus">元となる複数のイテレータ</param>
		/// <returns>結合されたイテレータ</returns>
		public static List<T> MergeList<T>(params IEnumerable<T>[] enus)
		{
			List<T> list = new List<T>();
			
			foreach (IEnumerable<T> enu in enus) {
				if (enu == null) continue;
			
				list.AddRange(enu);
			}
			return list;
		}
		
		/// <summary>
		/// イテレータを配列に効率的に変換する
		/// </summary>
		/// <remarks>与えられる型が具体的にわかっているならば、それに特化した手続きをとる方が好ましい</remarks>
		/// <param name="enu">元となるイテレータ</param>
		/// <returns>変換された配列</returns>
		public static T[] IEnumerable2Array<T>(IEnumerable<T> enu) {
			T[] retval = enu as T[];
			
			if (retval == null) {
				List<T> list = enu as List<T>;
				if (list == null) {
					list = new List<T>(enu);
				}
				retval = list.ToArray();
			}
			
			return retval;
		}
		
		/// <summary>
		/// リストに対して指定した2つの要素の位置を入れ替える
		/// </summary>
		/// <param name="list">操作対象のリスト</param>
		/// <param name="indexA">位置</param>
		/// <param name="indexB">位置</param>
		public static void ListSwap(System.Collections.IList list, int indexA, int indexB)
		{
			if ((indexA < 0) || (list.Count <= indexA) || (indexB < 0) || (list.Count <= indexB)) {
				throw new IndexOutOfRangeException();
			} else if (indexA != indexB) {
				object temp = list[indexA];
				list[indexA] = list[indexB];
				list[indexB] = temp;
			}
		}
		
		#endregion
				
		#region ファイル情報関連ユーテイリティ関数
		
		/// <summary>
		/// パス変数に指定のフォルダーを追加する
		/// </summary>
		/// <param name="dir">追加するフォルダー</param>
		public static void AddDirectoryToPath(string dir)
		{
			string path = Environment.GetEnvironmentVariable("PATH");
			
			if (path.IndexOf(dir) < 0) {
				path = dir + Path.PathSeparator + path;
				Environment.SetEnvironmentVariable("PATH", path);
			}
		}
		
		/// <summary>
		/// バイト単位で表現された容量を接尾語を活用して適切な文字列に変換
		/// </summary>
		/// <param name="bytes">バイト単位の容量</param>
		/// <returns>読みやすい形に直された容量文字列</returns>
        public static string FormatSize(double bytes)
        {
        	string[] units = new string[] {"B", "KB", "MB", "GB", "TB"};
        	
        	double size = bytes;
        	int i;
        	for (i = 0; size >= 1024 && i < units.Length-1 ; i++) {
        		size /= 1024.0;
        	}
        	
        	return string.Format("{0:F2}{1}", size, units[i]);
        }
        
        public static string FormatSize(long bytes)
        {
        	return FormatSize((double) bytes);
        }
		
		/// <summary>
		/// URLからそのファイル名を生成する
		/// </summary>
		/// <param name="url">対象のurl</param>
		public static string Url2filename(Uri url)
		{
			string filename = Path.GetFileName(System.Web.HttpUtility.UrlDecode(url.AbsolutePath, Encoding.UTF8));
			
			if (string.IsNullOrEmpty(filename)) {
				filename = Path.GetFileName(System.Web.HttpUtility.UrlDecode(url.ToString(), Encoding.UTF8));
			}
			
			int pos;
			if ((pos = filename.IndexOfAny(Path.GetInvalidFileNameChars())) >= 0) {
				// 不正な文字が含まれているならば、それ以降を削除
				filename = filename.Substring(0, pos);
			}
			
			// そうしてしまったら文字の内容がまったくなくなってしまったら、現在時刻から取得
			if (filename.Length == 0) {
				filename = string.Format("tmp_{0}", DateTime.Now.Ticks);
			}
			
			return filename;
			//return UrlDecode(Path.GetFileName(url), Encoding.UTF8);
		}
		
		/// <summary>
		/// 再帰的にファイルの属性を指定します。強制的にフォルダーの再帰削除の前に読み込み専用属性を消すのに使います。
		/// </summary>
		/// <param name="path">設定するフォルダー</param>
		/// <param name="attr">設定する属性値</param>
		public static void SetAttributeRecursive(string path, FileAttributes attr)
		{
			// 自分自身の属性を変更
			File.SetAttributes(path, attr);
			
			// 子ファイルの属性変更
			foreach (string file in Directory.GetFiles(path)) {
				File.SetAttributes(file, attr);
			}
			
			// 子フォルダーを再帰的に属性変更
			foreach (string file in Directory.GetDirectories(path)) {
				SetAttributeRecursive(file, attr);
			}
		}
		
		/// <summary>
		/// 再帰的にファイルのアクセス権限（ユーザ権限など）を設定します
		/// </summary>
		/// <param name="path">設定するフォルダー</param>
		/// <param name="filesec">変更先権限</param>
		public static void SetAccessControlRecursive(string path, FileSecurity filesec)
		{
			// 自分自身の権限を変更
			File.SetAccessControl(path, filesec);
			
			// 子ファイルの権限を変更
			foreach (string file in Directory.GetFiles(path)) {
				File.SetAccessControl(file, filesec);
			}
			
			// 子フォルダーを再帰的に権限変更
			foreach (string file in Directory.GetDirectories(path)) {
				SetAccessControlRecursive(file, filesec);
			}
		}
		
		/// <summary>
		/// ファイルまたはフォルダーの容量を算出して返す
		/// </summary>
		/// <param name="path">
		/// 対象ファイル及びフォルダーのパス
		/// </param>
		/// <returns>
		/// 計算された容量(バイト単位)
		/// </returns>
		public static ulong GetFileSize(string path)
		{
			return ((File.GetAttributes(path) & FileAttributes.Directory) != 0)?
				GetDirectoryFileSize(new DirectoryInfo(path)) : ((ulong) (new FileInfo(path)).Length);
		}
		
		/// <summary>
		/// フォルダーの容量を算出して返す
		/// </summary>
		/// <param name="dirInfo">
		/// 対象フォルダー
		/// </param>
		/// <returns>
		/// 計算された容量(バイト単位)
		/// </returns>
		public static ulong GetDirectoryFileSize(DirectoryInfo dirInfo)
		{
			ulong size = 0;
			foreach (FileInfo child in dirInfo.GetFiles("*", SearchOption.AllDirectories)) {
				size += (ulong) child.Length;
			}
			return size;
		}
				
		/// <summary>
		/// ワイルドカードを展開したファイルパス文字列を作り出す。
		/// 戻り値のそれぞれの文字列はフルパスとなる。
		/// </summary>
		/// <param name="baseDir">ベース(基点)のディレクトリ</param>
		/// <param name="pattern">ワイルドカードパターン</param>
		/// <returns>展開したファイルパス</returns>
		public static string[] ExtendWildcardFile(string baseDir, string pattern)
		{
			if (pattern.IndexOfAny(new char[]{'*','?'}) < 0) {
				return new string[]{Path.Combine(baseDir, pattern)}; // ワイルドカードがなければそのまま返す
			}

			string[] pathArray = pattern.Split(Path.DirectorySeparatorChar);
			List<string> extended = new List<string>();
			try {
				if (pathArray.Length == 1) {
					extended.AddRange(Directory.GetFiles(baseDir, pathArray[0], SearchOption.TopDirectoryOnly));
					extended.AddRange(Directory.GetDirectories(baseDir, pathArray[0], SearchOption.TopDirectoryOnly));
				} else { // pathArray.Length > 1
					string subPattern = string.Join(Path.DirectorySeparatorChar.ToString(), pathArray, 1, pathArray.Length-1);
					
					foreach (string subDir in Directory.GetDirectories(baseDir, pathArray[0], SearchOption.TopDirectoryOnly)) {
						// 再帰的に追加してゆく
						extended.AddRange(ExtendWildcardFile(subDir, subPattern));
					}
				}
			} catch (UnauthorizedAccessException) {
			}
			
			// 存在しないパスは消去する
			extended.RemoveAll(
				delegate(string path) {
					return ! File.Exists(path);
				}
			);
			
			return extended.ToArray();
		}
		
		/// <summary>
		/// パスをパス区切り文字列ごとに分割した配列を返す
		/// </summary>
		/// <param name="path">パス文字列。相対・絶対は区別しない</param>
		/// <returns>フォルダー名ごとに分けられた文字列配列</returns>
		private static string[] splitPath(string path)
		{
			return path.Split(new char[]{Path.DirectorySeparatorChar, Path.AltDirectorySeparatorChar});
		}
		
		/// <summary>
		/// パスがフォルダーのとき、最後がパスセパレータで終了するようにする。
		/// </summary>
		/// <param name="path">パス</param>
		/// <returns>処理されたパス</returns>
		private static string fixLastPathCharForDirectory(string path)
		{
			string fixedPath = path;
			if (Directory.Exists(path) && path[path.Length-1] != Path.DirectorySeparatorChar) {
				fixedPath += Path.DirectorySeparatorChar;
			}
			return fixedPath;
		}
		
		/// <summary>
		/// 絶対パスを相対パスに変換して返します。
		/// </summary>
		/// <param name="baseDirectoryPath">相対パスの基準のフォルダー</param>
		/// <param name="absPath">絶対パス</param>
		/// <returns><code>absPath</code>の絶対パス表現</returns>
		public static string GetRelativePath(string baseDirectoryPath, string absPath)
		{
			Uri baseuri	= new Uri(fixLastPathCharForDirectory(baseDirectoryPath));
			Uri absuri	= new Uri(fixLastPathCharForDirectory(absPath));
			
			string relative = baseuri.MakeRelativeUri(absuri).ToString();
			relative = System.Web.HttpUtility.UrlDecode(relative);
			relative = relative.Replace('/', Path.DirectorySeparatorChar);
			
			return relative;
		}
		
		/// <summary>
		/// 相対パスに含まれている".."などを消去する
		/// </summary>
		/// <param name="aPath"></param>
		/// <returns></returns>
		public static string GetDotsRemovedPath(string aPath)
		{
			string[] folders = splitPath(aPath);
			List<string> newFolders = new List<string>();
			
			foreach (string fol in folders) {
				if (fol == ".") {
					// 無視
				} else if (fol == "..") {
					// 一つ前のフォルダーを消す
					newFolders.RemoveAt(newFolders.Count-1);
				} else {
					newFolders.Add(fol);
				}
			}
			
			return string.Join(Path.DirectorySeparatorChar.ToString(), newFolders.ToArray());
		}
		
		#endregion
		
		#region シリアル化関連
		
		
		/// <summary>
		/// XMLでシリアル化したオブジェクトのXMLファイルを読み込み、デシリアル化したオブジェクトを取得する
		/// </summary>
		/// <param name="path">XMLファイルのパス</param>
		/// <param name="sr">シリアライザ</param>
		/// <returns>デシリアル化されたオブジェクト</returns>
		public static object GetDeserializedObject(string path, System.Xml.Serialization.XmlSerializer sr)
		{
			object retVal = null;
			using (FileStream fs = new FileStream(path, FileMode.Open, FileAccess.Read)) {
				using (System.Xml.XmlReader xreader = System.Xml.XmlReader.Create(fs)) {
					retVal = sr.Deserialize(xreader);
				}
			}
			return retVal;
		}
		
		/// <summary>
		/// XMLでシリアル化したオブジェクトのXMLファイルを読み込み、デシリアル化したオブジェクトを取得する
		/// </summary>
		/// <param name="path">XMLファイルのパス</param>
		/// <returns>デシリアル化されたオブジェクト</returns>
		public static T GetDeserializedObject<T>(string path)
		{
			T retVal = default(T);
			System.Xml.Serialization.XmlSerializer sr = new System.Xml.Serialization.XmlSerializer(typeof(T));
			retVal = (T) GetDeserializedObject(path, sr);
			return retVal;
		}
		
		/// <summary>
		/// オブジェクトをXMLでシリアル化してファイルに書き込む
		/// </summary>
		/// <param name="path">XMLファイルのパス</param>
		/// <param name="obj">シリアル化する対象のオブジェクト</param>
		public static void PutSerializeObject<T>(string path, T obj)
		{
			using (FileStream fs = new FileStream(path, FileMode.Create, FileAccess.Write)) {
				System.Xml.Serialization.XmlSerializer sr = new System.Xml.Serialization.XmlSerializer(typeof(T));
				sr.Serialize(fs, obj);
			}
		}
		
		#endregion
		
		#region 権限関連関数群
		
		/// <summary>
		/// 現在のユーザがAdministrators権限を持っているか否かを返す。
		/// </summary>
		/// <remarks>UAC有効時には権限昇格後になってtrueを返すようになります</remarks>
		public static bool IsAdministrators()
		{
			// 現在の Windows ユーザーを現在のスレッドのプリンシパルに反映する
			AppDomain.CurrentDomain.SetPrincipalPolicy(PrincipalPolicy.WindowsPrincipal );
			IPrincipal prin = System.Threading.Thread.CurrentPrincipal;
		   	return prin.IsInRole(@"BUILTIN\Administrators");
		}
		
		/// <summary>
		/// 現在のPCがUACが有効になっているか否かを返す。
		/// レジストリのHKLM\SOFTWARE¥Microsoft¥Windows¥CurrentVersion¥Policies¥System\EnableLUAの値を見る。
		/// </summary>
		/// <returns>UACが有効ならばtrue</returns>
		public static bool IsUACEnabled()
		{
			try {
				using(RegistryKey key = Registry.LocalMachine.CreateSubKey(@"SOFTWARE¥Microsoft¥Windows¥CurrentVersion¥Policies¥System")) {
					return ((int) key.GetValue("EnableLUA", 0)) == 1;
				}
			} catch (Exception) {
				return false;
			}
		}
		
		#endregion

		#region プロセス関連便利メソッド群
		
		/// <summary>
		/// プロセスに出力をリダイレクトした上で実行
		/// </summary>
		/// <param name="procInfo">プロセス起動情報</param>
		/// <param name="outputReceived">標準出力用リスナ(null可)</param>
		/// <param name="errorReceived">エラー出力用リスナ(null可)</param>
		/// <returns>実行プロセス</returns>
		public static Process ProcessStartWithOutputCapture(ProcessStartInfo procInfo,
		                                  DataReceivedEventHandler outputReceived,
		                                  DataReceivedEventHandler errorReceived)
		{
			if (outputReceived != null) {
				procInfo.RedirectStandardOutput = true;
			}
			if (errorReceived != null) {
				procInfo.RedirectStandardError = true;
			}
			procInfo.UseShellExecute = false;
			
			Process hProcess = Process.Start(procInfo);
			if (outputReceived != null) {
				hProcess.OutputDataReceived += outputReceived;
				hProcess.BeginOutputReadLine();
			}
			if (errorReceived != null) {
				hProcess.ErrorDataReceived += errorReceived;	
				hProcess.BeginErrorReadLine();
			}
			
			return hProcess;
		}
		
		
		/// <summary>
		/// プロセスに出力をリダイレクトした上で実行
		/// </summary>
		/// <param name="procInfo">プロセス起動情報</param>
		/// <param name="outputReceived">標準出力用リスナ(null可)</param>
		/// <param name="errorReceived">エラー出力用リスナ(null可)</param>
		/// <returns>実行プロセス</returns>
		public static Process ProcessStartWithOutputCapture(ProcessStartInfo procInfo,
		                                  EventHandler<AnyDataEventArgs<string>> outputReceived,
		                                  EventHandler<AnyDataEventArgs<string>> errorReceived)
		{
			return ProcessStartWithOutputCapture(procInfo,
			                                     ConvertToDataReceivedEventHandler(outputReceived),
			                                     ConvertToDataReceivedEventHandler(errorReceived));
		}
		
		public static DataReceivedEventHandler ConvertToDataReceivedEventHandler(EventHandler<AnyDataEventArgs<string>> handler)
		{
			if (handler == null) return null;
			return delegate (object sender, DataReceivedEventArgs e) {
				AnyDataEventArgs<string> args = new AnyDataEventArgs<string>(e.Data);
				handler.Invoke(sender, args);
			};
		}
		
		#endregion

		#region イベント情報
		
		/// <summary>
		/// 任意データのイベント情報を表現するクラス
		/// </summary>
		public class AnyDataEventArgs<T> : EventArgs
		{
			/// <summary>
			/// データ
			/// </summary>
			T data;
			
			public AnyDataEventArgs(T data)
			{
				this.data = data;
			}
			
			/// <summary>
			/// データを返す
			/// </summary>
			public T Data {
				get { return data; }
			}
		}

		#endregion
	}
}
