﻿//-------------------------------------------------------------------------------------------------
// File : asdxSse.inl
// Desc : Streaming SIMD Extention Wrapper Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

#if ASDX_IS_SSE

namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// Simd class
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      生成処理を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Create( f32 x, f32 y, f32 z, f32 w )
{ return _mm_set_ps( w, z, y, x ); }

//-------------------------------------------------------------------------------------------------
//      X成分を設定します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Simd::SetX( b128& value, f32 x )
{ value = _mm_insert_ps( value, _mm_set_ss(x), 0 << 4 ); }

//-------------------------------------------------------------------------------------------------
//      Y成分を設定します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Simd::SetY( b128& value, f32 y )
{ value = _mm_insert_ps( value, _mm_set_ss(y), 1 << 4 ); }

//-------------------------------------------------------------------------------------------------
//      Z成分を設定します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Simd::SetZ( b128& value, f32 z )
{ value = _mm_insert_ps( value, _mm_set_ss(z), 2 << 4 ); }

//-------------------------------------------------------------------------------------------------
//      W成分を設定します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Simd::SetW( b128& value, f32 w )
{ value = _mm_insert_ps( value, _mm_set_ss(w), 3 << 4 ); }

//-------------------------------------------------------------------------------------------------
//      X成分を取得します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Simd::GetX( const b128& value )
{ return _mm_cvtss_f32( value ); }

//-------------------------------------------------------------------------------------------------
//      Y成分を取得します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Simd::GetY( const b128& value )
{
    f32 ret;
    _MM_EXTRACT_FLOAT(ret, value, 1);
    return ret;
}

//-------------------------------------------------------------------------------------------------
//      Z成分を取得します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Simd::GetZ( const b128& value )
{
    f32 ret;
    _MM_EXTRACT_FLOAT(ret, value, 2);
    return ret;
}

//-------------------------------------------------------------------------------------------------
//      W成分を取得します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Simd::GetW( const b128& value )
{
    f32 ret;
    _MM_EXTRACT_FLOAT(ret, value, 3);
    return ret;
}

//-------------------------------------------------------------------------------------------------
//      加算計算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Add( const b128& a, const b128& b )
{ return _mm_add_ps( a, b ); }

//-------------------------------------------------------------------------------------------------
//      減算計算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Sub( const b128& a, const b128& b )
{ return _mm_sub_ps( a, b ); }

//-------------------------------------------------------------------------------------------------
//      乗算計算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Mul( const b128& a, const b128& b )
{ return _mm_mul_ps( a, b ); }

//-------------------------------------------------------------------------------------------------
//      除算計算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Div( const b128& a, const b128& b )
{ return _mm_div_ps( a, b ); }

//-------------------------------------------------------------------------------------------------
//      積和計算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Mad( const b128& a, const b128& b, const b128& c )
{
    b128 tmp = _mm_mul_ps( a, b );
    return _mm_add_ps( tmp, c );
}

//-------------------------------------------------------------------------------------------------
//      平方根を計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Sqrt( const b128& value )
{ return _mm_sqrt_ps( value ); }

//-------------------------------------------------------------------------------------------------
//      逆数を計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Rcp( const b128& value )
{ return _mm_rcp_ps( value ); }

//-------------------------------------------------------------------------------------------------
//      平方根の逆数を計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Rsqrt( const b128& value )
{ return _mm_rsqrt_ps( value ); }

//-------------------------------------------------------------------------------------------------
//      負の値を計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Negate( const b128& value )
{
    b128 zero = _mm_setzero_ps();
    return _mm_sub_ps(zero, value);
}

//-------------------------------------------------------------------------------------------------
//      絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Abs( const b128& value )
{
    b128 mask = _mm_set1_ps( -0.0f );
    return _mm_andnot_ps( mask, value );
}

//-------------------------------------------------------------------------------------------------
//      最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Min( const b128& a, const b128& b )
{ return _mm_min_ps( a, b ); }

//-------------------------------------------------------------------------------------------------
//      最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Max( const b128& a, const b128& b )
{ return _mm_max_ps( a, b ); }

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Dp3( const b128& a, const b128& b )
{ return _mm_dp_ps( a, b,  0x77 ); }

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
b128 Simd::Dp4( const b128& a, const b128& b )
{ return _mm_dp_ps( a, b,  0xff ); }

} // namespace asdx

#endif//ASDX_IS_SSE
