﻿//-------------------------------------------------------------------------------------------------
// File : Model.h
// Desc : Model Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxMath.h>
#include <asdxResTexture.h>
#include <asdxDevice.h>
#include <asdxDeviceContext.h>
#include <asdxVertexBuffer.h>
#include <asdxIndexBuffer.h>
#include <asdxConstantBuffer.h>
#include <asdxResMesh.h>
#include <vector>
#include <d3d12.h>


///////////////////////////////////////////////////////////////////////////////////////////////////
// Material structure
///////////////////////////////////////////////////////////////////////////////////////////////////
ASDX_ALIGN(256)
struct Material
{
    asdx::Vector3   Diffuse;        //!< 拡散反射.
    f32             Alpha;          //!< 透過.
    asdx::Vector3   Specular;       //!< 鏡面反射.
    f32             Power;          //!< 鏡面反射強度.
    asdx::Vector3   Emissive;       //!< 自己照明.
    u32             TextureId;      //!< テクスチャ番号.
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// SkinningVertex structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct SkinningVertex
{
    asdx::Vector3   Position;       //!< 位置座標.
    asdx::Vector3   Normal;         //!< 法線ベクトル.
    asdx::Vector2   TexCoord;       //!< テクスチャ座標.
    asdx::uint2     BoneIndices;    //!< ボーン番号.
    asdx::Vector2   BoneWeights;    //!< ボーンの重み.
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// Model class
///////////////////////////////////////////////////////////////////////////////////////////////////
class Model : private NonCopyable
{
    //=============================================================================================
    // public list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    Model();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~Model();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化処理を行います.
    //!
    //! @param[in]      device          デバイスです.
    //! @param[in]      meshFile        メッシュファイル名.
    //! @param[in]      materialFile    マテリアルファイル名.
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool Init( 
        asdx::Device& device,
        asdx::DeviceContext& context, 
        const char16* meshFile,
        const char16* materialFile);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void Term();

    //---------------------------------------------------------------------------------------------
    //! @brief      描画コマンドを発行します.
    //---------------------------------------------------------------------------------------------
    void DrawCmd( ID3D12GraphicsCommandList* pCmd );

    //---------------------------------------------------------------------------------------------
    //! @brief      ボーンを取得します.
    //---------------------------------------------------------------------------------------------
    asdx::ResBone* GetBones();

    u32 GetBoneCount() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    std::vector<SkinningVertex>     m_Vertices;     //!< 頂点データ.
    std::vector<u32>                m_Indices;      //!< インデックスデータ.
    std::vector<asdx::ResBone>      m_Bones;        //!< ボーンです.
    std::vector<asdx::ResSubset>    m_Subsets;      //!< サブセットです.
    std::vector<asdx::ResTexture>   m_ResTextures;  //!< テクスチャ.
    std::vector<Material>           m_Materials;    //!< マテリアルです.
    u8*                             m_pHeadCB;      //!< 定数バッファの戦闘ポインタ.

    asdx::VertexBuffer              m_VB;
    asdx::IndexBuffer               m_IB;
    asdx::ConstantBuffer            m_CB;
    std::vector<asdx::DescHandle>   m_CBV;
    std::vector<asdx::DescHandle>   m_SRV;
    std::vector<asdx::RefPtr<ID3D12Resource>> m_Textures;
    asdx::ResTexture                m_DummyResTexture;
    asdx::RefPtr<ID3D12Resource>    m_DummyTexture;
    asdx::DescHandle                m_DummySRV;

    bool CreateTexture(
        asdx::Device& device,
        const asdx::ResTexture& texture,
        ID3D12Resource** ppResource,
        asdx::DescHandle* pHandle ) const;
};
