﻿//-------------------------------------------------------------------------------------------------
// File : asdxHid.h
// Desc : Human Interface Device Module.
// Copyright(c) Project Asura. All right rerseved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>


#ifdef ASDX_AUTO_LINK
//-------------------------------------------------------------------------------------------------
// Linker
//-------------------------------------------------------------------------------------------------
#pragma comment( lib, "XInput9_1_0.lib" )
#endif//ASDX_AUTO_LINK


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// PAD_BUTTON enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum PAD_BUTTON
{
    PAD_UP          = 0x00001L,     //!< 方向パッド上
    PAD_DOWN        = 0x00002L,     //!< 方向パッド下
    PAD_LEFT        = 0x00004L,     //!< 方向パッド左
    PAD_RIGHT       = 0x00008L,     //!< 方向パッド右
    PAD_START       = 0x00010L,     //!< スタートボタン.
    PAD_BACK        = 0x00020L,     //!< バックボタン.
    PAD_THUMB_L     = 0x00040L,     //!< 左サムスティックボタン.
    PAD_THUMB_R     = 0x00080L,     //!< 右サムスティックボタン.
    PAD_SHOULDER_L  = 0x00100L,     //!< LBボタン.
    PAD_SHOULDER_R  = 0x00200L,     //!< RBボタン.
    PAD_A           = 0x01000L,     //!< Aボタン.
    PAD_B           = 0x02000L,     //!< Bボタン.
    PAD_X           = 0x04000L,     //!< Xボタン.
    PAD_Y           = 0x08000L,     //!< Yボタン.
    PAD_TRIGGER_L   = 0x10000L,     //!< 左トリガー.
    PAD_TRIGGER_R   = 0x20000L,     //!< 右トリガー.
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// KEY_CODE enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum KEY_CODE
{
    KEY_RETURN = 0x80,      //!< リターンキーです.
    KEY_TAB,                //!< Tabキーです.
    KEY_ESC,                //!< Escapeキーです.
    KEY_BACK,               //!< BackSpaceキーです.
    KEY_SHIFT,              //!< Shiftキーです.
    KEY_CONTROL,            //!< Controlキーです.
    KEY_ALT,                //!< Altキーです.
    KEY_F1,                 //!< F1キーです.
    KEY_F2,                 //!< F2キーです.
    KEY_F3,                 //!< F3キーです.
    KEY_F4,                 //!< F4キーです.
    KEY_F5,                 //!< F5キーです.
    KEY_F6,                 //!< F6キーです.
    KEY_F7,                 //!< F7キーです.
    KEY_F8,                 //!< F8キーです.
    KEY_F9,                 //!< F9キーです.
    KEY_F10,                //!< F10キーです.
    KEY_F11,                //!< F11キーです.
    KEY_F12,                //!< F12キーです.
    KEY_UP,                 //!< ↑キーです.
    KEY_DOWN,               //!< ↓キーです.
    KEY_LEFT,               //!< ←キーです.
    KEY_RIGHT,              //!< →キーです.
    KEY_NUM0,               //!< Num0キーです.
    KEY_NUM1,               //!< Num1キーです.
    KEY_NUM2,               //!< Num2キーです.
    KEY_NUM3,               //!< Num3キーです.
    KEY_NUM4,               //!< Num4キーです.
    KEY_NUM5,               //!< Num5キーです.
    KEY_NUM6,               //!< Num6キーです.
    KEY_NUM7,               //!< Num7キーです.
    KEY_NUM8,               //!< Num8キーです.
    KEY_NUM9,               //!< Num9キーです.
    KEY_INSERT,             //!< Insertキーです.
    KEY_DELETE,             //!< Deleteキーです.
    KEY_HOME,               //!< Homeキーです.
    KEY_END,                //!< Endキーです.
    KEY_PAGE_UP,            //!< PageUpキーです.
    KEY_PAGE_DOWN,          //!< PageDownキーです.
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// MOUSE_BUTTON enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum MOUSE_BUTTON
{
    MOUSE_BUTTON_L = 0x0,       //!< 左ボタンです.
    MOUSE_BUTTON_R,             //!< 右ボタンです.
    MOUSE_BUTTON_M,             //!< 中ボタンです.
    MOUSE_BUTTON_X1,            //!< サイドボタン1です.
    MOUSE_BUTTON_X2,            //!< サイドボタン2です.
    NUM_MOUSE_BUTTON,
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// GamePad class
///////////////////////////////////////////////////////////////////////////////////////////////////
class GamePad : private NonCopyable
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    GamePad();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~GamePad();

    //---------------------------------------------------------------------------------------------
    //! @brief      プレイヤーインデックスを設定します.
    //!
    //! @param [in]     index       設定するプレイヤーインデックス.
    //---------------------------------------------------------------------------------------------
    void SetPlayerIndex( const u32 index );

    //---------------------------------------------------------------------------------------------
    //! @brief      プレイヤーインデックスを取得します.
    //!
    //! @return     設定されているプレイヤーインデックスを返却します.
    //---------------------------------------------------------------------------------------------
    u32 GetPlayerIndex() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      コントローラーが接続されているかチェックします.
    //!
    //! @retval true    接続されています.
    //! @retval false   接続されていません.
    //---------------------------------------------------------------------------------------------
    bool IsConnected() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      コントローラーの状態を更新します.
    //---------------------------------------------------------------------------------------------
    void UpdateState();

    //---------------------------------------------------------------------------------------------
    //! @brief      バイブレーションさせます.
    //!
    //! @param[in]      leftMoter       左のモーターの振動量(有効値:0.0 ～ 1.0).
    //! @param[in]      rightMoter      右のモーターの振動量(有効値:0.0 ～ 1.0).
    //---------------------------------------------------------------------------------------------
    void Vibrate( f32 leftMotor, f32 rightMotor );

    //---------------------------------------------------------------------------------------------
    //! @brief      ボタンが押されたかチェックします.
    //!             フラグ切り替え向け用途として使用してください.
    //!
    //! @param[in]      type        チェックするボタン.
    //! @retval true    ボタンが押されました.
    //! @retval false   ボタンは押されません.
    //---------------------------------------------------------------------------------------------
    bool IsDown( PAD_BUTTON type ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ボタンが押されっぱなしかどうかチェックします.
    //!
    //! @param[in]      type        チェックするボタン.
    //! @return     ボタンが押されっぱなしの状態のときはtrueを返却.
    //---------------------------------------------------------------------------------------------
    bool IsPush( PAD_BUTTON type ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      左サムスティックのX成分を取得します.
    //!
    //! @return     左サムスティックのX成分を返却します.
    //---------------------------------------------------------------------------------------------
    s16 GetThumbLX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      左サムスティックのY成分を取得します.
    //!
    //! @return     左サムスティックのY成分を返却します.
    //---------------------------------------------------------------------------------------------
    s16 GetThumbLY() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      右サムスティックのX成分を取得します.
    //!
    //! @return     右サムスティックのX成分を返却します.
    //---------------------------------------------------------------------------------------------
    s16 GetThumbRX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      右サムスティックのY成分を取得します.
    //!
    //! @return     右サムスティックのY成分を返却します.
    //---------------------------------------------------------------------------------------------
    s16 GetThumbRY() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      正規化された左サムスティックのX成分を取得します.
    //!
    //! @return     正規化された左サムスティックのX成分を返却します.
    //---------------------------------------------------------------------------------------------
    f32  GetNormalizedThumbLX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      正規化された左サムスティックのY成分を取得します.
    //!
    //! @return     正規化された左サムスティックのY成分を返却します.
    //---------------------------------------------------------------------------------------------
    f32  GetNormalizedThumbLY() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      正規化された右サムスティックのX成分を取得します.
    //!
    //! @return     正規化された右サムスティックのX成分を返却します.
    //---------------------------------------------------------------------------------------------
    f32 GetNormalizedThumbRX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      正規化された右サムスティックのY成分を取得します.
    //!
    //! @return     正規化された右サムスティックのy成分を返却します.
    //---------------------------------------------------------------------------------------------
    f32 GetNormalizedThumbRY() const;

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      ボタンの値をリセットします.
    //---------------------------------------------------------------------------------------------
    void Reset();

    //=============================================================================================
    // protected methods.
    //=============================================================================================
    /* NOTHING */

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    s32                 m_PlayerIndex;          //!< プレーヤーインデックスです.
    bool                m_IsConnected;          //!< 接続中かどうか.

    u32                 m_Buttons;              //!< ボタンの状態.
    u8                  m_TriggerL;             //!< 左トリガーの状態.
    u8                  m_TriggerR;             //!< 右トリガーの状態.

    s16                 m_ThumbLX;              //!< 左サムスティックX成分.
    s16                 m_ThumbLY;              //!< 左サムスティックY成分.
    s16                 m_ThumbRX;              //!< 右サムスティックX成分.
    s16                 m_ThumbRY;              //!< 右サムスティックY成分.

    f32                 m_NormThumbLX;          //!< 正規化された左サムスティックX成分.
    f32                 m_NormThumbLY;          //!< 正規化された左サムスティックY成分.
    f32                 m_NormThumbRX;          //!< 正規化された右サムスティックX成分.
    f32                 m_NormThumbRY;          //!< 正規化された右サムスティックY成分.

    u32                 m_PressedButtons;       //!< 押されているボタン.
    u32                 m_LastButtons;          //!< 1つ前のボタンの状態.

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// Keyboard class
///////////////////////////////////////////////////////////////////////////////////////////////////
class Keyboard : private NonCopyable
{
    //=============================================================================================
    // friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    static const u32 MAX_KEYS = 256;        //!< キーの最大数です.

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    Keyboard();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~Keyboard();

    //---------------------------------------------------------------------------------------------
    //! @brief      キーの状態を更新します.
    //---------------------------------------------------------------------------------------------
    void UpdateState();

    //---------------------------------------------------------------------------------------------
    //! @brief      キーが押されっぱなしかどうかチェックします.
    //!
    //! @param[in]      keyCode     チェックするキーコード.
    //! @return     キーが押された場合はtrueを返します.
    //! @memo       現在のフレームでキーが押されている場合にtrueを返します.
    //---------------------------------------------------------------------------------------------
    bool IsPush( const u32 keyCode ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      キーが押されたかどうかチェックします.
    //!
    //! @param[in]      keyCode     チェックするキーコード.
    //! @return     キーが押されている場合はtrueを返します.
    //! @memo       前のフレームのキー状態と現在のフレームのキー状態が異なる場合にtrueを返します.
    //!             フラグの切り替え用途などに使用します.
    //---------------------------------------------------------------------------------------------
    bool IsDown( const u32 keyCode ) const;

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // protected methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      キーコードを変換します.
    //!
    //! @return     キーコードを変換した結果を返却します.
    //---------------------------------------------------------------------------------------------
    u32 ConvertKey( const u32 keyCode ) const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    bool    m_Keys[ 2 ][ MAX_KEYS ];        //!< キーの押下状態です.
    u32     m_Index;                        //!< バッファインデックスです.

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// Mouse class
///////////////////////////////////////////////////////////////////////////////////////////////////
class Mouse : NonCopyable
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    Mouse();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~Mouse();

    //---------------------------------------------------------------------------------------------
    //! @brief      マウスの状態を更新します.
    //---------------------------------------------------------------------------------------------
    void UpdateState();

    //---------------------------------------------------------------------------------------------
    //! @brief      マウスの状態をリセットします.
    //---------------------------------------------------------------------------------------------
    void ResetState();

    //---------------------------------------------------------------------------------------------
    //! @brief      カーソルのX座標を取得します.
    //!
    //! @return     カーソルのX座標を返却します.
    //---------------------------------------------------------------------------------------------
    s32 GetCursorX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      カーソルのY座標を取得します.
    //!
    //! @return     カーソルのY座標を返却します.
    //---------------------------------------------------------------------------------------------
    s32 GetCursorY() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      前のカーソルのX座標を取得します.
    //!
    //! @return     前のカーソルのX座標を返却します.
    //---------------------------------------------------------------------------------------------
    s32 GetPrevCursorX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      前のカーソルのY座標を取得します.
    //!
    //! @return     前のカーソルのY座標を返却します.
    //---------------------------------------------------------------------------------------------
    s32 GetPrevCursorY() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      現在のカーソルと前のカーソルのX座標の差分を取得します.
    //!
    //! @return     現在のカーソルと前のカーソルのX座標の差分を返却します.
    //---------------------------------------------------------------------------------------------
    s32 GetCursorDiffX() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      現在のカーソルと前のカーソルのY座標の差分を取得します.
    //!
    //! @return     現在のカーソルと前のカーソルのY座標の差分を返却します.
    //---------------------------------------------------------------------------------------------
    s32 GetCursorDiffY() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ボタンが押されっぱなしかどうかチェックします.
    //!
    //! @param[in]      button      ボタン.
    //! @return     ボタンが押された場合はtrueを返します.そうでない場合はfalseを返却します.
    //! @memo       現在のフレームでボタンが押されている場合にtrueを返します.
    //---------------------------------------------------------------------------------------------
    bool IsPush( const u32 button ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ボタンが押されたかどうかチェックします.
    //!
    //! @param[in]      button      ボタン.
    //! @return     ボタンが押されている場合はtrueを返します.
    //! @memo       前のフレームのボタン状態と現在のフレームのボタン状態が異なる場合にtrueを返します.
    //!             フラグの切り替え用途などに使用します.
    //---------------------------------------------------------------------------------------------
    bool IsDown( const u32 button ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ドラッグされているかどうかチェックします.
    //!
    //! @param[in]      button      ボタン.
    //! @return     ドラッグ中の場合はtrueを返します.
    //---------------------------------------------------------------------------------------------
    bool IsDrag( const u32 button ) const;

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // protected methods.
    //=============================================================================================
    /* NOTHING */

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    s32     m_CursorX;                          //!< カーソルのX座標です.
    s32     m_CursorY;                          //!< カーソルのY座標です.
    s32     m_PrevCursorX;                      //!< 前のカーソルのX座標.
    s32     m_PrevCursorY;                      //!< 前のカーソルのY座標.
    u32     m_Index;                            //!< バッファのインデックスです.
    bool    m_Button[ 2 ][ NUM_MOUSE_BUTTON ];  //!< ボタンの状態です.

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};

} // namespace asdx
