﻿//-------------------------------------------------------------------------------------------------
// File : asdxMotionPlayer.h
// Desc : Motion Player Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxMath.h>
#include <asdxResMotion.h>
#include <vector>
#include <map>


namespace asdx {

//-------------------------------------------------------------------------------------------------
// Forward Declarations.
//-------------------------------------------------------------------------------------------------
struct ResBone;


///////////////////////////////////////////////////////////////////////////////////////////////////
// MotionPlayer class
///////////////////////////////////////////////////////////////////////////////////////////////////
class MotionPlayer
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    MotionPlayer();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    ~MotionPlayer();

    //---------------------------------------------------------------------------------------------
    //! @brief      ボーンを関連付けします.
    //!
    //! @param[in]      boneCount       ボーン数です.
    //! @param[in]      pBones          ボーンデータへのポインタです.
    //---------------------------------------------------------------------------------------------
    void Bind( u32 boneCount, const ResBone* pBones );

    //---------------------------------------------------------------------------------------------
    //! @brief      ボーンの関連付けを解除します.
    //---------------------------------------------------------------------------------------------
    void Unbind();

    //---------------------------------------------------------------------------------------------
    //! @brief      モーションを設定します.
    //!
    //! @param[in]      pMotion         設定するモーションデータへのポインタ.
    //---------------------------------------------------------------------------------------------
    void SetMotion( const ResMotion* pMotion );

    //---------------------------------------------------------------------------------------------
    //! @brief      ループ再生フラグを設定します.
    //!
    //! @param[in]      isLoop      ループ再生する場合は true を指定.
    //---------------------------------------------------------------------------------------------
    void SetLoop( bool isLoop );

    //---------------------------------------------------------------------------------------------
    //! @brief      ループ再生フラグを取得します.
    //!
    //! @retval true    ループ再生です.
    //! @retval false   非ループ再生です.
    //---------------------------------------------------------------------------------------------
    bool IsLoop() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      更新処理を行います.
    //!
    //! @param[in]      elapsedSec      加算する経過時間(秒単位).
    //---------------------------------------------------------------------------------------------
    void Update( f32 elapsedSec );

    //---------------------------------------------------------------------------------------------
    //! @brief      再生時間を取得します
    //---------------------------------------------------------------------------------------------
    f32 GetFrameTime() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      変換行列の数を取得します.
    //!
    //! @return     変換行列の数を返却します.
    //---------------------------------------------------------------------------------------------
    u32  GetTransformCount() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ボーン行列を取得します.
    //!
    //! @return     ボーン行列を返却します.
    //---------------------------------------------------------------------------------------------
    const Matrix* GetBoneTransforms() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ワールド行列を取得します.
    //!
    //! @return     ワールド行列を返却します.
    //---------------------------------------------------------------------------------------------
    const Matrix* GetWorldTransforms() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      スキニング行列を取得します.
    //!
    //! @return     スキニング行列を返却します.
    //---------------------------------------------------------------------------------------------
    const Matrix* GetSkinTransforms() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    f32                 m_FrameTime;            //!< 現在時刻.
    u32                 m_BoneCount;            //!< ボーン数です.
    const ResBone*      m_pBones;               //!< ボーンデータです.
    const ResMotion*    m_pMotion;              //!< モーションです.
    std::vector<Matrix> m_BoneTransforms;       //!< ボーン行列です(親ボーン基準の行列).
    std::vector<Matrix> m_WorldTransforms;      //!< ワールド行列です(ワールド座標基準の行列).
    std::vector<Matrix> m_SkinTransforms;       //!< スキニング行列です(バインドポーズ基準の行列).
    bool                m_IsLoop;               //!< ループ再生フラグです.

    //=============================================================================================
    // private methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      指定された時間からボーン行列を求めます.
    //!
    //! @param[in]          time        フレーム時間.
    //! @param[in]          bone        ボーンのキーフレームセットです.
    //! @return     ボーン行列を返却します.
    //---------------------------------------------------------------------------------------------
    Matrix CalcBoneMatrix( f32 time, const ResKeyFrameSet& bone ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ボーン行列を更新します.
    //---------------------------------------------------------------------------------------------
    void UpdateBoneTransforms();

    //---------------------------------------------------------------------------------------------
    //! @brief      ワールド行列を更新します.
    //---------------------------------------------------------------------------------------------
    void UpdateWorldTransforms();

    //---------------------------------------------------------------------------------------------
    //! @brief      スキニング行列を更新します.
    //---------------------------------------------------------------------------------------------
    void UpdateSkinTransforms();
};


} // namespace asdx
