﻿//-------------------------------------------------------------------------------------------------
// File : asdxMisc.h
// Desc : Utility Moudle.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>
#include <string>


namespace asdx {

//-------------------------------------------------------------------------------------------------
//! @brief      指定された数値の倍数に切り上げます.
//!
//! @param [in]     val     数値.
//! @param [in]     base    倍数.
//! @return     val を base の倍数に切り上げた結果を返却します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u32 RoundUp( u32 val, u32 base )
{ return ( val + ( base - 1 ) ) & ~( base - 1 ); }

//-------------------------------------------------------------------------------------------------
//! @brief      指定された数値の倍数に切り上げます.
//!
//! @param [in]     val     数値.
//! @param [in]     base    倍数.
//! @return     val を base の倍数に切り上げた結果を返却します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u64 RoundUp( u64 val, u64 base )
{ return ( val + ( base - 1 ) ) & ~( base - 1 ); }

//-------------------------------------------------------------------------------------------------
//! @brief      指定された数値の倍数に切り下げます.
//!
//! @param [in]     val     数値.
//! @param [in]     base    倍数.
//! @return     val を base の倍数に切り下げた結果を返却します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u32 RoundDown( u32 val, u32 base )
{ return  val & ~( base - 1 ); }

//-------------------------------------------------------------------------------------------------
//! @brief      指定された数値の倍数に切り下げます.
//!
//! @param [in]     val     数値.
//! @param [in]     base    倍数.
//! @return     val を base の倍数に切り下げた結果を返却します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u64 RoundDown( u64 val, u64 base )
{ return  val & ~( base - 1 ); }

//-------------------------------------------------------------------------------------------------
//! @brief      ファイルパスを検索します.
//!
//! @param[in]      filePath        検索するファイスパス.
//! @param[out]     result          検索結果の格納先.
//! @retval true    ファイルを発見.
//! @retval false   ファイルが見つからなかった.
//! @memo 検索ルールは以下の通り.
//!      .\
//!      ..\
//!      ..\..\
//!      .\res\
//!      %EXE_DIR%\
//!      %EXE_DIR%\..\
//!      %EXE_DIR%\..\..\
//!      %EXE_DIR%\res\
//-------------------------------------------------------------------------------------------------
bool SearchFilePathW( const char16* filePath, std::wstring& result );

//-------------------------------------------------------------------------------------------------
//! @brief      ファイルパスを検索します.
//!
//! @param[in]      filePath        検索するファイスパス.
//! @param[out]     result          検索結果の格納先.
//! @retval true    ファイルを発見.
//! @retval false   ファイルが見つからなかった.
//! @memo 検索ルールは以下の通り.
//!      .\
//!      ..\
//!      ..\..\
//!      .\res\
//!      %EXE_DIR%\
//!      %EXE_DIR%\..\
//!      %EXE_DIR%\..\..\
//!      %EXE_DIR%\res\
//-------------------------------------------------------------------------------------------------
bool SearchFilePathA( const char8* filePath, std::string& result );

//-------------------------------------------------------------------------------------------------
//! @brief      ディレクトリ名を取得します.
//!
//! @param[in]      filePath        ファイルパス.
//! @return     指定されたファイルパスからディレクトリ名を抜き出します.
//-------------------------------------------------------------------------------------------------
std::wstring GetDirectoryPathW( const char16* fielPath );

//-------------------------------------------------------------------------------------------------
//! @brief      ディレクトリ名を取得します.
//!
//! @param[in]      filePath        ファイルパス.
//! @return     指定されたファイルパスからディレクトリ名を抜き出します.
//-------------------------------------------------------------------------------------------------
std::string GetDirectoryPathA( const char8* fielPath );

//-------------------------------------------------------------------------------------------------
//! @brief      拡張子を取得します.
//!
//! @param[in]      filePath        ファイルパス.
//! @return     指定されたファイルパスから拡張子を抜き出します.
//-------------------------------------------------------------------------------------------------
std::wstring GetExtW( const char16* filePath );

//-------------------------------------------------------------------------------------------------
//! @brief      拡張子を取得します.
//!
//! @param[in]      filePath        ファイルパス.
//! @return     指定されたファイルパスから拡張子を抜き出します.
//-------------------------------------------------------------------------------------------------
std::string GetExtA( const char8* filePath );

//-------------------------------------------------------------------------------------------------
//! @brief      実行ファイルのファイルパスを取得します.
//!
//! @return     実行ファイルのファイルパスを返却します.
//-------------------------------------------------------------------------------------------------
std::wstring GetExePathW();

//-------------------------------------------------------------------------------------------------
//! @brief      実行ファイルのファイルパスを取得します.
//!
//! @return     実行ファイルのファイルパスを返却します.
//-------------------------------------------------------------------------------------------------
std::string GetExePathA();

//-------------------------------------------------------------------------------------------------
//! @brief      指定されたファイルパスが存在するかチェックします.
//!
//! @param[in]      filePath        チェックするファイルパス.
//! @return     指定されたファイルパスが存在する場合はtrueを返却します.
//-------------------------------------------------------------------------------------------------
bool IsExistFilePathW( const char16* filePath );

//-------------------------------------------------------------------------------------------------
//! @brief      指定されたファイルパスが存在するかチェックします.
//!
//! @param[in]      filePath        チェックするファイルパス.
//! @return     指定されたファイルパスが存在する場合はtrueを返却します.
//-------------------------------------------------------------------------------------------------
bool IsExistFilePathA( const char8* filePath );

//-------------------------------------------------------------------------------------------------
//! @brief      指定されたフォルダパスが存在するかチェックします.
//!
//! @param[in]      folderPath      チェックするフォルダパス.
//! @return     指定されたフォルダパスが存在する場合はtrueを返却します.
//-------------------------------------------------------------------------------------------------
bool IsExistFolderPathW( const char16* folderPath );

//-------------------------------------------------------------------------------------------------
//! @brief      指定されたフォルダパスが存在するかチェックします.
//!
//! @param[in]      folderPath      チェックするフォルダパス.
//! @return     指定されたフォルダパスが存在する場合はtrueを返却します.
//-------------------------------------------------------------------------------------------------
bool IsExistFolderPathA( const char8* folderPath );

//-------------------------------------------------------------------------------------------------
//! @brief      ワイド文字列に変換します.
//!
//! @param[in]      value       変換するマルチバイト文字列.
//! @return     指定された文字列をワイド文字列に変換した結果を返却します.
//-------------------------------------------------------------------------------------------------
std::wstring ToStringW( const std::string& value );

//-------------------------------------------------------------------------------------------------
//! @brief      マルチバイト文字列に変換します.
//!
//! @param[in]      value       変換するワイド文字列.
//! @return     指定された文字列をマルチバイト文字列に変換した結果を返却します.
//-------------------------------------------------------------------------------------------------
std::string  ToStringA( const std::wstring& value );


} // namespacec asdx
