﻿//-------------------------------------------------------------------------------------------------
// File : asdxRef.h
// Desc : Reference Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// IReference interface
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IReference
{
    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを1つ増やします.
    //---------------------------------------------------------------------------------------------
    virtual void AddRef() = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを1つ減らします.
    //---------------------------------------------------------------------------------------------
    virtual void Release() = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを取得します.
    //!
    //! @return     参照カウントを返却します.
    //---------------------------------------------------------------------------------------------
    virtual s32 GetCount() const = 0;
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// RefPtr class
///////////////////////////////////////////////////////////////////////////////////////////////////
ASDX_TEMPLATE(T)
class RefPtr
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    ASDX_TEMPLATE(U) friend class RefPtr;

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    RefPtr() ASDX_NOTHROW
    : m_pPtr( nullptr )
    { /* DO_NOTHING */ }

    //---------------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //---------------------------------------------------------------------------------------------
    RefPtr( std::nullptr_t ) ASDX_NOTHROW
    : m_pPtr( nullptr )
    { /* DO_NOTHING */ }

    //---------------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //---------------------------------------------------------------------------------------------
    RefPtr( T* pInterface ) ASDX_NOTHROW
    : m_pPtr( pInterface )
    { AddRef(); }

    //---------------------------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //---------------------------------------------------------------------------------------------
    RefPtr( const RefPtr& value ) ASDX_NOTHROW
    : m_pPtr( value.m_pPtr )
    { AddRef(); }

    //---------------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //---------------------------------------------------------------------------------------------
    ASDX_TEMPLATE(U)
    RefPtr( U* value ) ASDX_NOTHROW
    : m_pPtr( value )
    { AddRef(); }

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    ~RefPtr() ASDX_NOTHROW
    { Release(); }

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //---------------------------------------------------------------------------------------------
    RefPtr& operator = ( T* value ) ASDX_NOTHROW
    {
        if ( m_pPtr != value )
        { RefPtr(value).Swap(*this); }
        return (*this);
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //---------------------------------------------------------------------------------------------
    RefPtr& operator = ( const RefPtr& value ) ASDX_NOTHROW
    {
        if ( m_pPtr != value.m_pPtr )
        { RefPtr(value).Swap(*this); }
        return (*this);
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //---------------------------------------------------------------------------------------------
    ASDX_TEMPLATE(U)
    RefPtr& operator = ( U* value ) ASDX_NOTHROW
    {
        RefPtr(value).Swap(*this);
        return (*this);
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //---------------------------------------------------------------------------------------------
    ASDX_TEMPLATE(U)
    RefPtr& operator = ( const RefPtr<U>& value ) ASDX_NOTHROW
    {
        RefPtr(value).Swap(*this);
        return (*this);
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //---------------------------------------------------------------------------------------------
    RefPtr& operator = ( std::nullptr_t ) ASDX_NOTHROW
    {
        Release();
        return (*this);
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      アタッチします.
    //---------------------------------------------------------------------------------------------
    void Attach( T* value ) ASDX_NOTHROW
    {
        if ( m_pPtr != nullptr )
        { m_pPtr->Release(); }
        m_pPtr = value;
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      デタッチします.
    //---------------------------------------------------------------------------------------------
    T* Detach() ASDX_NOTHROW
    {
        T* ptr = m_pPtr;
        m_pPtr = nullptr;
        return ptr;
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      値を入れ替えます.
    //---------------------------------------------------------------------------------------------
    void Swap( RefPtr& value ) ASDX_NOTHROW
    {
        T* temp = m_pPtr;
        m_pPtr = value.m_pPtr;
        value.m_pPtr = temp;
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      値を入れ替えます.
    //---------------------------------------------------------------------------------------------
    void Swap( RefPtr&& value ) ASDX_NOTHROW
    {
        T* temp = m_pPtr;
        m_pPtr = value.m_pPtr;
        value.m_pPtr = temp;
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      ポインタを取得します.
    //---------------------------------------------------------------------------------------------
    T* GetPtr() const ASDX_NOTHROW
    { return m_pPtr; }

    //---------------------------------------------------------------------------------------------
    //! @brief      アドレスを取得します.
    //---------------------------------------------------------------------------------------------
    T* const* GetAddress() const ASDX_NOTHROW
    { return &m_pPtr; }

    //---------------------------------------------------------------------------------------------
    //! @brief      アドレスを取得します.
    //---------------------------------------------------------------------------------------------
    T** GetAddress() ASDX_NOTHROW
    { return &m_pPtr; }

    //---------------------------------------------------------------------------------------------
    //! @brief      リセットします.
    //---------------------------------------------------------------------------------------------
    void Reset() ASDX_NOTHROW
    { Release(); }

    //---------------------------------------------------------------------------------------------
    //! @brief      指定された値にコピーを行います.
    //---------------------------------------------------------------------------------------------
    void CopyTo( T** ptr ) ASDX_NOTHROW
    {
        AddRef();
        (*ptr) = m_pPtr;
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      アロー演算子です.
    //---------------------------------------------------------------------------------------------
    T* operator -> () const ASDX_NOTHROW
    { return m_pPtr; }

    //---------------------------------------------------------------------------------------------
    //! @brief      bool型へのキャストです.
    //---------------------------------------------------------------------------------------------
    operator bool () const ASDX_NOTHROW
    { return m_pPtr != nullptr; }

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    T*  m_pPtr;     //!< 参照カウントインタフェースを実装するオブジェクトのポインタです.

    //=============================================================================================
    // protected methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを増やします.
    //---------------------------------------------------------------------------------------------
    void AddRef()
    {
        if ( m_pPtr != nullptr )
        { m_pPtr->AddRef(); }
    }

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを減らします.
    //---------------------------------------------------------------------------------------------
    void Release()
    {
        T* ptr = m_pPtr;
        if ( ptr != nullptr )
        {
            m_pPtr = nullptr;
            ptr->Release();
        }
    }

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};

ASDX_TEMPLATE(T)
bool operator == ( const RefPtr<T>& value, const T* ptr )
{ return ( value.GetPtr() == ptr ); }

ASDX_TEMPLATE(T)
bool operator == ( const T* ptr, const RefPtr<T>& value )
{ return ( value.GetPtr() == ptr ); }

ASDX_TEMPLATE(T)
bool operator == ( const RefPtr<T>& a, const RefPtr<T>& b )
{ return a.GetPtr() == b.GetPtr(); }

ASDX_TEMPLATE2(T, U)
bool operator == ( const RefPtr<T>& a, const RefPtr<U>& b )
{ return a.GetPtr() == b.GetPtr(); }

ASDX_TEMPLATE(T)
bool operator != ( const RefPtr<T>& value, const T* ptr )
{ return ( value.GetPtr() != ptr ); }

ASDX_TEMPLATE(T)
bool operator != ( const T* ptr, const RefPtr<T>& value )
{ return ( value.GetPtr() != ptr ); }

ASDX_TEMPLATE(T)
bool operator != ( const RefPtr<T>& a, const RefPtr<T>& b )
{ return a.GetPtr() != b.GetPtr(); }

ASDX_TEMPLATE2(T, U)
bool operator != ( const RefPtr<T>& a, const RefPtr<U>& b )
{ return a.GetPtr() != b.GetPtr(); }

ASDX_TEMPLATE(T)
bool operator == ( const RefPtr<T>& value, std::nullptr_t )
{ return ( value.GetPtr() == nullptr ); }

ASDX_TEMPLATE(T)
bool operator == ( std::nullptr_t, const RefPtr<T>& value )
{ return ( value.GetPtr() == nullptr ); }

ASDX_TEMPLATE(T)
bool operator != ( const RefPtr<T>& value, std::nullptr_t )
{ return ( value.GetPtr() != nullptr ); }

ASDX_TEMPLATE(T)
bool operator != ( std::nullptr_t, const RefPtr<T>& value )
{ return ( value.GetPtr() != nullptr ); }


} // namespace asdx
