﻿//-------------------------------------------------------------------------------------------------
// File : asdxResHDR.h
// Desc : HDR File Format.
// Copyright(c) Project Asura. All right reserved
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>
#include <asdxMath.h>
#include <asdxILoadable.h>
#include <asdxIDisposable.h>


namespace asdx {

//-------------------------------------------------------------------------------------------------
// Forward Declarations.
//-------------------------------------------------------------------------------------------------
struct RGBE;

///////////////////////////////////////////////////////////////////////////////////////////////////
// ResHDR class
///////////////////////////////////////////////////////////////////////////////////////////////////
class ResHDR : public ILoadable, public IDisposable
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    ResHDR();

    //---------------------------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param[in]      value       コピー元の値です.
    //---------------------------------------------------------------------------------------------
    ResHDR( const ResHDR& value );

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~ResHDR();

    //---------------------------------------------------------------------------------------------
    //! @brief      ファイルから読み込みします.
    //!
    //! @param[in]      filename        ファイル名です.
    //! @retval true    読み込みに成功.
    //! @retval false   読み込みに失敗.
    //---------------------------------------------------------------------------------------------
    bool Load( const char16* filename ) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリを解放します.
    //---------------------------------------------------------------------------------------------
    void Dispose() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      画像の横幅を取得します.
    //!
    //! @return     画像の横幅を返却します.
    //---------------------------------------------------------------------------------------------
    const u32 GetWidth() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      画像の縦幅を取得します.
    //!
    //! @return     画像の横幅を返却します.
    //---------------------------------------------------------------------------------------------
    const u32 GetHeight() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      露光値を取得します.
    //!
    //! @return     露光値を返却します.
    //---------------------------------------------------------------------------------------------
    const f32 GetExposure() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ピクセルデータを取得します.
    //!
    //! @return     RBGE形式のピクセルデータを返却します.
    //---------------------------------------------------------------------------------------------
    const u8* GetPixels() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      デコードされたピクセルデータを取得します.
    //!
    //! @param[out]     ppReuslt        浮動小数点RGB形式のピクセルデータを格納先です.
    //---------------------------------------------------------------------------------------------
    void GetFloatPixels( f32** ppResults ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param[in]      value       代入する値です.
    //! @return     代入結果を返却します.
    //---------------------------------------------------------------------------------------------
    ResHDR& operator = ( const ResHDR& value );

    //---------------------------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param[in]      value       比較する値です.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //---------------------------------------------------------------------------------------------
    bool operator == ( const ResHDR& value ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param[in]      value       比較する値です.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //---------------------------------------------------------------------------------------------s
    bool operator != ( const ResHDR& value ) const;

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    u32     m_Width;        //!< 画像の横幅です.
    u32     m_Height;       //!< 画像の縦幅です.
    f32     m_Exposure;     //!< 露出値です.
    f32     m_Gamma;        //!< ガンマ値です.
    RGBE*   m_pPixels;      //!< RGBEピクセルデータです.
    u32     m_HashKey;      //!< ハッシュキーです.

    //=============================================================================================
    // protected methods.
    //=============================================================================================
    /* NOTHING */

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};


} // namespace asdx
